import { TestBed } from '@angular/core/testing';
import { ComponentsApiService } from './components-api.service';
import { ApiService, CoreModule, TableSortDirection, and, or, operator, attribute } from '@bref/core';
import { FilterLikeTerm, AttributeKey } from '../interfaces/component-filters.interface';

const get = jest.fn((url): any => ({
  "data": [
    {
      "id": "1",
      "attributes": {
        "name": "fryer",
        "modelNumber": 606,
        "serialNumber": 123,
        "warranty": "the_warranty",
        "reportedVersion": "1.3"
      },
      "relationships": {
        "restaurant": { "data": { "id": "1" } },
        "vendor": { "data": { "id": "1" } },
        "type": { "data": { "id": "3"} }
      }
    }
  ],
  "included": [
    {
      "id": "2",
      "type": "component_props",
      "attributes": { "propertyName": "deviceIP", "propertyValue": "xyz" }
    },
    {
      "id": "2",
      "type": "restaurants",
      "attributes": { "name": "some restaurant" },
    },
    {
      "id": "1",
      "type": "component_props",
      "attributes": { "propertyName": "deviceIP", "propertyValue": "foobar" }
    },
    {
      "id": "1",
      "type": "restaurants",
      "attributes": { "name": "McDonald's Global Menu Restaurant (HQ)" },
      "relationships": { "market": { "data": { "id": "1" } } }
    },
    {
      "id": "1",
      "type": "vendors",
      "attributes": { "name": "some vendor" },
    },
    {
      "id": "1",
      "type": "markets",
      "attributes": { "name": "some market" },
    }
  ],
  "meta": {
    "totalResourceCount": 11
  }
}));

class ApiServiceMock {
  async get(url) {
    return get(url);
  }
}

xdescribe('ComponentsApiService', () => {
  let service: ComponentsApiService;

  beforeEach(() => {
    TestBed.configureTestingModule({
      imports: [CoreModule],
      providers: [
        { provide: ApiService, useValue: new ApiServiceMock() }
      ]
    });
    service = TestBed.inject(ComponentsApiService);
  });
  afterEach(() => {
    get.mockClear();
  });

  it('should be created', () => {
    expect(service).toBeTruthy();
  });

  it('should properly request devices on getDevices', async () => {
    const devices = await service.getDevices(2, 25, 'name', TableSortDirection.desc, { componentName: ['%foo%', '%bar%'], storeName: ['%test%', '%cool%'] });

    const urlParams = new URLSearchParams({
      'include': 'componentProps,restaurant.market,vendor,type',
      'sort': '-name,id',
      'page[offset]': '25',
      'page[limit]': '25',
      'filter': JSON.stringify({
        'AND': [
          {
            'type': {
              'name': 'GREENGRASS_DEVICE',
            },
          },{
          'like': {
            'name': ['%foo%', '%bar%']
            },
          },
          {
          'restaurant': {
            'like': {
              'name': ['%test%', '%cool%']
            }
          }
        }]
      })
    });
    expect(get).toHaveBeenCalledWith(`https://asset-service.bre.mcd.com/restaurant_assets/components?${urlParams.toString()}`);
    expect(devices).toEqual({
      totalComponentCount: 11,
      data: [
        {
          id: '1',
          status: '-',
          name: 'fryer',
          model: '606',
          type: '3',
          store: 'SOME MARKET-1',
          storeId: '1',
          function: '-',
          ipAddress: 'foobar',
          serialNumber: 123,
          cellLocation: null,
          certificate: null,
          description: null,
          lineOfBusiness: null,
          mainUsage: null,
          manufacturer: null,
          market: 'some market',
          technicalName: null,
          version: '1.3',
          warranty: 'the_warranty',
          supplier: 'some vendor'
        }
      ]
    });
  });

  it('should properly request devices on searchDevices by name', async () => {
    const devices = await service.searchDevices('foo', FilterLikeTerm.name);

    const urlParams = new URLSearchParams({
      'include': 'type',
      'sort': 'name,id',
      'filter[type.name]': 'GREENGRASS_DEVICE',
      'filter[name][like]': '%foo%'
    });
    expect(get).toHaveBeenCalledWith(`https://asset-service.bre.mcd.com/restaurant_assets/components?${urlParams.toString()}`);
    expect(devices).toEqual([
      {
        id: '1',
        status: '-',
        name: 'fryer',
        model: '606',
        type: '3',
        store: 'SOME MARKET-1',
        storeId: '1',
        function: '-',
        ipAddress: 'foobar',
        serialNumber: 123,
        cellLocation: null,
        certificate: null,
        description: null,
        lineOfBusiness: null,
        mainUsage: null,
        manufacturer: null,
        market: 'some market',
        technicalName: null,
        version: '1.3',
        warranty: 'the_warranty',
        supplier: 'some vendor'
      }
    ]);
  });

  it('should properly request assets on searchAssets by name', async () => {
    const assets = await service.searchAssets('foo', FilterLikeTerm.name);

    const urlParams = new URLSearchParams({
      'include': 'type',
      'sort': 'id,name',
      'filter[type.name][neq]': 'GREENGRASS_DEVICE',
      'filter[name][like]': '%foo%'
    });
    expect(get).toHaveBeenCalledWith(`https://asset-service.bre.mcd.com/restaurant_assets/components?${urlParams.toString()}`);
    expect(assets).toEqual([
      {
        id: '1',
        status: '-',
        name: 'fryer',
        model: '606',
        type: '3',
        store: 'SOME MARKET-1',
        storeId: '1',
        function: '-',
        ipAddress: 'foobar',
        serialNumber: 123,
        cellLocation: null,
        certificate: null,
        description: null,
        lineOfBusiness: null,
        mainUsage: null,
        manufacturer: null,
        market: 'some market',
        technicalName: null,
        version: '1.3',
        warranty: 'the_warranty',
        supplier: 'some vendor'
      }
    ]);
  });
});