import { Injectable, Inject } from '@angular/core';
import { CertificatesFilters } from '../interfaces/certificates-filters.interface';
import { IncludedApiData, CertificateApiData, CertificateApiResponse } from '../interfaces/certificates-api-response.interface';
import { CoreModule, TableSortDirection, ApiService, operator, and, or } from '@bref/core';
import { SolutionsEnvService } from '../solutions-env.service';
import { Certificates, CertificateList, CertificateStatus } from '../interfaces/certificates.interface';
import { arraysAreEqual } from '../utility/general-utils';


@Injectable({
    providedIn: CoreModule
})
export class CertificatesApiService {
    private componentConfigBaseUrl: string;

    constructor(private api: ApiService) {
        this.componentConfigBaseUrl = SolutionsEnvService.get('configSolutions') + '/component_config';
    }

    private getBackendBaseHeaders() {
        return {
            Authorization: "Bearer " + SolutionsEnvService.get('msalIDToken')
        };
    }

  /**
   * 
   * @param page page number
   * @param pageSize number fo certificates per page
   * @param sortBy column name to sort by
   * @param sortDirection as in asc or dec of type TableSortDirection
   * @param filters of type CertificatesFilters
   * This function calls Config API and returns certificate data along with component and restaurant info.
   * @returns certificates of type CertificateList
   */
    async getCertificates(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection, filters: CertificatesFilters): Promise<CertificateList> {
        const urlSearchParams = new URLSearchParams({
            'include': 'restaurant,component',
            'sort': sortBy === 'expiry' ? `${sortDirection === TableSortDirection.asc ? '' : '-'}${sortBy},restaurant.id` : `${sortDirection === TableSortDirection.asc ? '' : '-'}${sortBy}`,
            'page[limit]': pageSize.toString(),
            'page[offset]': ((page - 1) * pageSize).toString(),
            'filter[type.name]': 'SecretCert'
        });

        if (filters?.storeName !== undefined && filters?.storeName.length > 0) {
            urlSearchParams.append('filter[restaurant.name]', filters?.storeName.toString());
        }

        if (filters?.component !== undefined && filters?.component.length > 0) {
            urlSearchParams.append('filter[component.name]', filters?.component.toString());
        }
        if (filters?.certificateARN !== undefined && filters?.certificateARN.length > 0) {
            urlSearchParams.append('filter[propertyValue]', filters?.certificateARN.toString());
        }

        if (filters?.status !== undefined && filters?.status.length > 0 && filters?.status.length < 3) {

            const statusFilters = this.getExpiryTimeRangeBasedOnStatus(filters?.status)
            if (statusFilters) {
                urlSearchParams.append('filter', JSON.stringify(statusFilters));
            }
        }
        if(filters?.deviceId) {
            urlSearchParams.append('filter[component.id]', filters?.deviceId.toString());
        }

        const response = await this.api.get<CertificateApiResponse>(`${this.componentConfigBaseUrl}?${urlSearchParams.toString()}`, null, null, this.getBackendBaseHeaders());
        const certificateData = response.data.map(certificateData => this.reshapeCertificateResponse(certificateData, response.included));
        return {
            data: certificateData,
            totalItemCount: response.meta.totalResourceCount
        };
    }

/**
 * 
 * @param certificates of type CertificateApiData
 * @param included of type IncludedApiData
 * This function reshapes and returns Certificate object for UI binding
 * @returns component
 */
    private reshapeCertificateResponse(certificates: CertificateApiData, included: IncludedApiData[]): Certificates {
        let componentConfigId = certificates.id;
        let configurationTypeName = "SecretCert";
        let componentConfigName = certificates.attributes.propertyName;
        let componentConfigValue = certificates.attributes.propertyValue;
        let componentConfigCreatedDate = certificates.attributes.created;
        let componentConfigUpdatedDate = certificates.attributes.updated;
        let restaurant = this.getRestaurant(certificates, included);
        let component = this.getComponent(certificates, included);
        let cerStatus = this.getStatus(certificates.attributes.expiry);
        return {
            configurationTypeName: configurationTypeName,
            componentConfigId: componentConfigId,
            componentConfigName: componentConfigName,
            componentConfigValue: componentConfigValue,
            componentId: component.id,
            componentName: component.attributes.name,
            restaurantId: restaurant.id,
            restaurantName: restaurant.attributes.name,
            createdDate: componentConfigCreatedDate,
            issueDate: componentConfigCreatedDate,
            updatedDate: componentConfigUpdatedDate,
            expiryDate: cerStatus,
            status: cerStatus.status
        };
    }

    /**
 * 
 * @param certificates of type CertificateApiData
 * @param included of type IncludedApiData
 * This function returns component object (application/device) which has a certificate
 * @returns component
 */
    private getComponent(certificates: CertificateApiData, included: IncludedApiData[]) {
        let component;
        let componentId = certificates.relationships.component.data.id;
        component = included?.find(i => i.id === componentId && i.type === 'components');
        return component;
    }

    /**
* 
* @param certificates of type CertificateApiData
* @param included of type IncludedApiData
* This function returns restaurants which uses certificates
* @returns restaurants
*/
    private getRestaurant(certificates: CertificateApiData, included: IncludedApiData[]) {
        let restaurantId;
        let restaurant = { id: "", attributes: { name: "" } };
        if (certificates.relationships.restaurant.data !== null) {
            restaurantId = certificates.relationships.restaurant.data.id;
            restaurant = included?.find(i => i.id === restaurantId && i.type === 'restaurants');
        }
        return restaurant;
    }

    /**
    * 
    * @param deployments of type DeploymentApiData
    * @param included of type IncludedApiData
    * This function returns Status (either Active, Needs Attention or Expired) for each certificate
    * @returns status
    */
    private getStatus(expiry: string): CertificateStatus {
        let response: CertificateStatus;
        const expiryDate = new Date(expiry);
        const today = new Date();
        let diffDays: number;
        let diffTime: number;
        if (today.getTime() < expiryDate.getTime()) {
            diffTime = Math.abs(today.getTime() - expiryDate.getTime());
            diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
        } else {
            diffDays = -1;
        }

        if (diffDays > 30) {
            response = {
                expiry: expiry,
                status: "active",
                days: ""
            };

        } else if (diffDays > 0 && diffDays <= 30) {
            response = {
                expiry: expiry,
                status: "needsAttention",
                days: " (" + diffDays.toString() + " days left)"
            };
        } else if (diffDays <= 0) {
            response = {
                expiry: expiry,
                status: "expired",
                days: ""
            };
        }
        else {
            response = {
                expiry: expiry,
                status: "undefined",
                days: ""
            };
        }
        return response;
    }

    /**
    * 
    * @param status of type String[]
    * This function returns filter expression ( expiry date with GE and LE filter) based on certificate Status (either Active, Needs Attention or Expired)
    * @returns filters
    */
    private getExpiryTimeRangeBasedOnStatus(status: string[]) {
        let filters;
        const today = new Date();
        const sixtyDaysAfterToday = new Date(today.setDate(today.getDate() + 60));

        if (arraysAreEqual(status, ["expired"])) {
            filters = and(
                operator('ge', false, { expiry: null }),
                operator('le', false, { expiry: new Date().getTime() })
            );
        } else if (arraysAreEqual(status, ["expired", "attention"])) {
            filters = and(
                operator('ge', false, { expiry: null }),
                operator('le', false, { expiry: sixtyDaysAfterToday.getTime() })
            );
        } else if (arraysAreEqual(status, ["attention", "active"])) {
            filters = and(
                operator('ge', false, { expiry: new Date().getTime() }),
                operator('le', false, { expiry: null })
            );
        } else if (arraysAreEqual(status, ["attention"])) {
            filters = and(
                operator('ge', false, { expiry: new Date().getTime() }),
                operator('le', false, { expiry: sixtyDaysAfterToday.getTime() })
            );
        } else if (arraysAreEqual(status, ["active"])) {
            filters = and(
                operator('ge', false, { expiry: sixtyDaysAfterToday.getTime() }),
                operator('le', false, { expiry: null })
            );
        }
        return filters;
    }
}