import { Component, OnInit } from '@angular/core';
import { DatePipe } from '@angular/common';
import { ActivatedRoute, Router } from '@angular/router';
import { ComponentsApiService } from '../../services/components-api.service';
import { DialogService, FilterType, Filter, TablePageChangeEvent, PageSizes, TableSortDirection, TableRowClickEvent } from '@bref/core';
import { ComponentFilters, FilterLikeTerm, AssetType } from '../../interfaces/component-filters.interface';
import { Breadcrumb } from '@bref/core/lib/components/breadcrumbs/breadcrumb.interface';
import { RestaurantsApiService } from '../../services/restaurants-api.service';
import { DeviceDialogService } from '../../services/device-dialog.service';
import { BulkOnboardingDevicesService } from '../../services/bulk-onboarding-service';
import { Component as Device } from '../../interfaces/component.interface';
import { VendorOnboardingComponent } from '../vendor-onboarding/vendor-onboarding.component';
import { SelectedVendorDataService } from '../../services/selected-vendor-data.service';

@Component({
  selector: 'bre-vendor-list',
  templateUrl: './vendor-list.component.html',
  styleUrls: ['./vendor-list.component.scss'],
  providers: [DatePipe]
})
export class VendorListComponent implements OnInit {
vendors:any = [];
isLoading: boolean = true;

totalVendors: number = 0;

  types: AssetType[];

  page: number;
  pageSize: PageSizes;
  sortBy = 'id';
  sortDirection = TableSortDirection.desc;
  configReset = false;
  eventSet =new Set();
  itemCount:number;
  hideResetFiltersOnLoad = 0;

  selectedFilters: ComponentFilters = {};
  filters: Filter[] = [
    {
      name: 'id',
      type: FilterType.autocomplete,
      placeholder: 'id',
      options: [],
      isAdvanced: false,
    },
    {
      name: 'name',
      type: FilterType.autocomplete,
      placeholder: 'Display name',
      options: [],
      isAdvanced: false,
    },
    {
      name: 'contactName',
      type: FilterType.autocomplete, 
      placeholder: 'Contact Name',
      options: [],
      isAdvanced: false,
    },
    {
      name: 'contactInfo',
      type: FilterType.autocomplete,
      placeholder: 'Contact info',
      options: [],
      isAdvanced: false,
    },
    {
      name: 'created',
      type: FilterType.autocomplete,
      placeholder: 'Onboarded Date',
      options: [],
      isAdvanced: false,
    }
  ];

  constructor( 
    private devicesApi: ComponentsApiService,
    private restaurantsApi: RestaurantsApiService,
    private route: ActivatedRoute,
    private dialog: DeviceDialogService,
    private router: Router,
    private datepipe: DatePipe,
    private vendorData: SelectedVendorDataService
    ) { 
      this.vendorData.onboardSuccess.subscribe(status => {
        if(status) {
            this.loadData(this.page,
              this.pageSize,
              this.sortBy,
              this.sortDirection
              );
        }
     });

    }

    async ngOnInit(): Promise<void>{
      

  this.page = parseInt(this.route.snapshot.queryParams.page, 10) || 1;
    this.pageSize =
      parseInt(this.route.snapshot.queryParams.pageSize, 10) ||
      PageSizes.TwentyFive;
    this.sortBy = this.route.snapshot.queryParams.sortBy || 'id';
    this.sortDirection = TableSortDirection.desc;
    this.isLoading = true;
    this.loadData(
      this.page,
      this.pageSize,
      this.sortBy,
      this.sortDirection
    );
  }

  getComponentTypeName(typeId) {
    return this.types.find(x => x.id == typeId).name;
  }

  pageData(event: TablePageChangeEvent): void {
    this.page = event.page;
    this.pageSize = event.pageSize;
    this.sortBy = event.sortBy;
    this.sortDirection = event.sortDirection;
    this.updateQueryParams();
  }

  filterData(event): void {
    this.page = 1;
    this.selectedFilters = {
      componentNameId: event.filters.componentNameId,
      storeNameId: event.filters.storeNameId,
      componentName: event.filters.componentNameId?.filter((x) => parseInt(x).toString() !== x),
      componentId: event.filters.componentNameId?.filter((x) => parseInt(x).toString() === x),
      storeName: event.filters.storeNameId?.filter((x) => parseInt(x).toString() !== x),
      storeId: event.filters.storeNameId?.filter((x) => parseInt(x).toString() === x)
    };

    this.loadData(
      this.page,
      this.pageSize,
      this.sortBy,
      this.sortDirection
    );

    this.updateQueryParams();
  }

  navigateToDeviceDetails(event: TableRowClickEvent<Device>) {
    this.router.navigate([event.item.id], { relativeTo: this.route });
  }

  private async loadData(
    page: number,
    pageSize: number,
    sortBy: string,
    sortDirection: TableSortDirection
  ) {
    
    await this.devicesApi.getVendorsList(
      page,
      pageSize,
      sortBy,
      sortDirection
    ).then(response => {
      this.isLoading = false;
      this.totalVendors = response.totalComponentCount;

      let res =  response.data?.map(item => {
        let date = this.datepipe.transform(item.created, 'MM/dd/yyyy');
        item.created = `${date}`;  
        item.id = parseInt(item.id);
        return item;
      })     

      this.vendors = res;
      
    }).catch((error) => {
      this.isLoading = false;
    });
  }


  private async setDevicesAutoCompleteOptions(
    searchTerm?: string
  ): Promise<void> {
    const sortMatches = (a, b) => {
      a = a.label.toLowerCase();
      b = b.label.toLowerCase();
      if (a < b) {
        return -1;
      } else if (a > b) {
        return 1;
      } else {
        return 0;
      }
    }
    const deviceNameMatchesRequest =
      searchTerm?.length > 0
        ? this.devicesApi.searchDevices(searchTerm, FilterLikeTerm.name)
        : [];
    const deviceIdMatchesRequest =
      searchTerm?.length > 0
        ? this.devicesApi.searchDevices(searchTerm, FilterLikeTerm.id)
        : [];
    const deviceIdRequest =
      this.selectedFilters?.componentId?.length > 0
        ? this.devicesApi.getDevices(this.page, this.pageSize, this.sortBy, this.sortDirection, { componentId: this.selectedFilters.componentId }).then(x => x.data)
        : []
    const [deviceNameMatches, deviceIdMatches, deviceIdAll] = await Promise.all([deviceNameMatchesRequest, deviceIdMatchesRequest, deviceIdRequest]);
    const allOptions = [
      !!searchTerm ? { label: `View all results with '${searchTerm}'`, value: `%${searchTerm}%` } : { label: undefined, value: undefined },
      ...(this.selectedFilters?.componentName?.map((x) => ({ label: `View all results with '${x.replace(/%/g, '')}'`, value: `%${x.replace(/%/g, '')}%` })) || []),
      ...(deviceIdAll?.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id })) || []),
      ...[...deviceNameMatches.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id })),
      ...deviceIdMatches.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id }))].sort(sortMatches)
    ];

    this.filters[0].options = allOptions.filter((x) => !!x.value).reduce((acc, curr) => {
      if (!acc.find((x) => x.label === curr.label)) {
        acc.push(curr);
      }
      return acc;
    }, [])
  }  

  private updateQueryParams() {
    this.router.navigate([], {
      relativeTo: this.route,
      queryParams: {
        componentName: this.selectedFilters.componentName?.length > 0 ? this.selectedFilters.componentName : undefined,
        componentId: this.selectedFilters.componentId?.length > 0 ? this.selectedFilters.componentId : undefined,
        storeName: this.selectedFilters.storeName?.length > 0 ? this.selectedFilters.storeName : undefined,
        storeId: this.selectedFilters.storeId?.length > 0 ? this.selectedFilters.storeId : undefined,
        page: this.page,
        pageSize: this.pageSize,
        sortBy: this.sortBy,
        sortDirection: this.sortDirection,
      },
    });
  }

  showOnboardingVendorDialog() {
    this.dialog.open(
      VendorOnboardingComponent
    );
  }

  /**
   * 
   * @param $event to get the item counts
   */
  getItems($event){
    this.itemCount=$event
   }

  /**
   * Called on clear and reset to hide the reset buttons
   */
  filterCleared(){
    this.configReset=false;
  }
  
  /**
   * @param $event click event
   * This function is triggered on load and on every apply and clear of filters
   */
  updateApply($event){
    if(this.hideResetFiltersOnLoad === 0) {
      this.configReset = false;
      this.hideResetFiltersOnLoad++;
    } else {
      this.configReset = true;
    }
    if($event[2]==='apply'){
      this.eventSet.add($event[0])
    }
    else{
      this.eventSet.delete($event[0])
      if(this.eventSet.size===1){
        this.filterCleared()
      }
  }
  }
}


