import { Component, OnInit, ViewEncapsulation } from '@angular/core';
import { FilterType, Filter, TablePageChangeEvent, PageSizes, TableSortDirection, TableRowClickEvent } from '@bref/core';
import { ActivatedRoute, Router } from '@angular/router';
import { Component as Device } from '../../interfaces/component.interface';
import { ComponentFilters, FilterLikeTerm, AssetType } from '../../interfaces/component-filters.interface';
import { ComponentsApiService } from '../../services/components-api.service';
import { RestaurantsApiService } from '../../services/restaurants-api.service';
import { DeviceOnboardingComponent } from '../device-onboarding/device-onboarding.component';
import { DeviceDialogService } from '../../services/device-dialog.service';
import { BulkOnboardingDevicesService } from '../../services/bulk-onboarding-service';
@Component({
  selector: 'bre-devices-list',
  templateUrl: './devices-list.component.html',
  styleUrls: ['./devices-list.component.scss'],
  encapsulation: ViewEncapsulation.None,
})
export class DevicesListComponent implements OnInit {
  isLoading: boolean = false;
  devices = [];
  totalDevices: number = 0;

  types: AssetType[];

  page: number;
  pageSize: PageSizes;
  sortBy: string = 'id';
  sortDirection = TableSortDirection.desc;

  selectedFilters: ComponentFilters = {};
  configReset = false;
  eventSet =new Set();
  itemCount:number;
  hideResetFiltersOnLoad = 0;
  filters: Filter[] = [
    {
      name: 'componentNameId',
      type: FilterType.autocomplete,
      placeholder: 'Search Device',
      options: [],
      loadOptions: (searchTerm: string) =>
        this.setDevicesAutoCompleteOptions(searchTerm),
      isAdvanced: false,
    },
    {
      name: 'storeNameId',
      type: FilterType.autocomplete,
      placeholder: 'Search Store',
      options: [],
      loadOptions: (searchTerm: string) =>
        this.setRestaurantsAutoCompleteOptions(searchTerm),
      isAdvanced: false,
    },
  ];
  constructor(
    private devicesApi: ComponentsApiService,
    private restaurantsApi: RestaurantsApiService,
    private route: ActivatedRoute,
    private dialog: DeviceDialogService,
    private router: Router,
    private onboardSuccess: BulkOnboardingDevicesService
  ) {
    this.onboardSuccess.onboardSuccess.subscribe(status => {
      if(status) {
          this.loadData(this.page,
            this.pageSize,
            this.sortBy,
            this.sortDirection,
            this.selectedFilters);
      }
   });
   }

  async ngOnInit(): Promise<void> {
    this.page = parseInt(this.route.snapshot.queryParams.page, 10) || 1;
    this.pageSize =
      parseInt(this.route.snapshot.queryParams.pageSize, 10) ||
      PageSizes.TwentyFive;
    this.sortBy = this.route.snapshot.queryParams.sortBy || 'id';
    this.sortDirection = TableSortDirection.desc;
    this.selectedFilters = Object.keys(this.route.snapshot.queryParams).reduce(
      (result, key) => {
        const value = this.route.snapshot.queryParams[key];
        result[key] = Array.isArray(value) ? [...value] : [value];
        if (key === 'componentName' || key === 'componentId') {
          if (!result['componentNameId']) {
            result['componentNameId'] = [];
          }
          Array.isArray(value) ? result['componentNameId'].push(...value) : result['componentNameId'].push(value);
        } else if (key === 'storeName' || key === 'storeId') {
          if (!result['storeNameId']) {
            result['storeNameId'] = [];
          }
          Array.isArray(value) ? result['storeNameId'].push(...value) : result['storeNameId'].push(value);
        }
        return result;
      },
      {}
    );

    this.isLoading = true;
    this.loadData(
      this.page,
      this.pageSize,
      this.sortBy,
      this.sortDirection,
      this.selectedFilters,
    );
    await Promise.all([this.setDevicesAutoCompleteOptions(), this.setRestaurantsAutoCompleteOptions()]);
  }

  devicePageChange(event) {
    
  }
  // Commented for future reference 
  //async getTypes() {
  //   this.types = await this.devicesApi.getAssetTypes();
  // }

  getComponentTypeName(typeId) {
    return this.types.find(x => x.id == typeId).name;
  }

  pageData(event: TablePageChangeEvent): void {
    this.page = event.page;
    this.pageSize = event.pageSize;
    this.sortBy = event.sortBy;
    this.sortDirection = event.sortDirection;
    this.updateQueryParams();
  }

  filterData(event): void {
    this.page = 1;
    this.selectedFilters = {
      componentNameId: event.filters.componentNameId,
      storeNameId: event.filters.storeNameId,
      componentName: event.filters.componentNameId?.filter((x) => parseInt(x).toString() !== x),
      componentId: event.filters.componentNameId?.filter((x) => parseInt(x).toString() === x),
      storeName: event.filters.storeNameId?.filter((x) => parseInt(x).toString() !== x),
      storeId: event.filters.storeNameId?.filter((x) => parseInt(x).toString() === x)
    };

    this.loadData(
      this.page,
      this.pageSize,
      this.sortBy,
      this.sortDirection,
      this.selectedFilters
    );

    this.updateQueryParams();
  }

  navigateToDeviceDetails(event: TableRowClickEvent<Device>) {
    this.router.navigate([event.item.id], { relativeTo: this.route });
  }

  private async loadData(
    page: number,
    pageSize: number,
    sortBy: string,
    sortDirection: TableSortDirection,
    filters: ComponentFilters,
  ) {
    await this.devicesApi.getDevicesList(
      page,
      pageSize,
      sortBy,
      sortDirection,
      filters,
    ).then(response => {
      this.isLoading = false;
      this.totalDevices = response.totalComponentCount;
      this.devices = response.data;
    }).catch((error) => {
      this.isLoading = false;
    });
  }


  private async setDevicesAutoCompleteOptions(
    searchTerm?: string
  ): Promise<void> {
    const sortMatches = (a, b) => {
      a = a.label.toLowerCase();
      b = b.label.toLowerCase();
      if (a < b) {
        return -1;
      } else if (a > b) {
        return 1;
      } else {
        return 0;
      }
    }
    const deviceNameMatchesRequest =
      searchTerm?.length > 0
        ? this.devicesApi.searchDevices(searchTerm, FilterLikeTerm.name)
        : [];
    const deviceIdMatchesRequest =
      searchTerm?.length > 0
        ? this.devicesApi.searchDevices(searchTerm, FilterLikeTerm.id)
        : [];
    const deviceIdRequest =
      this.selectedFilters?.componentId?.length > 0
        ? this.devicesApi.getDevices(this.page, this.pageSize, this.sortBy, this.sortDirection, { componentId: this.selectedFilters.componentId }).then(x => x.data)
        : []
    const [deviceNameMatches, deviceIdMatches, deviceIdAll] = await Promise.all([deviceNameMatchesRequest, deviceIdMatchesRequest, deviceIdRequest]);
    const allOptions = [
      !!searchTerm ? { label: `View all results with '${searchTerm}'`, value: `%${searchTerm}%` } : { label: undefined, value: undefined },
      ...(this.selectedFilters?.componentName?.map((x) => ({ label: `View all results with '${x.replace(/%/g, '')}'`, value: `%${x.replace(/%/g, '')}%` })) || []),
      ...(deviceIdAll?.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id })) || []),
      ...[...deviceNameMatches.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id })),
      ...deviceIdMatches.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id }))].sort(sortMatches)
    ];

    this.filters[0].options = allOptions.filter((x) => !!x.value).reduce((acc, curr) => {
      if (!acc.find((x) => x.label === curr.label)) {
        acc.push(curr);
      }
      return acc;
    }, [])
  }

  private async setRestaurantsAutoCompleteOptions(
    searchTerm?: string
  ): Promise<void> {
    const sortMatches = (a, b) => {
      a = a.label.toLowerCase();
      b = b.label.toLowerCase();
      if (a < b) {
        return -1;
      } else if (a > b) {
        return 1;
      } else {
        return 0;
      }
    }
    const storeNameMatchesRequest =
      searchTerm?.length > 0
        ? this.restaurantsApi.searchRestaurants(searchTerm, FilterLikeTerm.name)
        : [];
    const storeIdMatchesRequest =
      searchTerm?.length > 0
        ? this.restaurantsApi.searchRestaurants(searchTerm, FilterLikeTerm.id)
        : [];
    const storeIdRequest =
      this.selectedFilters?.storeId?.length > 0
        ? this.restaurantsApi.getStores(this.page, this.pageSize, this.sortBy, this.sortDirection, { storeId: this.selectedFilters.storeId }).then(x => x.data)
        : []
    const [storeNameMatches, storeIdMatches, storeIdAll] = await Promise.all([storeNameMatchesRequest, storeIdMatchesRequest, storeIdRequest]);
    const allOptions = [
      !!searchTerm ? { label: `View all results with '${searchTerm}'`, value: `%${searchTerm}%` } : { label: undefined, value: undefined },
      ...(this.selectedFilters?.storeName?.map((x) => ({ label: `View all results with '${x.replace(/%/g, '')}'`, value: `%${x.replace(/%/g, '')}%` })) || []),
      ...(storeIdAll?.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id })) || []),
      ...[...storeNameMatches.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id })),
      ...storeIdMatches.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id }))].sort(sortMatches)
    ];

    this.filters[1].options = allOptions.filter((x) => !!x.value).reduce((acc, curr) => {
      if (!acc.find((x) => x.label === curr.label)) {
        acc.push(curr);
      }
      return acc;
    }, [])
  }

  private updateQueryParams() {
    this.router.navigate([], {
      relativeTo: this.route,
      queryParams: {
        componentName: this.selectedFilters.componentName?.length > 0 ? this.selectedFilters.componentName : undefined,
        componentId: this.selectedFilters.componentId?.length > 0 ? this.selectedFilters.componentId : undefined,
        storeName: this.selectedFilters.storeName?.length > 0 ? this.selectedFilters.storeName : undefined,
        storeId: this.selectedFilters.storeId?.length > 0 ? this.selectedFilters.storeId : undefined,
        page: this.page,
        pageSize: this.pageSize,
        sortBy: this.sortBy,
        sortDirection: this.sortDirection,
      },
    });
  }

  showOnboardingDialog() {
    this.dialog.open(
      DeviceOnboardingComponent
    );
  }

  /**
   * 
   * @param $event to get the item counts
   */
  getItems($event){
    this.itemCount=$event
   }

   /**
   * Called on clear and reset to hide the reset buttons
   */
  filterCleared(){
    this.configReset=false;
  }
  
   /**
   * @param $event click event
   * This function is triggered on load and on every apply and clear of filters
   */
  updateApply($event){
    if(this.hideResetFiltersOnLoad === 0) {
      this.configReset = false;
      this.hideResetFiltersOnLoad++;
    } else {
      this.configReset = true;
    }
    if($event[2]==='apply'){
      this.eventSet.add($event[0])
    }
    else{
      this.eventSet.delete($event[0])
      if(this.eventSet.size===1){
        this.filterCleared()
      }
  }
  }
}
