import { async, ComponentFixture, TestBed } from '@angular/core/testing';
import { RouterTestingModule } from '@angular/router/testing';

import { DevicesListComponent } from './devices-list.component';
import { ComponentStatusPillComponent } from '../../components/component-status-pill/component-status-pill.component';
import { Router } from '@angular/router';
import { ComponentsApiService } from '../../services/components-api.service';
import { ComponentList } from '../../interfaces/component.interface';
import { CoreModule } from '@bref/core';
import { SolutionsNavigationBarComponent } from '../../components/solutions-navigation-bar/solutions-navigation-bar.component';
import { AssetType } from '../../interfaces/component-filters.interface';

class ComponentsApiServiceMock {
  async getAssetTypes(): Promise<AssetType[]> {
    return [
      {id: '1', name: 'adipiscing'}
    ];
  }
  async getDevices(): Promise<ComponentList> {
    return {
      data: [
        {
          id: '29386',
          status: 'active',
          name: 'porttitor eros',
          model: 'lacus.',
          type: '1',
          store: '65429',
          storeId: '123',
          function: 'urna. Vivamus',
          ipAddress: '16380723 .1158',
          cellLocation: null,
          certificate: null,
          description: null,
          lineOfBusiness: null,
          mainUsage: null,
          manufacturer: null,
          supplier: null,
          market: null,
          serialNumber: null,
          technicalName: null,
          warranty: null,
          version: '1.3',
          laneNumber: '1',
          macAddress: 'DC:AC:7E:G3:AB:D9',
          onboardingStatus: 'Onboarding message sent',
          installDate: '',
          hardwareVersion: '0',
          hierarchyNode: 50080,
          isIOTDevice: true
        }
      ],
      totalComponentCount: 42
    };
  }
}

xdescribe('DevicesListComponent', () => {
  let component: DevicesListComponent;
  let fixture: ComponentFixture<DevicesListComponent>;
  let router: Router;

  beforeEach(async(() => {
    TestBed.configureTestingModule({
      declarations: [DevicesListComponent, ComponentStatusPillComponent, SolutionsNavigationBarComponent],
      imports: [RouterTestingModule, CoreModule],
      providers: [
        { provide: ComponentsApiService, useValue: new ComponentsApiServiceMock() }
      ]
    })
      .compileComponents();
  }));

  beforeEach(() => {
    fixture = TestBed.createComponent(DevicesListComponent);
    component = fixture.componentInstance;
    fixture.detectChanges();
    router = TestBed.inject(Router);
  });

  it('should create', () => {
    expect(component).toBeTruthy();
  });

  it('should display a filter group with two auto-complete components', () => {
    expect(fixture.nativeElement.querySelector('.filter-group .filter-auto-complete:first-child')).toBeTruthy();
    expect(fixture.nativeElement.querySelector('.filter-group .filter-auto-complete:nth-child(2)')).toBeTruthy();
  });

  it('should display table with the correct columns', async(() => {
    fixture.whenStable().then(() => {
      fixture.detectChanges();

      const columns = fixture.nativeElement.querySelectorAll('table th');
      expect(columns.length).toEqual(6); // return to 8 once Status and Function are valid columns
      expect(columns[0].innerHTML).toContain('ID');
      // expect(columns[1].innerHTML).toContain('Status');
      expect(columns[1].innerHTML).toContain('Display Name');
      expect(columns[2].innerHTML).toContain('Model');
      expect(columns[3].innerHTML).toContain('Type');
      expect(columns[4].innerHTML).toContain('Store');
      // expect(columns[6].innerHTML).toContain('Function');
      expect(columns[5].innerHTML).toContain('IP Address');

      const firstRowColumns = [...fixture.nativeElement.querySelectorAll('table tbody tr:first-child td')].map(x => x.innerHTML);
      expect(firstRowColumns[0]).toContain('29386');
      // expect(firstRowColumns[1]).toContain('Active');
      expect(firstRowColumns[1]).toContain('porttitor eros');
      expect(firstRowColumns[2]).toContain('lacus.');
      expect(firstRowColumns[3]).toContain('adipiscing');
      expect(firstRowColumns[4]).toContain('65429');
      // expect(firstRowColumns[6]).toContain('urna. Vivamus');
      expect(firstRowColumns[5]).toContain('16380723 .1158');
    });
  }));

  it('should display gutter with open ticket button', () => {
    //todo: test actual functionality once implemented
    expect(fixture.nativeElement.querySelector('.devices-list-gutter .devices-list-card-action button').innerHTML).toContain('Open a Ticket');
  });

  it('should display gutter with onboard device button', () => {
    //todo: test actual functionality once implemented
    expect(fixture.nativeElement.querySelector('.devices-list-gutter .devices-list-onboard button').innerHTML).toContain('Onboard a Device');
  });

  it('should set a default page, page size, sort by and sort direction if none are declared in query params', () => {
    expect(component.page).toBe(1);
    expect(component.pageSize).toBe(25);
    expect(component.sortBy).toBe('id');
    expect(component.sortDirection).toBe('asc');
  });

  it('should paginate, sort and filter data based on query params', async(() => {
    fixture.ngZone.run(() => {
      router.navigate([], { queryParams: { page: 1, pageSize: 100, sortBy: 'name', sortDirection: 'desc', componentName: ['%fryer%', '%brand%'] } }).then(() => {
        component.ngOnInit();
        fixture.whenStable().then(() => {
          expect(component.page).toBe(1);
          expect(component.pageSize).toBe(100);
          expect(component.sortBy).toBe('name');
          expect(component.sortDirection).toBe('desc');
          expect(component.selectedFilters.componentName).toEqual(['%fryer%', '%brand%']);
          expect(component.filters[0].options).toEqual([{ label: 'View all results with \'fryer\'', value: '%fryer%' }, { label: 'View all results with \'brand\'', value: '%brand%' }]);
        });
      });
    });
  }));
});
