import { Component, OnInit, Input, Output, EventEmitter, ViewEncapsulation } from '@angular/core';

import { Filter, FilterType, FiltersChangeEvent, PageSizes, TablePageChangeEvent, TableSortDirection } from '@bref/core';
import { CertificatesFilters } from '../../interfaces/certificates-filters.interface';
import { Certificates } from '../../interfaces/certificates.interface'
import { ActivatedRoute, Router } from '@angular/router';
import { CertificatesApiService } from '../../services/certificates-api.service';
import { datesToArray } from '../../utility/general-utils';

@Component({
  selector: 'bre-certificates-list',
  templateUrl: './certificates-list.component.html',
  styleUrls: ['./certificates-list.component.scss'],
  encapsulation: ViewEncapsulation.None
})
export class CertificatesListComponent implements OnInit {
  certificates: Certificates[] = [];
  totalCertificates: number;
  certificatesPage: number;
  certificatesPageSize: number;
  notifyPageChange: EventEmitter<any> = new EventEmitter();
  page: number;
  pageSize: PageSizes;
  sortBy: string;
  sortDirection: TableSortDirection;
  selectedFilters: CertificatesFilters = {};
  isLoading: boolean = false;


  constructor(private route: ActivatedRoute, private router: Router, private api: CertificatesApiService) { }

  filters: Filter[] = [
    {
      name: 'stores',
      type: FilterType.autocomplete,
      placeholder: "Store ID or Name",
      options: [],
      loadOptions: query => {
        this.setCertificateFiltersAutoCompleteOptions(query, "storeName")
      },
      isAdvanced: false,
    },
    {
      name: 'component',
      type: FilterType.autocomplete,
      placeholder: "Device/Solution",
      options: [],
      loadOptions: query => {
        this.setCertificateFiltersAutoCompleteOptions(query, "component")
      },
      isAdvanced: false,
    },
    {
      name: 'status',
      type: FilterType.multiselect,
      placeholder: "Status(es)",
      options: this.loadFilterOptions('status'),
      isAdvanced: false,
    }
  ];

  ngOnInit(): void {
    this.isLoading = true;
    //On page load if the route params are set they are taken while loading the data else default values are taken
    this.totalCertificates = 0;
    this.certificatesPage = parseInt(this.route.snapshot.queryParams.page, 10) ? parseInt(this.route.snapshot.queryParams.page, 10) : 1;
    this.certificatesPageSize = parseInt(this.route.snapshot.queryParams.pageSize, 10) ? parseInt(this.route.snapshot.queryParams.pageSize, 10) : PageSizes.TwentyFive;
    this.page = parseInt(this.route.snapshot.queryParams.page, 10) || 1;
    this.pageSize = parseInt(this.route.snapshot.queryParams.pageSize, 10) || PageSizes.TwentyFive;
    this.sortBy = this.route.snapshot.queryParams.sortBy || 'expiry';
    this.sortDirection = this.route.snapshot.queryParams.sortDirection as TableSortDirection || TableSortDirection.asc;
    const queryParams = this.route.snapshot.queryParamMap;
    if(Object.keys(this.route.snapshot.paramMap["params"]).length > 0) {
      this.selectedFilters = {
        certificateARN: queryParams.getAll('certificateARN'),
        component: queryParams.getAll('component'),
        startDate: datesToArray(this.route.snapshot.queryParams.startDate)?.map(x => new Date(+x)),
        endDate: datesToArray(this.route.snapshot.queryParams.endDate)?.map(x => new Date(+x)),
        status: [this.route.snapshot.paramMap.get('statusActive'), this.route.snapshot.paramMap.get('statusAttention')],
        deviceId: this.route.snapshot.paramMap.get('deviceId')
      };
      this.updateQueryParams();
    } else {
      this.selectedFilters = {
        certificateARN: queryParams.getAll('certificateARN'),
        component: queryParams.getAll('component'),
        storeName: queryParams.getAll('storeName'),
        startDate: datesToArray(this.route.snapshot.queryParams.startDate)?.map(x => new Date(+x)),
        endDate: datesToArray(this.route.snapshot.queryParams.endDate)?.map(x => new Date(+x)),
        status: queryParams.getAll('status')
      };
    }
    
    this.loadData(this.page, this.pageSize, this.sortBy, this.sortDirection, this.selectedFilters);
  }

  /**
    * Takes one parameter and load the data on UI based on selected Page number
    * @param event Event with payload when any pagination button is clicked
    */
  pageData(event) {
    this.certificatesPage = event.page;
    this.certificatesPageSize = event.pageSize;
    this.page = this.certificatesPage;
    this.pageSize = this.certificatesPageSize;
    this.sortBy = event.sortBy;
    this.sortDirection = event.sortDirection ? event.sortDirection : TableSortDirection.desc;
    this.loadData(this.page, this.pageSize, this.sortBy, this.sortDirection, this.selectedFilters);
    this.updateQueryParams();
  }


  /**
    * Takes one parameter when filter event is triggered
    * @param event Event with payload when any change in filters data
    */
  filterData(event: FiltersChangeEvent): void {
    this.page = 1;
    this.selectedFilters = {
      ...event.filters,
      certificateARN: event.filters.certificateARN === undefined ? undefined : event.filters.certificateARN,
      component: event.filters.component === undefined ? undefined : event.filters.component,
      status: event.filters.status?.filter(x => typeof x === 'string'),
      storeName: event.filters.stores === undefined ? (this.route.snapshot.paramMap.get('storeID') ?  [this.route.snapshot.paramMap.get('storeID')] : undefined) : event.filters.stores, // ['13087'],
      startDate: event.filters.expiry !== undefined && event.filters.expiry[0] !== undefined ? event.filters.expiry[0]?.getTime() : undefined,
      endDate: event.filters.expiry !== undefined && event.filters.expiry[1] !== undefined ? event.filters.expiry[1]?.getTime() : undefined,
    };

    this.loadData(this.page, this.pageSize, this.sortBy, this.sortDirection, this.selectedFilters);
    this.updateQueryParams();
  }

    /**
    * Function helps to load the values in autocomplete boxes (StoreName, Device/Solution etc.) based on type searchTerm and parameter
    * @param type type of filter box
    */
  private async setCertificateFiltersAutoCompleteOptions(searchTerm?: string, filterType?: string) {
    if (filterType === "storeName") {
      var searchTermMatches = searchTerm?.length > 0
        ? await this.api.getCertificates(1, 100, 'id', TableSortDirection.desc, { storeName: [searchTerm] }).then(x => x.data) : [];

      var allStores = await this.api.getCertificates(1, 100, 'id', TableSortDirection.desc, null).then(x => x.data);
      this.filters[0].options = [
        ...searchTermMatches.map(x => ({ label: `${x.restaurantName}`, value: x.restaurantName })),
        ...allStores.map(x => ({ label: `${x.restaurantName}`, value: x.restaurantName }))
      ].reduce((options, x) => {
        if (x.value && !options.find(o => o.value === x.value)) {
          options.push(x);
        }
        return options;
      }, []);
    }
    else if (filterType === "component") {
      var searchTermMatches = searchTerm?.length > 0
        ? await this.api.getCertificates(1, 100, 'id', TableSortDirection.desc, { component: [searchTerm] }).then(x => x.data) : [];

      var allComponents = await this.api.getCertificates(1, 100, 'id', TableSortDirection.desc, null).then(x => x.data);
      this.filters[1].options = [
        ...searchTermMatches.map(x => ({ label: `${x.componentName}`, value: x.componentName })),
        ...allComponents.map(x => ({ label: `${x.componentName}`, value: x.componentName }))
      ].reduce((options, x) => {
        if (x.value && !options.find(o => o.value === x.value)) {
          options.push(x);
        }
        return options;
      }, []);
    }
    else if (filterType === "certificateARN") {
      var searchTermMatches = searchTerm?.length > 0
        ? await this.api.getCertificates(1, 100, 'id', TableSortDirection.desc, { certificateARN: [searchTerm] }).then(x => x.data) : [];

      var allCertificates = await this.api.getCertificates(1, 100, 'id', TableSortDirection.desc, null).then(x => x.data);
      this.filters[2].options = [
        ...searchTermMatches.map(x => ({ label: `${x.componentConfigValue}`, value: x.componentConfigValue })),
        ...allCertificates.map(x => ({ label: `${x.componentConfigValue}`, value: x.componentConfigValue }))
      ].reduce((options, x) => {
        if (x.value && !options.find(o => o.value === x.value)) {
          options.push(x);
        }
        return options;
      }, []);
    }
  }

  /**
    * Function helps to load the values in filters boxes based on type parameter
    * @param type type of filter box
    */
  private loadFilterOptions(type) {
    let options = [];

    switch (type) {
      case 'status':
        options = [{ label: 'Expired', value: 'expired' },
        { label: 'Needs Attention', value: 'attention' },
        { label: 'Active', value: 'active' }];
        break;
    }
    return options;
  }

  /**
    * Function helps to update the query parameters for filtered data
    */
  private updateQueryParams() {
    this.router.navigate([], {
      relativeTo: this.route,
      queryParams: {
        certificateARN: this.selectedFilters.certificateARN !== undefined ? this.selectedFilters.certificateARN : undefined,
        storeName: this.selectedFilters.storeName !== undefined ? this.selectedFilters.storeName : undefined,
        component: this.selectedFilters.component?.length > 0 ? this.selectedFilters.component : undefined,
        status: this.selectedFilters.status?.length > 0 ? this.selectedFilters.status : undefined,
        page: this.page,
        pageSize: this.pageSize,
        sortBy: this.sortBy,
        sortDirection: this.sortDirection,
        startDate: this.selectedFilters.startDate !== undefined ? this.selectedFilters.startDate : undefined,
        endDate: this.selectedFilters.endDate !== undefined ? this.selectedFilters.endDate : undefined
      }
    })
  }

  /**
    * Takes three parameters and load the data on UI based on selectedFilters
    * @param page page number
    * @param pageSize Number of rows in table
    * @param filters query parameters values for filters
    */
  private async loadData(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection, filters: CertificatesFilters) {
    const response = await this.api.getCertificates(page, pageSize, sortBy, sortDirection, filters);
    this.totalCertificates = response.totalItemCount;
    this.certificates = response.data;
    this.isLoading = false;
  }
}