import { Component, OnInit, ViewEncapsulation } from '@angular/core';
import { DialogService, FilterType, Filter, TablePageChangeEvent, PageSizes, TableSortDirection } from '@bref/core';
import { ActivatedRoute, Router } from '@angular/router';
import { Component as Asset } from '../../interfaces/component.interface';
import { ComponentFilters, FilterLikeTerm, AssetType } from '../../interfaces/component-filters.interface';
import { ComponentsApiService } from '../../services/components-api.service';
import { OnboardingDialogComponent } from '../../components/onboarding-dialog/onboarding-dialog.component';
import { RestaurantsApiService } from '../../services/restaurants-api.service';

@Component({
  selector: 'bre-assets-list',
  templateUrl: './assets-list.component.html',
  styleUrls: ['./assets-list.component.scss'],
  encapsulation: ViewEncapsulation.None
})
export class AssetsListComponent implements OnInit {
  assets: Asset[] = [];
  totalAssets: number = 0;

  page: number;
  pageSize: PageSizes;
  sortBy: string;
  sortDirection: TableSortDirection;
  types: AssetType[];

  selectedFilters: ComponentFilters = {};
  filters: Filter[] = [
    {
      name: 'componentNameId',
      type: FilterType.autocomplete,
      placeholder: 'Search Software & Assets',
      options: [],
      loadOptions: (searchTerm: string) =>
        this.setComponentsAutoCompleteOptions(searchTerm),
      isAdvanced: false
    },
    {
      name: 'componentType',
      type: FilterType.multiselect,
      placeholder: 'Type',
      options: [],
      isAdvanced: false
    },
  ];

  constructor(
    private componentsApi: ComponentsApiService,
    private route: ActivatedRoute,
    private dialog: DialogService,
    private router: Router
  ) { }

  ngOnInit(): void {
    this.page = parseInt(this.route.snapshot.queryParams.page, 10) || 1;
    this.pageSize = parseInt(this.route.snapshot.queryParams.pageSize, 10) || PageSizes.TwentyFive;
    this.sortBy = this.route.snapshot.queryParams.sortBy || 'id';
    this.sortDirection = this.route.snapshot.queryParams.sortDirection || TableSortDirection.asc;

    this.getTypes();

    const queryParams = this.route.snapshot.queryParamMap;
    this.selectedFilters = {
      componentId: queryParams.getAll('componentId'),
      componentName: queryParams.getAll('componentName'),
      componentNameId: [...queryParams.getAll('componentName'), ...queryParams.getAll('componentId')],
      componentType: queryParams.getAll('componentType')
    };

    this.loadData(this.page, this.pageSize, this.sortBy, this.sortDirection, this.selectedFilters);
    this.setComponentsAutoCompleteOptions();
  }

  async getTypes() {
    this.types = await this.componentsApi.getAssetTypes();
    this.filters[1].options = this.types.map(x => ({label: x.name, value: x.id}));
  }

  getAssetTypeName(typeId) {
    return this.types.find(x => x.id == typeId).name;
  }

  pageData(event: TablePageChangeEvent): void {
    this.page = event.page;
    this.pageSize = event.pageSize;
    this.sortBy = event.sortBy;
    this.sortDirection = event.sortDirection;

    this.loadData(this.page, this.pageSize, this.sortBy, this.sortDirection, this.selectedFilters);

    this.updateQueryParams();
  }

  filterData(event): void {
    this.page = 1;
    this.selectedFilters = {
      componentNameId: event.filters.componentNameId,
      componentName: event.filters.componentNameId?.filter((x) => parseInt(x).toString() !== x),
      componentId: event.filters.componentNameId?.filter((x) => parseInt(x).toString() === x),
      componentType: event.filters.componentType
    };

    this.loadData(this.page, this.pageSize, this.sortBy, this.sortDirection, this.selectedFilters);

    this.updateQueryParams();
  }

  private async loadData(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection, filters: ComponentFilters) {
    const response = await this.componentsApi.getAssets(page, pageSize, sortBy, sortDirection, filters);
    this.totalAssets = response.totalComponentCount;
    this.assets = response.data;
  }

  private async setComponentsAutoCompleteOptions(searchTerm?: string): Promise<void> {
    const assetNameMatchesRequest =
      searchTerm?.length > 0
        ? this.componentsApi.searchAssets(searchTerm, FilterLikeTerm.name)
        : [];
    const assetIdMatchesRequest =
      searchTerm?.length > 0
        ? this.componentsApi.searchAssets(searchTerm, FilterLikeTerm.id)
        : [];
    const assetIdRequest =
      this.selectedFilters?.componentId?.length > 0
        ? this.componentsApi.getAssets(this.page, this.pageSize, this.sortBy, this.sortDirection, { componentId: this.selectedFilters.componentId }).then(x => x.data)
        : []
    const [assetNameMatches, assetIdMatches, assetIdAll] = await Promise.all([assetNameMatchesRequest, assetIdMatchesRequest, assetIdRequest]);
    const allOptions = [
      !!searchTerm ? { label: `View all results with '${searchTerm}'`, value: `%${searchTerm}%` } : { label: undefined, value: undefined },
      ...(this.selectedFilters?.componentName?.map((x) => ({ label: `View all results with '${x.replace(/%/g, '')}'`, value: `%${x.replace(/%/g, '')}%` })) || []),
      ...(assetIdAll?.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id })) || []),
      ...[...assetNameMatches.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id })),
      ...assetIdMatches.map((x) => ({ label: `${x.id} - ${x.name}`, value: x.id }))]
    ];

    this.filters[0].options = allOptions.filter((x) => !!x.value).reduce((acc, curr) => {
      if (!acc.find((x) => x.label === curr.label)) {
        acc.push(curr);
      }
      return acc;
    }, []);
  }

  private updateQueryParams() {
    this.router.navigate([], {
      relativeTo: this.route,
      queryParams: {
        componentName: this.selectedFilters.componentName?.length > 0 ? this.selectedFilters.componentName : undefined,
        componentId: this.selectedFilters.componentId?.length > 0 ? this.selectedFilters.componentId : undefined,
        componentType: this.selectedFilters.componentType?.length > 0 ? this.selectedFilters.componentType : undefined,
        page: this.page,
        pageSize: this.pageSize,
        sortBy: this.sortBy,
        sortDirection: this.sortDirection,
      },
    });
  }

  goToServiceCafe() {
    window.open('https://servicecafe.service-now.com/');
  }

}
