import { async, ComponentFixture, TestBed } from '@angular/core/testing';
import { RouterTestingModule } from '@angular/router/testing';

import { AssetsListComponent } from './assets-list.component';
import { ComponentStatusPillComponent } from '../../components/component-status-pill/component-status-pill.component';
import { Router } from '@angular/router';
import { ComponentsApiService } from '../../services/components-api.service';
import { ComponentList } from '../../interfaces/component.interface';
import { CoreModule } from '@bref/core';
import { SolutionsNavigationBarComponent } from '../../components/solutions-navigation-bar/solutions-navigation-bar.component';
import { AssetType } from '../../interfaces/component-filters.interface';

class ComponentsApiServiceMock {
  async getAssetTypes(): Promise<AssetType[]> {
    return [
      {id: '1', name: 'adipiscing'}
    ];
  }
  async getAssets(): Promise<ComponentList> {
    return {
      data: [
        {
          id: '29386',
          status: 'active',
          name: 'porttitor eros',
          model: 'lacus.',
          type: '1',
          store: '65429',
          storeId: '123',
          function: 'urna. Vivamus',
          ipAddress: '16380723 .1158',
          cellLocation: null,
          certificate: null,
          description: null,
          lineOfBusiness: null,
          mainUsage: null,
          manufacturer: null,
          supplier: null,
          market: 'market',
          serialNumber: null,
          technicalName: null,
          warranty: null,
          version: '1.3',
          laneNumber: '1',
          macAddress: 'DC:AC:7E:G3:AB:D9',
          onboardingStatus: 'Onboarding message sent',
          installDate: '',
          hardwareVersion: '0',
          hierarchyNode: 50080,
          isIOTDevice: true
        }
      ],
      totalComponentCount: 42
    };
  }
}

describe('AssetsListComponent', () => {
  let component: AssetsListComponent;
  let fixture: ComponentFixture<AssetsListComponent>;
  let router: Router;

  beforeEach(async(() => {
    TestBed.configureTestingModule({
      declarations: [AssetsListComponent, ComponentStatusPillComponent, SolutionsNavigationBarComponent],
      imports: [RouterTestingModule, CoreModule],
      providers: [
        { provide: ComponentsApiService, useValue: new ComponentsApiServiceMock() }
      ]
    })
      .compileComponents();
  }));

  beforeEach(() => {
    fixture = TestBed.createComponent(AssetsListComponent);
    component = fixture.componentInstance;
    fixture.detectChanges();
    router = TestBed.inject(Router);
  });

  it('should create', () => {
    expect(component).toBeTruthy();
  });

  it('should display a filter group with auto-complete and multi-select components', () => {
    expect(fixture.nativeElement.querySelectorAll('.filter-group .filter-auto-complete').length).toBe(1);
    expect(fixture.nativeElement.querySelectorAll('.filter-group .filter-multi-select').length).toBe(1);
  });

  it('should display table with the correct columns', async(() => {
    fixture.whenStable().then(() => {
      fixture.detectChanges();

      const columns = fixture.nativeElement.querySelectorAll('table th');
      expect(columns.length).toEqual(6); // return to 7 once Status is a valid column
      expect(columns[0].innerHTML).toContain('ID');
      expect(columns[1].innerHTML).toContain('Name');
      expect(columns[2].innerHTML).toContain('Model');
      expect(columns[3].innerHTML).toContain('Running Version');
      expect(columns[4].innerHTML).toContain('Type');
      expect(columns[5].innerHTML).toContain('Store');

      const firstRowColumns = [...fixture.nativeElement.querySelectorAll('table tbody tr:first-child td')].map(x => x.innerHTML);
      expect(firstRowColumns[0]).toContain('29386');
      expect(firstRowColumns[1]).toContain('porttitor eros');
      expect(firstRowColumns[2]).toContain('lacus.');
      expect(firstRowColumns[3]).toContain('1.3');
      expect(firstRowColumns[4]).toContain('adipiscing');
      expect(firstRowColumns[5]).toContain('market-123');
    });
  }));

  /* uncomment once gutter is uncommented
  it('should display gutter with open ticket button', () => {
    expect(fixture.nativeElement.querySelector('.assets-list-gutter .assets-list-card-action button').innerHTML).toContain('Open a Ticket');
  });

  it('should display gutter with onboard solution button', () => {
    expect(fixture.nativeElement.querySelector('.assets-list-gutter .assets-list-onboard button').innerHTML).toContain('Onboard a Solution');
  });
  */

  it('should set a default page, page size, sort by and sort direction if none are declared in query params', () => {
    expect(component.page).toBe(1);
    expect(component.pageSize).toBe(25);
    expect(component.sortBy).toBe('id');
    expect(component.sortDirection).toBe('asc');
  });

  it('should paginate, sort and filter data based on query params', async(() => {
    fixture.ngZone.run(() => {
      router.navigate([], { queryParams: { page: 1, pageSize: 100, sortBy: 'name', sortDirection: 'desc', componentName: ['fryer', 'brand'] } }).then(() => {
        component.ngOnInit();
        fixture.whenStable().then(() => {
          expect(component.page).toBe(1);
          expect(component.pageSize).toBe(100);
          expect(component.sortBy).toBe('name');
          expect(component.sortDirection).toBe('desc');
          expect(component.selectedFilters.componentName).toEqual(['fryer', 'brand']);
          expect(component.filters[0].options).toEqual([{ label: 'View all results with \'fryer\'', value: '%fryer%' }, { label: 'View all results with \'brand\'', value: '%brand%' }]);
        });
      });
    });
  }));
});
