import { Component, OnInit } from '@angular/core';
import { DefaultKeyFilters, ToastService } from '@bref/core';
import { SelectItem } from 'primeng/api/selectitem';
import { OnboardingApiService } from '../../services/onboarding-api.service';
import { FormBuilder, FormGroup, Validators, FormControl } from '@angular/forms';
import { Component as Device } from '../../interfaces/component.interface';
import { distinct } from "rxjs/operators";
import { RestaurantsApiService } from '../../services/restaurants-api.service';
import { Vendor } from '../../interfaces/vendor.interface';
import { Router } from '@angular/router';

@Component({
  selector: 'bre-onboarding-dialog',
  templateUrl: './onboarding-dialog.component.html',
  styleUrls: ['./onboarding-dialog.component.scss']
})
export class OnboardingDialogComponent implements OnInit {
  public currentPageNumber: number = 1;

  //page 1
  public readonly componentTypes = {
    iotDevice: 'iot',
    solution: 'solution',
    nonIoTDevice: 'non-iot'
  };

  //Default Action
  public readonly defaultAction = {
    axis: 'snapshot',
    audioBridge: 'ab_action',
    nexeo: 'nexeo_heartbeat'
  };
  public selectedType: string;
  public selectedVendor: string
  public selectedVendorType: string;
  public warrantyDate: Date;

  //page 2
  public availableDevices: SelectItem[];
  public selectedDevice: Device;

  //page 3
  public vendors: Vendor[];
  public modelNumberOptions: SelectItem[];
  public hardwareVersionOptions: SelectItem[];
  public firmwareVersionOptions: SelectItem[];
  public laneOptions: SelectItem[];
  public stores: SelectItem[];
  public suppliers: SelectItem[];
  public onboardingForm: FormGroup;
  submittingForm: boolean = false;
  public serialNumber;
  public hardwareVersion;
  public ipAddress;
  public macAddress;
  public keyFilters = {
    ipAddress: /^(?=\d+\.\d+\.\d+\.\d+$)(?:(?:25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9][0-9]|[0-9])\.?){4}$/,
    macAddress: /^([0-9a-fA-F][0-9a-fA-F]:){5}([0-9a-fA-F][0-9a-fA-F])$/
  }

  //page 4
  public deviceName: string;

  constructor(private onboardingApi: OnboardingApiService, private restaurantsApi: RestaurantsApiService, private fb: FormBuilder, private toastService: ToastService, private router: Router) {
    this.serialNumber = {
      value: ''
    }
    this.ipAddress = {
      regex: this.keyFilters.ipAddress,
      value: '',
      showError: false
    }
    this.macAddress = {
      regex: this.keyFilters.macAddress,
      value: '',
      showError: false
    }
    this.hardwareVersion = {
      value: ''
    }
  }

  ngOnInit() {
    this.getVendorSuppliers();
    this.getStoreOptions();
    this.onboardingForm = this.fb.group({

      store: [undefined, Validators.required],
      deviceDetailsList: this.fb.array([
        this.fb.group({
          deviceType: 'iot-device',
          vendor: [undefined, Validators.required],
          modelNumber: [undefined, Validators.required],
          serialNumber: ['', Validators.required],
          deviceName: [''],
          firmwareVersion: ['', Validators.required],
          hardwareVersion: [''],
          warranty: [this.warrantyDate],
          installDate: [new Date()]
        })
      ]),
    });

    this.onboardingForm.get('deviceDetailsList.0.vendor').valueChanges.subscribe(async (val) => {
      this.selectedVendorType = this.getVendorType(this.onboardingForm.get('deviceDetailsList.0.vendor').value)
      this.selectedVendor = this.getVendorName(this.onboardingForm.get('deviceDetailsList.0.vendor').value).toLowerCase();
      if (this.selectedVendor.toLowerCase() === 'axis') {
        this.selectType(this.componentTypes.nonIoTDevice);
        let deviceDetailsFormGroup = this.onboardingForm.get('deviceDetailsList.0') as FormGroup;
        deviceDetailsFormGroup.removeControl('deviceProps');
        let defaultAction = new FormControl();
        defaultAction.setValue(this.defaultAction.axis);
        let deviceProps = this.fb.group({
          laneNumber: ['', Validators.required],
          macAddress: ['', Validators.required],
          ipAddress: ['', Validators.required],
        })
        deviceDetailsFormGroup.addControl('action', defaultAction);
        deviceDetailsFormGroup.addControl('deviceProps', deviceProps);
        this.laneOptions = [];

        const laneNumbers = [{ "id": "1", "name": "Lane 1" }, { "id": "2", "name": "Lane 2" }];
        if (!!laneNumbers && laneNumbers.length > 0) {
          this.laneOptions = laneNumbers.map(x => ({
            label: x.name,
            value: x.id
          }));
        }
      }
      else if (this.selectedVendor.toLowerCase() === 'signal_essence') {
        this.selectType(this.componentTypes.nonIoTDevice);
        let deviceDetailsFormGroup = this.onboardingForm.get('deviceDetailsList.0') as FormGroup;
        deviceDetailsFormGroup.removeControl('deviceProps');
        let defaultAction = new FormControl();
        defaultAction.setValue(this.defaultAction.audioBridge);
        let deviceProps = this.fb.group({
          ipAddress: ['', Validators.required]
        })
        deviceDetailsFormGroup.addControl('action', defaultAction);
        deviceDetailsFormGroup.addControl('deviceProps', deviceProps);

      }
      else if (this.selectedVendor.toLowerCase() === 'nexeo') {
        this.selectType(this.componentTypes.nonIoTDevice);
        let deviceDetailsFormGroup = this.onboardingForm.get('deviceDetailsList.0') as FormGroup;
        deviceDetailsFormGroup.removeControl('deviceProps');
        let defaultAction = new FormControl();
        defaultAction.setValue(this.defaultAction.nexeo);
        let deviceProps = this.fb.group({
          ipAddress: ['', Validators.required]
        })
        deviceDetailsFormGroup.addControl('action', defaultAction);
        deviceDetailsFormGroup.addControl('deviceProps', deviceProps);

      }
      this.onboardingForm.get('deviceDetailsList.0.modelNumber').setValue(undefined);
      this.onboardingForm.get('deviceDetailsList.0.firmwareVersion').setValue('');
      this.onboardingForm.get('deviceDetailsList.0.hardwareVersion').setValue('');

      this.modelNumberOptions = [];
      this.firmwareVersionOptions = [];
      this.hardwareVersionOptions = [];

      const modelNumbers = await this.restaurantsApi.getModelNumberByVendor(this.getVendorName(this.onboardingForm.get('deviceDetailsList.0.vendor').value),'camera')
      if (!!modelNumbers && modelNumbers.length > 0) {
        this.modelNumberOptions = modelNumbers.map(x => ({
          label: x,
          value: x
        }));
      }
      const laneNumbers = [{ "id": "1", "name": "Lane 1" }, { "id": "2", "name": "Lane 2" }];
      if (!!laneNumbers && laneNumbers.length > 0) {
        this.laneOptions = laneNumbers.map(x => ({
          label: x.name,
          value: x.id
        }));
      }
    });

    this.onboardingForm.get('deviceDetailsList.0.modelNumber').valueChanges
      .pipe(distinct())
      .subscribe(async (value) => {
        this.onboardingForm.get('deviceDetailsList.0.firmwareVersion').setValue('');
        this.onboardingForm.get('deviceDetailsList.0.hardwareVersion').setValue('');

        this.firmwareVersionOptions = [];
        this.hardwareVersionOptions = [];

        if (!!value) {
          const firmwareVersions = await this.restaurantsApi.getFirmwareVersions(this.getVendorName(this.onboardingForm.get('deviceDetailsList.0.vendor').value),'camera', this.onboardingForm.get('deviceDetailsList.0.modelNumber').value);
          this.firmwareVersionOptions = firmwareVersions.map(x => ({
            label: x,
            value: x
          }));
        }
      });
  }

  async getVendorSuppliers() {
    this.vendors = await this.restaurantsApi.getVendors();
    this.suppliers = this.vendors.map(x => ({
      label: `${x.id} - ${x.name}`,
      value: x.id
    }));
  }

  getVendorName(vendorId) {
    return this.vendors.find(vendor => {
      return vendor.id === vendorId
    }).name;
  }

  getVendorType(vendorId) {
    let deviceFlag: boolean
    deviceFlag = this.vendors.find(vendor => {
      return vendor.id === vendorId
    }).deviceFlag;
    return ((deviceFlag === true) ? "iot" : "non-iot");
  }

  async getStoreOptions() {
    const storeData = await this.restaurantsApi.getAllStores();
    this.stores = storeData.map(x => ({
      label: `${x.marketName} - ${x.id} - ${x.name}`,
      value: x.id
    }));
  }

  setWarrantyExpirationValue(event): void {
    if (event) {
      this.onboardingForm.get('deviceDetailsList.0.warranty').patchValue(event, { onlySelf: true, emitEvent: false });
    }
  }

  setInstallDateValue(event): void {
    if (event) {
      this.onboardingForm.get('deviceDetailsList.0.installDate').patchValue(event, { onlySelf: true, emitEvent: false });
    }
  }

  handlePageChange(pageNumber: number) {
    this.currentPageNumber = pageNumber;
  }

  validateInput(event): void {
    if (event) {
      let val = event.value;
      let expressionType = event.controlName;
      switch (expressionType) {
        case "serialNumber": {
          this.serialNumber.value = val;
          this.serialNumber.showError = !DefaultKeyFilters.num.test(val);
          break;
        }
        case "ipAddress": {
          this.ipAddress.value = val;
          this.ipAddress.showError = !this.keyFilters.ipAddress.test(val);
          break;
        }
        case "macAddress": {
          this.macAddress.value = val;
          this.macAddress.showError = !this.keyFilters.macAddress.test(val);
          break;
        }
      }
    }
  }

  validateMACAdd = {
    macAddRegex: this.keyFilters.macAddress,
    value: '',
    hasError: (value) => (value && !this.keyFilters.macAddress.test(value))
  }

  async selectType(type: string) {
    this.selectedType = type;
    this.availableDevices = [];
    this.availableDevices = (await this.onboardingApi.getAvailableDevices(type))
      .map(x => ({
        value: x,
        label: `${x.manufacturer} - ${x.model}`
      }));
  }

  onboardDevice(next, type: string) {
    if (type === this.componentTypes.iotDevice) {
      this.submittingForm = true;
      if (this.onboardingForm.valid) {
        this.onboardingForm.get('deviceDetailsList.0.deviceName').setValue(`${this.componentTypes.iotDevice}-${this.getVendorName(this.onboardingForm.get('deviceDetailsList.0.vendor').value).toLowerCase()}-${this.onboardingForm.get('deviceDetailsList.0.modelNumber').value}-${this.onboardingForm.get('deviceDetailsList.0.hardwareVersion').value}-${this.onboardingForm.get('deviceDetailsList.0.serialNumber').value}`);
        this.onboardingApi.addNewDevice(this.onboardingForm.value).then((response) => {
          this.submittingForm = false;
          this.deviceName = this.onboardingForm.get('deviceDetailsList.0.deviceName').value;
          this.submittingForm = false;
          next();
        });
      } else {
        const controls = {
          ...this.onboardingForm.get('deviceDetailsList.0')['controls'],
          store: this.onboardingForm.controls['store'],
        };
        const invalidControlNames = [];
        for (const name in controls) {
          if (controls[name].invalid) {
            invalidControlNames.push(name.replace(/([A-Z])/g, ' $1').replace(/^./, function (str) { return str.toUpperCase(); }));
          }
        }
        this.toastService.error(`Please fill out the following required fields: ${invalidControlNames.join(', ')}`);
        this.submittingForm = false;
      }
    } else if (type === this.componentTypes.nonIoTDevice) {
      this.submittingForm = true;
      if (this.onboardingForm.valid) {
        this.onboardingForm.get('deviceDetailsList.0.deviceName').setValue(`${this.componentTypes.nonIoTDevice}-${this.getVendorName(this.onboardingForm.get('deviceDetailsList.0.vendor').value).toLowerCase()}-${this.onboardingForm.get('deviceDetailsList.0.modelNumber').value}-${this.onboardingForm.get('deviceDetailsList.0.serialNumber').value}`);
        this.onboardingApi.addNewNonIoTDevice(this.onboardingForm.value).then((response) => {
          this.submittingForm = false;
          this.deviceName = this.onboardingForm.get('deviceDetailsList.0.deviceName').value;
          this.submittingForm = false;
          next();
        });
      } else {
        const controls = {
          ...this.onboardingForm.get('deviceDetailsList.0')['controls'],
          store: this.onboardingForm.controls['store'],
        };
        const invalidControlNames = [];
        for (const name in controls) {
          if (controls[name].invalid) {
            invalidControlNames.push(name.replace(/([A-Z])/g, ' $1').replace(/^./, function (str) { return str.toUpperCase(); }));
          }
        }
        if(invalidControlNames.includes('Device Props')){
          const index = invalidControlNames.indexOf('Device Props');
          if (index > -1) {
            invalidControlNames.splice(index, 1);
          }
          if (this.getVendorName(this.onboardingForm.get('deviceDetailsList.0.vendor').value).toLowerCase() === 'axis') {
            invalidControlNames.push('IP Address', 'MAC Address');
          }
          if (this.getVendorName(this.onboardingForm.get('deviceDetailsList.0.vendor').value).toLowerCase() === 'signal_essence') {
            invalidControlNames.push('IP Address');
          }
        }
        this.toastService.error(`Please fill out the following required fields: ${invalidControlNames.join(', ')}`);
        this.submittingForm = false;
      }
    }
  }

  navigateToDeviceInfo(closeFunction) {
    closeFunction();
    this.router.navigate(['solutions/devices'], { queryParams: { componentName: `%${this.deviceName}%` } }).then(() => {
      window.location.reload();
    });
  }
}