import { Component, Inject, OnInit } from "@angular/core";
import { FormBuilder } from "@angular/forms";
import { Vendor } from "../../interfaces/vendor.interface";
import { RestaurantsApiService } from "../../services/restaurants-api.service";
import { SelectItem } from 'primeng/api/selectitem';
import { SelectedDeviceDataService } from "../../services/selected-device-data.service";
import { OnboardingVendorDetailsComponent } from "../../components/onboarding-vendor-details/onboarding-vendor-details.component";
import { DialogService } from '@bref/core';
import { DOCUMENT } from '@angular/common';
import { DeviceFlag } from "../../interfaces/device-onboarding-api.interface";
import { ComponentsApiService } from '../../services/components-api.service';
import * as CONSTANTS from '../../constants/input-patterns';

@Component({
    selector: 'bre-add-device',
    templateUrl: './add-device.component.html',
    styleUrls: ['./add-device.component.scss']
})

export class AddDeviceComponent implements OnInit {
    activeTab: boolean[] = [];
    addedStores = {};
    storeOptions = [];
    nextStoreRowIdIndex = 0;
    nextPropertyRowIdIndex = {};
    vendors: Vendor[]
    vendorOptions: SelectItem[] = [];
    public suppliers: SelectItem[];
    firmwareOptions: SelectItem[] = [];
    displayNameOptions: SelectItem[] = [];
    modelNoOptions: SelectItem[] = [];
    deviceFlagList: DeviceFlag[] = [];
    todayDate: Date;
    market = [];
    region = [];
    coop = [];
    clone = [];
    deviceFlag = [];
    vendorBtndisabled: boolean = true;
    vendorLoading: boolean = false;
    vendorDialogData: any = {};
    activeIndex: any = 1;
    vendorDevice: any;
    showLoader: boolean;
    disableStoreNext: boolean;
    showIp: boolean = false;
    showMac: boolean = false;
    showLane: boolean = false;
    selectedModelNo: string;
    showError = [{ ip: [], mac: [], unitNo: [] }];
    disableAddDevice: boolean[] = [];
    hierarchyAvailable: boolean;
    vendorInfo: any;
    abbreviationOfSelectedDevice: string;
    isNumber: boolean = true;
    inputPatterns = CONSTANTS.INPUT_PATTERNS;
    constructor(
        private restaurantsApi: RestaurantsApiService,
        private fb: FormBuilder,
        private selectedDeviceService: SelectedDeviceDataService,
        private dialog: DialogService,
        @Inject(DOCUMENT) private document,
        private api: ComponentsApiService
    ) {
        this.showLoader = true;
        this.initialize();
    }

    ngOnInit() {
        this.disableAddDevice[0] = true;
        this.activeTab[0] = true;
        this.todayDate = new Date();
        this.selectedDeviceService.addStore.subscribe(() => {
            this.disableStoreNext = this.selectedDeviceService.isAnyNullOnDeviceObject();
        });
    }

    async initialize() {
        this.getStoreOptions();
        this.getVendorOptions();
        this.insertNewDevice();
    }

    /**
   * Get the row id
   * this function is called to get all the store ids
   */
    getAddedStoresRowIds() {
        return Object.keys(this.addedStores);
    }

    /**
   * Get the property id
   * @param rowId: row index
   * this function is called to get all the property id of the selected row
   */
    getPropertyIdsForRow(rowId) {
        return Object.keys(this.addedStores[rowId].value.properties);
    }

    /**
   * Check if there are fields with errors 
   * Error state is set and is being subscribed to enable or disable next button in onboarding flow.
   */
    checkForErrors(){
        this.showError.forEach(item => {
            for (let key in item) {
                let isError = item[key].filter(prop => {
                    return prop === true; 
                })              
                isError.length ? this.selectedDeviceService.setErrorState(true) : this.selectedDeviceService.setErrorState(false);    
              }
        });
    }

    /**
   * Check unit number pattern
   * @param event: input event
   * this function is called to check if unit number is of number format.
   */
     checkunitNoPattern(event, rowId, propId) {       
        if (event.value !== '' && event.value !== undefined && this.inputPatterns.UNIT_NO.test(event.value)) {        
           this.showError[rowId].unitNo[propId] = false; 
           this.updateDeviceDataToService();        
        }else{
            this.showError[rowId].unitNo[propId] = true;   
            this.updateDeviceDataToService();   
        }
        this.checkForErrors();
    }
    /**
   * Get store options
   * 
   * this function is called to retrieve store options from api
   */
    getStoreOptions() {
        this.showLoader = true;
        this.restaurantsApi.getAllStores().then((response) => {
            this.storeOptions = response.map(x => ({
                label: `${x.marketName} - ${x.id} - ${x.name}`,
                value: `${x.marketName} - ${x.id} - ${x.name}`
            }));
            this.showLoader = false;
        }).catch((error) => {
            this.showLoader = false;
        });
    }

    /**
   * Get vendor options
   * 
   * this function is called to retrieve vendor options from api
   */
    getVendorOptions() {
        this.restaurantsApi.getVendors().then((response) => {
            this.vendors = response;
            this.vendorOptions = response.map(x => ({
                label: `${x.id} - ${x.name}`,
                value: `${x.id} - ${x.name}`
            }));
            this.vendorOptions.forEach((vd) => {
                this.vendorDevice = {
                    ...this.vendorDevice,
                    [vd.value]: {
                        device: {
                            options: []
                        },
                        model: {
                            options: [],
                            firmware: {}
                        }
                    }
                }
            });
        }).catch((error) => {
            this.showLoader = false;
        });
        return this.vendorOptions;
    }

    /**
   * Get vendor name
   * @param vendorId: row index
   * this function is called to get vendor name from vendor id
   */
    getVendorName(vendorId) {
        return this.vendors.find(vendor => {
            return vendor.id === vendorId.split(' -')[0]
        }).name;
    }

    /**
     * @param selectedVendor: vendor selected from the dropdown
     * this function is fetching the devices related to selected vendor and mapping it to vendorDevice 
     */
    fetchDisplayName(selectedVendor) {
        this.restaurantsApi.getAllDevices(selectedVendor.split(' -')[0]).then((response) => {
            this.vendorDevice[selectedVendor].device.options = response.reduce((acc, prop) => {
                return [
                    ...acc,
                    {
                        label: prop.name,
                        value: prop.name
                    }
                ]
            }, []);
            this.deviceFlagList = response.map(x => ({
                value: x.name,
                deviceFlag: x.deviceFlag,
                id: x.id,
                abbreviation: x.abbreviation
            }))
        }).catch((error) => {
            this.showLoader = false;
        });

    }

    /**
   * Get display name
   * @param selectedVendor: vendor value selected from the dropdown
   * this function is called to map display name to the dropdowns
   */
    getDisplayName(selectedVendor) {
        if (
            selectedVendor !== "" &&
            selectedVendor !== null &&
            this.vendorDevice[selectedVendor]
        ) {
            return this.vendorDevice[selectedVendor].device.options;
        }
    }

    /**
     * @param selectedVendor: vendor selected from the dropdown
     * @param selectedDevice: device selected from the dropdown
     * this function is fetching the model nos related to selected vendor and mapping it to vendorDevice 
     */
    fetchModelNo(selectedVendor, selectedDevice) {
        this.restaurantsApi.getModelNumberByVendor(this.getVendorName(selectedVendor), selectedDevice).then((response) => {
            this.modelNoOptions = response.reduce((ven, prop) => {
                return [
                    ...ven,
                    {
                        label: prop,
                        value: prop
                    }
                ]
            }, [])
            this.vendorDevice[selectedVendor].model.options = this.modelNoOptions;
            this.modelNoOptions.forEach((mod) => {
                this.vendorDevice[selectedVendor].model.firmware = {
                    ...this.vendorDevice[selectedVendor].model.firmware,
                    [mod.value]: {
                        options: []
                    }
                }
            })
            this.showLoader = false;
        }).catch((error) => {
            this.showLoader = false;
        });
    }

    /**
   * Get model no options
   * @param rowId: row index
   * @param propertyRowId: parent row index
   * this function is called to retrieve model no options from api
   */
    getModelNoOptions(selectedVendor) {
        if (selectedVendor !== "" && selectedVendor !== null && this.vendorDevice[selectedVendor]) {
            return this.vendorDevice[selectedVendor].model.options;
        }
    }

    fetchFirmware(selectedVendor, selectedDevice, selectedModel) {
        this.restaurantsApi.getFirmwareVersions(this.getVendorName(selectedVendor), selectedDevice, selectedModel).then((response) => {
            this.vendorDevice[selectedVendor].model.firmware[selectedModel].options = response.reduce((firm, prop) => {
                return [
                    ...firm, {
                        label: prop,
                        value: prop
                    }
                ]
            }, [])
            this.showLoader = false;
        }).catch((error) => {
            this.showLoader = false;
        });
    }

    /**
   * Get firmware options
   * @param rowId: row index
   * @param propertyRowId: parent row index
   * this function is called to retrieve firmware options from api
   */
    getFirmwareOptions(selectedVendor, selectedModel) {
        if (selectedVendor !== '' && selectedVendor !== null && selectedModel !== '' && selectedModel !== null) {
            return this.vendorDevice[selectedVendor].model.firmware[selectedModel].options;
        }
    }

    /**
   * Get hierarchy
   * @param rowId: row index
   * @param parentId: parent node id of the selected store
   * this function is called when user selects a store, it provides market, coop and region details
   */
    getHierarchy(rowId, parentId) {
        this.restaurantsApi.getHierarchyDeploymentNodeValues(parentId).then((response) => {
            this.market[rowId] = response.deploymentMarket[0].attributes.mcdInternalNodeName;
            this.coop[rowId] = response.deploymentCoOp[0].attributes.mcdInternalNodeName;
            this.region[rowId] = response.deploymentRegion[0].attributes.mcdInternalNodeName;
            this.showLoader = false;
            this.hierarchyAvailable = true;
        }).catch((error) => {
            this.hierarchyAvailable = false;
            this.showLoader = false;
        })
    }

    /**
   * Store update
   * 
   * this function is called when user changes the store, 
   * it resets all the other elements
   * If store is cloned value doesnt reset
   */
    storeChangeDetectors() {
        for (const rowId in this.addedStores) {
            this.addedStores[rowId]?.controls.store.valueChanges.subscribe(
                (changes) => {
                    this.showLoader = true;
                    if (changes !== '' && changes !== undefined) {
                        const storeId = changes.split(' - ');
                        this.restaurantsApi.getStoreHeirarchy(storeId[1]).then((response) => {
                            const parentNodeId = response.data.attributes.parentNodeId;
                            this.getHierarchy(rowId, parentNodeId);
                        }).catch((error) => {
                            this.hierarchyAvailable = false;
                            this.showLoader = false;
                        });
                    }
                    if (this.addedStores[rowId].value.store !== changes && !this.clone[rowId]) {
                        this.addedStores[rowId].patchValue({
                            properties: [
                                this.fb.group({
                                    vendor: "",
                                    dName: "",
                                    dLane: "",
                                    dAngle: "",
                                    modelNo: "",
                                    sNo: "",
                                    unitNo: "",
                                    warranty: "",
                                    install: "",
                                    hardware: "",
                                    firmware: "",
                                    ip: "",
                                    mac: "",
                                    lane: "",
                                    isIpAvailable: false,
                                    isMacAvailable: false,
                                    isLaneAvailable: false,
                                    isHardwareVersionAvailable: false
                                })
                            ]
                        });
                        this.updateDeviceDataToService();
                        this.showLoader = false;
                    }
                    this.showLoader = false;
                }
            );
        }
    }

    /**
   * Add parent row
   * 
   * this function is called when user adds a new store
   */

    async insertNewDevice() {
        const rowId = this.nextStoreRowIdIndex++;
        this.addedStores = {
            ...this.addedStores,
            [rowId]: this.fb.group({
                store: "",
                properties: {}
            })
        };
        this.addProperty(rowId);
        this.storeChangeDetectors();
        this.onTabOpen(rowId);
    }

    /**
   * Add child row
   * @param deviceRowId: row index
   * @param prop: default data in any
   * this function is called when user adds a new device
   */
    addProperty(deviceRowId, prop?) {

        const propertyRowId = this.nextPropertyRowIdIndex[deviceRowId] || 0;

        this.vendorBtndisabled = true;
        this.nextPropertyRowIdIndex[deviceRowId] = propertyRowId + 1;
        this.activeIndex = this.nextPropertyRowIdIndex[deviceRowId];

        this.addedStores[deviceRowId].patchValue({
            properties: {
                ...this.addedStores[deviceRowId].value.properties,
                [propertyRowId]: this.fb.group({
                    vendor: prop?.vendor || "",
                    dName: prop?.dName || "",
                    dLane: prop?.dLane || "",
                    dAngle: prop?.dAngle || "",
                    modelNo: prop?.modelNo || "",
                    sNo: prop?.sNo || "",
                    unitNo: prop?.unitNo || "",
                    warranty: prop?.warranty || "",
                    install: prop?.install || "",
                    hardware: prop?.hardware || "",
                    firmware: prop?.firmware || "",
                    ip: prop?.ip || "",
                    mac: prop?.mac || "",
                    lane: prop?.lane || "",
                    isIpAvailable: false,
                    isMacAvailable: false,
                    isLaneAvailable: false,
                    isHardwareVersionAvailable: false
                })
            }
        });
        this.updateDeviceDataToService();
    }

    /**
   * Add cloned properties
   * @param rowId: row index
   * @param rowIdNext: index of the newly generated cloned row
   * @param propRowId: index of the child row
   * this function is called when user clicks on clone a store button
   * it will add all the default properties like vendor, dName to the cloned store
   */
    addClonedProperties(rowId, rowIdNext, propRowId) {
        this.addedStores[rowIdNext].patchValue({
            properties: {
                ...this.addedStores[rowIdNext].value.properties,
                [propRowId]: this.fb.group({
                    vendor: this.addedStores[rowId].value.properties[propRowId].value.vendor || "",
                    dName: this.addedStores[rowId].value.properties[propRowId].value.dName || "",
                    dLane: "",
                    dAngle: "",
                    modelNo: this.addedStores[rowId].value.properties[propRowId].value.modelNo || "",
                    sNo: "",
                    unitNo: "",
                    warranty: "",
                    install: this.addedStores[rowId].value.properties[propRowId].value.install || "",
                    hardware: this.addedStores[rowId].value.properties[propRowId].value.hardware || "",
                    firmware: this.addedStores[rowId].value.properties[propRowId].value.firmware || "",
                    ip: "",
                    mac: "",
                    lane: "",
                    isIpAvailable: this.addedStores[rowId].value.properties[propRowId].value.isIpAvailable || false,
                    isMacAvailable: this.addedStores[rowId].value.properties[propRowId].value.isMacAvailable || false,
                    isLaneAvailable: this.addedStores[rowId].value.properties[propRowId].value.isLaneAvailable || false,
                    isHardwareVersionAvailable: this.addedStores[rowId].value.properties[propRowId].value.isHardwareVersionAvailable || false
                })
            }
        });
        this.updateDeviceDataToService();
    }
    /**
* Clone store
* @param rowId: row index
* this function is called when user clicks on clone a store button
* it will add a new store to the next index
*/
    cloneStore(rowId) {
        const rowIdNext = this.nextStoreRowIdIndex++;
        this.addedStores = {
            ...this.addedStores,
            [rowIdNext]: this.fb.group({
                store: "",
                properties: {}
            })
        };
        this.storeChangeDetectors();
        this.onTabOpen();
        this.clone[rowIdNext] = true;
        const propertyRowId = this.nextPropertyRowIdIndex[rowId];
        this.showError[rowIdNext] = { ip: [], mac: [], unitNo: [] };
        if (propertyRowId >= 1) {
            for (let propId in this.addedStores[rowId].value.properties) {
                this.addClonedProperties(rowId, rowIdNext, propId);
            }
        }
    }

    /**
   * Remove child row
   * @param configRowId: row index
   * @param propertyRowId: child row index
   * this function is called when user removes the device
   */
    removeProperty(configRowId, propertyRowId) {
        let properties = this.addedStores[configRowId].value.properties;
        delete properties[propertyRowId];
        this.addedStores[configRowId].patchValue({
            properties
        });
        this.updateDeviceDataToService();
    }

    /**
 * Remove parent row
 * @param rowId: row index
 * this function is called when user clicks on bin icon to remove the store
 */
    removeConfigRow(rowId) {
        this.activeTab.splice(rowId, 1);
        this.nextStoreRowIdIndex--;
        delete this.addedStores[rowId];
        this.storeChangeDetectors();
        this.updateDeviceDataToService();
    }

    /**
 * Update the data to the service
 * 
 * this function is called when user changes anything on the form,
 * it will store the changes to the service
 */
    updateDeviceDataToService() {
        const data = [];
        for (let rowId in this.addedStores) {
            const {
                store,
                properties: propertiesForms
            } = this.addedStores[rowId].value;

            const properties = [];
            for (let propertyRowId in propertiesForms) {
                let selectedDevice: string;
                const selectedDeviceFlag = this.deviceFlagList.filter(x => x.value == propertiesForms[propertyRowId].value.dName);
                if (selectedDeviceFlag.length > 0) {
                    selectedDevice = selectedDeviceFlag[0]?.deviceFlag ? 'iot' : 'non-iot';
                }
                
                properties.push({
                    vendor: propertiesForms[propertyRowId].value.vendor,
                    dName: propertiesForms[propertyRowId].value.dName,
                    dLane: propertiesForms[propertyRowId].value.dLane,
                    dAngle: propertiesForms[propertyRowId].value.dAngle,
                    modelNo: propertiesForms[propertyRowId].value.modelNo,
                    sNo: propertiesForms[propertyRowId].value.sNo,
                    unitNo: propertiesForms[propertyRowId].value.unitNo,
                    warranty: propertiesForms[propertyRowId].value.warranty,
                    install: propertiesForms[propertyRowId].value.install,
                    hardware: propertiesForms[propertyRowId].value.hardware,
                    firmware: propertiesForms[propertyRowId].value.firmware,
                    ip: propertiesForms[propertyRowId].value.ip,
                    lane: propertiesForms[propertyRowId].value.lane,
                    mac: propertiesForms[propertyRowId].value.mac,
                    isIpAvailable: propertiesForms[propertyRowId].value.isIpAvailable,
                    isMacAvailable: propertiesForms[propertyRowId].value.isMacAvailable,
                    isLaneAvailable: propertiesForms[propertyRowId].value.isLaneAvailable,
                    isHardwareVersionAvailable: propertiesForms[propertyRowId].value.isHardwareVersionAvailable,
                    deviceFlag: selectedDevice,
                    abbreviation: this.abbreviationOfSelectedDevice
                });
            }
            data.push({
                store,
                properties
            });
        }
        this.checkFormFilled(data);
        this.selectedDeviceService.setAddStoreObject(data);
    }

    /**
     * 
     * @param response : latest data that user has entered in the form
     * this function checks if for a particular store any of the mandatoey field is empty
     * If empty then returns true
     */

    checkFormFilled(response) {
        if(response !== null && response != undefined && response.length > 0) {
            response.forEach((e, index) => {
                if (!e.store || !(e.properties?.length)) {
                    this.disableAddDevice[index] = true;
                }
                for (let ele of e.properties) {
                        if (
                            !ele.vendor || 
                            !ele.dName || 
                            !ele.dLane ||
                            !ele.dAngle ||
                            !ele.modelNo ||
                            !ele.install ||
                            !ele.warranty ||
                            (ele.isIpAvailable && !ele.ip) ||
                            (ele.isMacAvailable && !ele.mac) ||
                            (ele.isLaneAvailable && !ele.lane) ||
                            (ele.isHardwareVersionAvailable && !ele.hardware) ||
                            !ele.sNo ||
                            !ele.unitNo ||
                            !ele.firmware
                        ) {
                            this.disableAddDevice[index] = true;
                            break;
                        } else {
                            this.disableAddDevice[index] = false;
                        }
                    }
            });
        }
    }

    /**
     * 
     * @param prop : device property name coming from api
     * @param rowId : parent row index
     * @param propertyId : child row index
     * this function is mapping the properties fetched to availability of ip address, mac address and lane fields
     */

    fetchDeviceProp(prop: string, rowId, propertyId) {
        const propType = prop.replace(/\s/g, "").toLowerCase();
        switch (propType) {
            case "ipaddress":
                this.addedStores[rowId].value.properties[propertyId].patchValue({
                    isIpAvailable: true
                });
                break;
            case "macaddress":
                this.addedStores[rowId].value.properties[propertyId].patchValue({
                    isMacAvailable: true
                });
                break;
            case "lanenumber":
                this.addedStores[rowId].value.properties[propertyId].patchValue({
                    isLaneAvailable: true
                });
                break;
            case "hardwareversion":
                this.addedStores[rowId].value.properties[propertyId].patchValue({
                    isHardwareVersionAvailable: true
                });
                break;
        }
    }

    /**
   * Vendor Name update
   * @param e - event
   * @param rowId: id of the parent row
   * @param propertyRowId: id of the child row
   * this function is called when user changes the vendor name
   */

    onPropertyVendorChange(e, rowId, propertyRowId) {
        this.vendorLoading = true;
        this.addedStores[rowId].value.properties[propertyRowId].patchValue({
            vendor: e.value,
            modelNo: '',
            firmware: ''
        });
        if (this.clone[rowId]) {
            this.addedStores[rowId].value.properties[propertyRowId].patchValue({
                dName: "",
                dLane: "",
                dAngle: "",
                modelNo: "",
                sNo: "",
                unitNo: "",
                warranty: "",
                install: "",
                hardware: "",
                firmware: "",
                ip: "",
                mac: "",
                lane: "",
                isIpAvailable: false,
                isMacAvailable: false,
                isLaneAvailable: false,
                isHardwareVersionAvailable: false
            });
        }
        if (e.value !== '' && e.value !== undefined) {
            this.fetchDisplayName(e.value);
        }
        this.updateDeviceDataToService();
        this.activeIndex = propertyRowId;
        this.showLoader = false;
        this.fetchVendorData(e.value);
    }

    /**
   * Display Name update
   * @param e - event
   * @param rowId: id of the parent row
   * @param propertyRowId: id of the child row
   * this function is called when user changes the display name
   */
    onPropertydNameChange(e, rowId, propertyRowId) {
        this.abbreviationOfSelectedDevice = this.deviceFlagList.filter(device => {return device.value === e.value})[0]?.abbreviation;
        let deviceId: string;
        this.addedStores[rowId].value.properties[propertyRowId].patchValue({
            isIpAvailable: false,
            isMacAvailable: false,
            isLaneAvailable: false,
            isHardwareVersionAvailable: false,
            dName: e.value
        });
        if (e.value !== '' && e.value !== undefined) {
            this.fetchModelNo(this.addedStores[rowId].value.properties[propertyRowId].value.vendor, e.value);
            this.deviceFlagList.filter(x => x.value === e.value).map((key, value) => deviceId = key.id)
            this.restaurantsApi.getDeviceProps(deviceId).then((response) => {
                response.forEach((deviceProp) => {
                    const propsAvailable = deviceProp.attributes.propertyName;
                    this.fetchDeviceProp(propsAvailable, rowId, propertyRowId);
                })
            })
        }
        this.updateDeviceDataToService();
    }

    /**
  * Warranty date update
  * @param e - event
  * @param rowId: id of the parent row
  * @param propertyRowId: id of the child row
  * this function is called when user changes the warranty date
  */
    onWarrantyChange(e, rowId, propertyRowId) {
        this.addedStores[rowId].value.properties[propertyRowId].patchValue({
            warranty: e
        });
        this.updateDeviceDataToService();
    }

    /**
  * Install Date update
  * @param e - event
  * @param rowId: id of the parent row
  * @param propertyRowId: id of the child row
  * this function is called when user changes the install date
  */
    onInstallChange(e, rowId, propertyRowId) {
        this.addedStores[rowId].value.properties[propertyRowId].patchValue({
            install: e
        });
        this.updateDeviceDataToService();
    }

    /**
   * Model Number update
   * @param e - event
   * @param rowId: id of the parent row
   * @param propertyRowId: id of the child row
   * this function is called when user changes the model number
   */
    onPropertyModelNoChange(e, rowId, propertyRowId) {
        this.addedStores[rowId].value.properties[propertyRowId].patchValue({
            modelNo: e.value
        });
        if (e.value !== '' && e.value !== undefined) {
            this.fetchFirmware(
                this.addedStores[rowId].value.properties[propertyRowId].value.vendor,
                this.addedStores[rowId].value.properties[propertyRowId].value.dName, e.value
            );
        }
        this.updateDeviceDataToService();
    }

    /**
  * Hardware update
  * @param e - event
  * @param rowId: id of the parent row
  * @param propertyRowId: id of the child row
  * this function is called when user changes the hardware version
  */
    onPropertyHardwareChange(e, rowId, propertyRowId) {
        this.addedStores[rowId].value.properties[propertyRowId].patchValue({
            hardware: e.value
        });
        this.updateDeviceDataToService();
    }

    /**
  * Firmware update
  * @param e - event
  * @param rowId: id of the parent row
  * @param propertyRowId: id of the child row
  * this function is called when user changes the firmware version
  */
    onPropertyFirmwareChange(e, rowId, propertyRowId) {
        this.addedStores[rowId].value.properties[propertyRowId].patchValue({
            firmware: e.value
        });
        this.updateDeviceDataToService();
    }

    /**
     * @param event : fetch the entered value
     * @param rowId : parent row id
     * @param propId : child row id
     * this function checks if the ip address entered is matching the pattern
     * it stores the result in showError
     */
    checkIpPattern(event, rowId, propId) {
        if (event.value !== '' && event.value !== undefined) {
            if (this.inputPatterns.IP_ADDRESS.test(event.value)) {
                this.showError[rowId].ip[propId] = false;
                this.updateDeviceDataToService();
            } else {
                this.showError[rowId].ip[propId] = true;
            }
            this.checkForErrors();
        }
    }

    /**
     * @param event : fetch the entered value
     * @param rowId : parent row id
     * @param propId : child row id
     * this function checks if the mac address entered is matching the pattern
     * it stores the result in showError
     */
    checkMacPattern(event, rowId, propId) {
        if (event.value !== '' && event.value !== undefined) {
            if (this.inputPatterns.MAC_ADDRESS.test(event.value)) {
                this.showError[rowId].mac[propId] = false;
                this.updateDeviceDataToService();
            } else {
                this.showError[rowId].mac[propId] = true;
            }
            this.checkForErrors();
        }
    }

    /**
  * Close tab
  * @param event - tab index
  * this function is called when user closes the tab
  */
    onTabClose(event) {
        let ids = "header-display" + event;
        var x = this.document.getElementById(ids);
        if (x !== null) x.style.display = "block";
    }

    /**
   * Open tab
   * @param event - tab index
   * this function is called when user opens the tab or close another tab
   */
    onTabOpen(event?) {
        let headerElem = this.document.querySelectorAll(".add-device__accordion-tab_header");
        headerElem.forEach((headElem: HTMLElement) => {
            headElem.style.display = "block";
        });
        let contentElem = document.querySelectorAll(".maindiv");
        contentElem.forEach((content, index, array) => {
            this.activeTab[index] = false;
            if (!event && index === array.length - 1) {
                this.activeTab[index + 1] = true;
            }
        });
        if (event !== null || event !== undefined) {
            this.activeTab[event] = true;
            let ids = "header-display" + event;
            let x = this.document.getElementById(ids);
            if (x !== null) {
                x.style.display = "none";
            }
        }
    }

    /**
    * Fetch vendor details
    * @param vendorId - vendor name
    * Fetch vendor details from the vendors data.
    */
    getVendorDet(vendorId) {
        this.vendorDialogData = this.vendors.filter(vendor => {
            return vendor.id === vendorId.split(' -')[0];
        });
    }

    fetchVendorData(propertyRowId){
        this.getVendorDet(propertyRowId);
        this.loadVendor(this.vendorDialogData[0].id).then(()=>{
            let vendorData = {
                vendor: this.vendorInfo.vendorData.data.attributes,
                contacts: this.vendorInfo.contacts?.filter(item => {
                    return item.level === 'primary' || item.level === 'Primary'
                  })
            }
            vendorData.vendor.id = propertyRowId;
            this.vendorInfo = vendorData;
            this.vendorLoading = false;
          })
    }

    /**
   * vendor details
   * @param propertyRowId - Current row
   * this function is called when ever user clicks on vendor details button.
   * Fetch vendor details based on the vendor selected for that particular device row.
   */
    openVendorDetailsDialog(propertyRowId) {
        this.dialog.open(
            OnboardingVendorDetailsComponent, this.vendorInfo);
        
    }

    async loadVendor(vendorId: number) {
        await this.api.getVendorComponent(vendorId).then(res => {
          this.vendorInfo = res;
        }); 
      }

}