import jwt
import yaml
import json

def read_yaml(authorization_token, yaml_file):
    """Function to decode jwt token and read yaml file

        Args:
            authorization_token: Bearer Token

        Returns:
            List of user roles
    """
    decoded_token = jwt.decode(authorization_token, options = {"verify_signature": False})
    keywords = set()
    if "group" in decoded_token:
        user_roles = decoded_token["group"]
        with open(yaml_file, "r") as stream:
            data = yaml.safe_load(stream)
            yaml_obj = data["roles-fe-mapping"]
            keywords = group_user_role(user_roles, yaml_obj)
    else:
        keywords = ["bre_menu"]
    return list(keywords)


def group_user_role(user_roles, yaml_obj): 
    keywords = set()
    keywords.add("bre_resturant")
    if isinstance(user_roles, str):
        # When user has only one associated group
        keywords = fetch_user_roles(keywords, user_roles, yaml_obj)
    elif len(user_roles):
        # When user has multiple associated groups
        for user_role in user_roles:
            if user_role in yaml_obj:
                for yaml_role in yaml_obj[user_role]:
                    keywords.add(yaml_role)
    else:
        keywords = ["bre_menu"]
    return keywords


def fetch_user_roles(keywords, user_role, yaml_obj):
    if user_role in yaml_obj:
        for yaml_role in yaml_obj[user_role]:
            keywords.add(yaml_role)
    return keywords


def handler(event, context):
    """ Lambda Handler

        Args:
            event
            context

        Returns:
            response
    """
    try:
        authorization_token = event['headers']['Authorization'].split(" ")[1]
        status_code = 200
        error = None
        yaml_response = None
        yaml_file = "resources/rbac_roles_mapping.yaml"
        yaml_response = {
            "uiModules": read_yaml(authorization_token, yaml_file)
        }
    except yaml.YAMLError:
        status_code = 500
        yaml_response = None
        error = "Error while parsing the data",

    except Exception:
        status_code = 400
        yaml_response = None
        error = "Something went wrong.",

    response_body = {
            "status": "success" if status_code == 200 else "failure",
            "error": error,
            "data": yaml_response
        }
    # Capture the data returned from read_yaml method. Based on this data set the value for "statusCode" in the response.
    response = {
        "statusCode": status_code,
        "headers": {
            "Access-Control-Allow-Origin" : "*", # Required for CORS support to work
            "Access-Control-Allow-Credentials" : True # Required for cookies, authorization headers with HTTPS 
            },
        "body": json.dumps(response_body)
    }

    return response
