import { Injectable } from "@angular/core";
import { ConfigureEnvService } from "@bref/configure";
import { ApiService, RBACService } from "@bref/core";

declare global {
  interface Window {
    appConfig: any;
  }
}

@Injectable({
  providedIn: "root"
})
export class AuthorizationService {
  private idToken: string;
  private accessToken: string;
  private clientId: string;
  private envName: string;
  private rbacUrl: string;
  hasFetchedAuthorizedComponents: boolean = false;
  authorizedComponents: any;

  constructor(private api: ApiService, private rbac: RBACService) {
    this.clientId = window.appConfig?.auth?.clientId;
    this.envName = window.appConfig?.env;
    this.rbacUrl = window.appConfig?.rbacUrl;
    this.setIDToken();
    this.setAccessToken();
  }

  /**
   * Set value for ID Token
   * @param token
   */
  setIDToken(token: string = "") {
    let idToken = token;
    if (!idToken) {
      idToken =
        localStorage.getItem(`msal.${this.clientId}.idtoken`) ||
        localStorage.getItem(`msal.idtoken`);
    }
    this.idToken = idToken;
  }

  /**
   * Get ID Token
   * @returns ID Token string
   */
  getIdToken() {
    return this.idToken;
  }

  /**
   * Set value for Access Token
   * @param token
   */
  setAccessToken(token: string = "") {
    let accessToken = token;
    if (!accessToken) {
      accessToken = localStorage.getItem(`access_token`);
    }
    this.accessToken = accessToken;
  }

  /**
   * Get value for Access Token
   * @returns Access Token string
   */
  getAccessToken() {
    return this.accessToken;
  }

  getEnvName(): string {
    return this.envName;
  }

  /**
   * Fetches authorized components
   * @method fetchAuthorizedComponents
   * @param accessToken - ADFS access token
   */
  async fetchAuthorizedComponents(accessToken: string) {
    try {
      const res: any = await this.api.get<object>(
        `${this.rbacUrl}`,
        null,
        null,
        {
          Authorization: `Bearer ${accessToken}`
        }
      );
      if (res) {
        this.authorizedComponents = res.data?.uiModules;
      }
      this.rbac.setAuthorizedComponents(this.authorizedComponents);
      ConfigureEnvService.setAccessModules(this.authorizedComponents);
      this.hasFetchedAuthorizedComponents = true;
    } catch (e) {
      this.rbac.setAuthorizedComponents([]);
    }
  }
}