import { Component, OnInit } from "@angular/core";
import { MsalService, BroadcastService } from "@azure/msal-angular";
import { RBACService } from "@bref/core";
import { ROLE_BASED_COMPONENTS } from "./constants/role-based-components";
import { AppEnvService } from "./services/app-env.service";
import { AuthorizationService } from "./services/authorization.service";

@Component({
  selector: "bre-root",
  templateUrl: "./app.component.html",
  styleUrls: ["./app.component.scss"]
})
export class AppComponent implements OnInit {
  isLoggedIn = false;

  constructor(
    private broadcastService: BroadcastService,
    private authService: MsalService,
    private appEnvService: AppEnvService,
    private authorizationService: AuthorizationService,
    private rbac: RBACService
  ) {}

  async ngOnInit() {
    // Provide role based components to RBAC service
    this.rbac.setRoleBasedComponents(ROLE_BASED_COMPONENTS);

    this.checkAccount();
    if (!this.isLoggedIn) {
      this.login();
    } else {
      const accessToken = this.authorizationService.getAccessToken();
      if (accessToken && !this.authorizationService?.hasFetchedAuthorizedComponents) {
        // Get authorized components for RBAC
        await this.authorizationService.fetchAuthorizedComponents(accessToken);
      }
    }

    this.broadcastService.subscribe("msal:acquireTokenSuccess", (data) => {
      this.acquireAccessToken(data.accessToken);
      this.authorizationService.setIDToken(data.idToken.rawIdToken);
    });

    this.authService.handleRedirectCallback((err, data) => {
      if (!err) {
        this.checkAccount();
        this.acquireAccessToken(data.accessToken);
        this.authorizationService.setIDToken();
      }
    });
    this.broadcastService.subscribe("msal:loginSuccess", (data) => {
      this.checkAccount();
      this.acquireAccessToken(data.accessToken);
      this.authorizationService.setIDToken();
    });

    this.authService.handleRedirectCallback((err, data) => {
      if (!err) {
        this.checkAccount();
        this.acquireAccessToken(data.accessToken);
        this.authorizationService.setIDToken();
      }
    });
  }

  /**
   * Checks if user account details exist already
   */
  checkAccount() {
    this.isLoggedIn = !!this.authService.getAccount();
  }

  /**
   * Redirects user to SSO Login
   */
  login() {
    this.authService.loginRedirect({
      extraScopesToConsent: ["user.read", "openid", "profile"]
    });
  }

  /**
   * Checks if user has logged in with both tokens
   * @returns boolean
   */
  hasLoggedIn() {
    return (
      this.authService.getLoginInProgress() ||
      !(
        this.isLoggedIn &&
        this.authorizationService.getAccessToken() &&
        this.authorizationService.hasFetchedAuthorizedComponents
      )
    );
  }

  /**
   * Method to fetch/set the access token
   * @param accessToken
   */
  async acquireAccessToken(accessToken) {
    if (!accessToken) {
      // Acquire access_token
      this.authService.acquireTokenRedirect({
        scopes: ["openid", "User.Read", "profile"],
        sid: this.authService.getAccount()?.sid
      });
    } else {
      if (!this.authorizationService.getAccessToken()) {
        localStorage.setItem("access_token", accessToken);
        this.authorizationService.setAccessToken(accessToken);
        this.appEnvService.setTokenInModules();
        // Get authorized components for RBAC
        await this.authorizationService.fetchAuthorizedComponents(accessToken);
      }
    }
  }
}