import { TestBed, async, fakeAsync, inject, tick } from "@angular/core/testing";
import { AppComponent } from "./app.component";
import { RouterTestingModule } from "@angular/router/testing";
import { NavigationMenuComponent } from "./navigation-menu/navigation-menu.component";
import { HeaderComponent } from "./header/header.component";
import { StandaloneDeployComponentsModule } from "@bref/deploy";
import { StandaloneSolutionsComponentsModule } from "@bref/solutions";
import { StandaloneMenuComponentsModule } from "@bref/menu";
import { StandaloneConfigureComponentsModule } from "@bref/configure";
import { BroadcastService, MsalService } from "@azure/msal-angular";
import { Account } from "msal";
import { LogoutNavButtonComponent } from "./components/logout-nav-button/logout-nav-button.component";
import { ApiService, CoreModule, RBACService, ToastService } from "@bref/core";
import { AppEnvService } from "./services/app-env.service";
import { StandaloneStoresComponentsModule } from "@bref/stores";
import { AuthorizationService } from "./services/authorization.service";

let isLoggedIn = false;

const mockGetAccount = jest.fn(() => {
  if (isLoggedIn) {
    return {
      accountIdentifier: "",
      userName: "",
      idToken: null,
      idTokenClaims: null,
      homeAccountIdentifier: "",
      name: "",
      sid: "",
      environment: ""
    };
  }
  return undefined;
});

const mockLoginRedirect = jest.fn(() => {
  isLoggedIn = true;
});

const mockAcquireTokenRedirect = jest.fn();
const mockFetchAuthorizedComponents = jest.fn();

class MsalServiceMock {
  getAccount(): Account {
    return mockGetAccount();
  }
  loginRedirect(): void {
    mockLoginRedirect();
  }
  acquireTokenRedirect() {
    mockAcquireTokenRedirect();
  }
  getLoginInProgress() {}
  handleRedirectCallback(): void {}
}

const rbacService = new RBACService();

fdescribe("AppComponent", () => {
  let mockAuthorizationService: AuthorizationService;

  AppEnvService.env = { auth: {} };
  let mockAppEnvService: AppEnvService;

  beforeEach(async(() => {
    window.appConfig = {
      rbacUrl: "rbacUrl"
    };

    TestBed.configureTestingModule({
      providers: [
        BroadcastService,
        ApiService,
        ToastService,
        { provide: RBACService, useValue: rbacService },
        { provide: MsalService, useValue: new MsalServiceMock() }
      ],
      imports: [
        RouterTestingModule,
        StandaloneDeployComponentsModule,
        StandaloneSolutionsComponentsModule,
        StandaloneMenuComponentsModule,
        StandaloneConfigureComponentsModule,
        StandaloneStoresComponentsModule,
        CoreModule
      ],
      declarations: [
        AppComponent,
        NavigationMenuComponent,
        HeaderComponent,
        LogoutNavButtonComponent
      ]
    }).compileComponents();

    mockAuthorizationService = TestBed.inject(AuthorizationService);
    spyOn(mockAuthorizationService, "fetchAuthorizedComponents").and.callFake(
      () => {
        mockFetchAuthorizedComponents();
        rbacService.setAuthorizedComponents([
          "bre_menu",
          "bre_deploy",
          "bre_solutions",
          "bre_config"
        ]);
        mockAuthorizationService.hasFetchedAuthorizedComponents = true;
      }
    );

    mockAppEnvService = TestBed.inject(AppEnvService);

    spyOn(mockAppEnvService, "setTokenInModules").and.callThrough();

    spyOn(mockAppEnvService, "setEnv").and.callThrough();
  }));

  it("should create the app", () => {
    const fixture = TestBed.createComponent(AppComponent);
    const app = fixture.componentInstance;
    expect(app).toBeTruthy();
  });

  it("should render header component", () => {
    const fixture = TestBed.createComponent(AppComponent);
    const compiled = fixture.nativeElement;
    const header = compiled.querySelector("bre-header");
    expect(header.textContent).toBeTruthy();
  });

  it("should show loader", () => {
    const fixture = TestBed.createComponent(AppComponent);
    const compiled = fixture.nativeElement;
    const loader = compiled.querySelector("bre-loader");
    expect(loader).toBeDefined();
  });

  it("should redirect to ADFS", fakeAsync(() => {
    const fixture = TestBed.createComponent(AppComponent);
    fixture.componentInstance.ngOnInit();
    expect(mockLoginRedirect).toHaveBeenCalled();
  }));

  it("should redirect to acquire token", fakeAsync(() => {
    const fixture = TestBed.createComponent(AppComponent);
    fixture.componentInstance.acquireAccessToken("");
    expect(mockAcquireTokenRedirect).toHaveBeenCalled();
  }));

  it("should redirect to acquire token", fakeAsync(() => {
    const fixture = TestBed.createComponent(AppComponent);
    fixture.componentInstance.acquireAccessToken("");
    expect(mockAcquireTokenRedirect).toHaveBeenCalled();
  }));

  it("should set acquired token and init RBAC", fakeAsync(() => {
    const fixture = TestBed.createComponent(AppComponent);
    fixture.componentInstance.acquireAccessToken("abc");
    expect(mockFetchAuthorizedComponents).toHaveBeenCalled();
  }));

  it("should check login status", fakeAsync(() => {
    const fixture = TestBed.createComponent(AppComponent);
    const status = fixture.componentInstance.hasLoggedIn();
    expect(status).toBeTruthy();
  }));
});