import { and, or, operator, attribute } from './json-api-helpers';

describe('or', () => {
  const filters = {
    componentName: ['potato', 'fries', 'burger'],
    componentId: ['1', '34', '5', '10'],
  }
  test('should return an object with key \'or\' with filters', () => {
    const output = {
      OR: [
        { id: ['1', '34', '5', '10'] },
        { name: ['potato', 'fries', 'burger'] }
      ]
    }
    expect(or({ id: filters.componentId }, { name: filters.componentName })).toMatchObject(output)
  })

  test('should return null if empty filters', () => {
    const output = null;
    expect(or()).toEqual(output);
  })

  test('should return filter object if there is only one filter', () => {
    const output = {
      id: ['1', '34', '5', '10'],
    }
    expect(or({ id: filters.componentId })).toMatchObject(output)
  })

  test('should return object with key or and multiple key like', () => {
    const output = {
      OR: [
        { LIKE: { id: ['1', '34', '5', '10'] } },
        { LIKE: { name: ['%potato%', '%fries%', '%burger%'] } }
      ]
    };
    expect(
      or(
        operator('LIKE', false, { id: filters.componentId }),
        operator('LIKE', true, { name: filters.componentName })
      )
    ).toMatchObject(output);
  })
})

describe('operator', () => {
  test('should return an object with key \'like\' and id filter', () => {
    const filters = {
      componentName: ['potato', 'fries', 'burger'],
      componentId: ['1', '34', '5', '10']
    };
    const output = {
      LIKE: {
        id: ['1', '34', '5', '10'],
      }
    };
    expect(operator('LIKE', false, { id: filters.componentId })).toMatchObject(output);
  })

  test('should return an object with key \'id\' and name filter', () => {
    const filters = {
      componentName: ['potato', 'fries', 'burger'],
      componentId: ['1', '34', '5', '10'],
    };
    const output = {
      LIKE: {
        name: ['%potato%', '%fries%', '%burger%']
      }
    };
    expect(operator('LIKE', true, { name: filters.componentName })).toMatchObject(output);
  })

})

describe('and', () => {
  test('should return an object with no key with filters', () => {
    const filters = {
      componentType: ['OVEN_DEVICE'],
      storeName: ['Spring', 'Summer', 'Winter'],
      storeId: ['5', '12', '55'],
    };
    const output = {
      AND: [
        { type: ['OVEN_DEVICE'] },
        { name: ['Spring', 'Summer', 'Winter'] },
        { id: ['5', '12', '55'] }
      ]
    };
    expect(
      and(
        { type: filters.componentType },
        { name: filters.storeName },
        { id: filters.storeId }
      )
    ).toMatchObject(output)
  })
})

describe('and(or())', () => {
  const filters = {
    componentType: ['OVEN_DEVICE'],
    storeName: ['Spring', 'Summer', 'Winter'],
    storeId: ['5', '12', '55']
  }

  test('should return an object with key value pairs and key \'or\' with filters', () => {
    const output = {
      AND: [
        { type: ['OVEN_DEVICE'] },
        {
          OR: [
            { name: ['Spring', 'Summer', 'Winter'] },
            { id: ['5', '12', '55'] }
          ]
        }
      ]
    };
    expect(
      and(
        { type: filters.componentType },
        or(
          { name: filters.storeName },
          { id: filters.storeId })
      )
    ).toMatchObject(output);
  })

  test('should accept or with no paramenters and return correct object', () => {
    const output = {
      type: ['OVEN_DEVICE'],
    }
    expect(
      and(
        { type: filters.componentType },
        or()
      )
    ).toMatchObject(output)
  })
})

describe('and(or(),or())', () => {
  const filters = {
    id: ['foo'],
    componentId: ['1', '3'],
    componentType: ['OVEN_DEVICE'],
    storeName: ['Spring', 'Summer', 'Winter'],
    storeId: ['5', '12', '55']
  }

  test('should return an object with key value pairs and key \'or\' with filters', () => {
    const output = {
      AND: [
        { id: ['foo'] },
        {
          OR: [
            { storeName: ['Spring', 'Summer', 'Winter'] },
            { storeId: ['5', '12', '55'] }
          ]
        },
        {
          OR: [
            { componentType: ['OVEN_DEVICE'] },
            { componentId: ['1', '3'] }
          ]
        }
      ]
    };
    expect(
      and(
        { id: filters.id },
        or(
          { storeName: filters.storeName },
          { storeId: filters.storeId }
        ),
        or(
          { componentType: filters.componentType },
          { componentId: filters.componentId }
        )
      )
    ).toMatchObject(output);
  })
})

describe('and(or(operator()))', () => {
  test('should return an object with key value pairs and key \'or\' and key \'like\' with filters', () => {
    const filters = {
      componentType: ['OVEN_DEVICE'],
      storeName: ['Spring', 'Summer', 'Winter'],
      storeId: ['5', '12', '55'],
    };
    const output = {
      AND: [
        { type: ['OVEN_DEVICE'] },
        {
          OR: [
            { id: ['5', '12', '55'] },
            { LIKE: { name: ['%Spring%', '%Summer%', '%Winter%'] } }
          ]
        }
      ]
    };
    expect(
      and(
        { type: filters.componentType },
        or(
          { id: filters.storeId },
          operator('LIKE', true, { name: filters.storeName }))
      )
    ).toMatchObject(output);
  })
})

describe('attribute() with empty filters', () => {
  test('should return null', () => {
    const attributeTerm = 'restaurant';
    const output = null;
    expect(attribute(attributeTerm, or())).toEqual(output)
  })
})

describe('operator() with empty filters', () => {
  test('should return null', () => {
    const output = null;
    expect(operator('LIKE', false, or())).toEqual(output)
  })
})

describe('and(or(), attribute(or(operator()))) with empty attribute filters', () => {
  test('should return an object with key type and key id', () => {
    const filters = {
      componentId: ['1', '3'],
      componentType: ['OVEN_DEVICE'],
    };
    const attributeTerm = 'restaurant';
    const output = {
      AND: [
        { type: ['OVEN_DEVICE'] },
        { id: ['1', '3'] }
      ]
    };
    expect(
      and(
        { type: filters.componentType },
        { id: filters.componentId },
        or(operator('LIKE', false, undefined)),
        attribute(attributeTerm, or())
      )
    ).toMatchObject(output);
  })
})

describe('and(or(operator())) with empty filters', () => {
  test('should return an object with key value pairs of id and type with filters', () => {
    const filters = {
      componentType: ['OVEN_DEVICE'],
      storeId: ['5', '12', '55']
    };
    const output = {
      AND: [
        { type: ['OVEN_DEVICE'] },
        { id: ['5', '12', '55'] }
      ]
    };
    expect(
      and(
        { type: filters.componentType },
        or(
          { id: filters.storeId },
          operator('LIKE', true, undefined)
        )
      )
    ).toMatchObject(output);
  })
})

describe('and(or(), attribute(or(operator()))) with some empty filters', () => {
  test('should return an object with key type, id, restaurant', () => {
    const filters = {
      componentId: ['1', '3'],
      componentType: ['OVEN_DEVICE'],
      storeName: ['Spring', 'Summer', 'Winter'],
      storeId: ['5', '12', '55']
    };
    const attributeTerm = 'restaurant';
    const output = {
      AND: [
        { type: ['OVEN_DEVICE'] },
        { id: ['1', '3'] },
        {
          restaurant: {
            OR: [
              { id: ['5', '12', '55'] },
              { LIKE: { name: ['%Spring%', '%Summer%', '%Winter%'] } }
            ]
          }
        }
      ]
    };
    expect(
      and(
        { type: filters.componentType },
        { id: filters.componentId },
        or(operator('LIKE', true, undefined)),
        attribute(attributeTerm, or({ id: filters.storeId }, operator('LIKE', true, { name: filters.storeName })))
      )
    ).toMatchObject(output);
  })
})

describe('and(or(), attribute(or())) with empty filters', () => {
  test('should return an object with key value pairs and key \'or\' and key \'like\' with filters', () => {
    const filters = { componentType: 'OVEN_DEVICE' };
    const attributeTerm = 'restaurant';
    const operatorTerm = 'like';
    const output = { type: 'OVEN_DEVICE' };
    expect(
      and(
        { type: filters.componentType },
        or({ foo: undefined }, operator(operatorTerm, undefined, undefined)),
        attribute(attributeTerm, or({ bar: [] }, operator(operatorTerm, true, null)))
      )
    ).toMatchObject(output);
  })
})

describe('and(or(), attribute(or(operator()))', () => {
  test('should return an JSON object with correct key values', () => {
    const filters = { componentType: 'OVEN_DEVICE' };
    const attributeTerm = 'restaurant';
    const operatorTerm = 'like';
    const output = { type: 'OVEN_DEVICE' };
    expect(
      and(
        { type: filters.componentType },
        or(undefined, operator(operatorTerm, true, undefined)),
        attribute(attributeTerm, or(undefined, operator(operatorTerm, true, undefined)))
      )
    ).toMatchObject(output);
  })
})
