import { Injectable } from "@angular/core";
import { CoreModule } from "../../core.module";

import Toastify from "toastify-js";

@Injectable({
  providedIn: 'root'
})

export class ToastService {
  constructor() {}

  success(
    message: string,
    buttonText?: string,
    buttonCallback?: () => void,
    position?: string,
    bottomOffset?: string
  ) {
    this.showToast(
      message,
      "success",
      buttonText,
      buttonCallback,
      position,
      bottomOffset
    );
  }
  informational(
    message: string,
    buttonText?: string,
    buttonCallback?: () => void,
    position?: string,
    bottomOffset?: string
  ) {
    this.showToast(
      message,
      "info",
      buttonText,
      buttonCallback,
      position,
      bottomOffset
    );
  }
  banner(
    message: string,
    buttonText?: string,
    buttonCallback?: () => void,
    position?: string,
    bottomOffset?: string
  ) {
    this.showBanner(
      message,
      "banner",
      buttonText,
      buttonCallback,
      position,
      bottomOffset
    );
  }

  error(
    message: string,
    buttonText?: string,
    buttonCallback?: () => void,
    position?: string,
    bottomOffset?: string
  ) {
    this.showToast(
      message,
      "error",
      buttonText,
      buttonCallback,
      position,
      bottomOffset
    );
  }

  private isMultiLine = (message: string) => {
    return message.includes("\r\n");
  };
  private showToast(
    message: string,
    type: string,
    buttonText?: string,
    buttonCallback?: () => void,
    position: string = "center",
    bottomOffset?: string
  ) {
    let multiLine = "";
    if (this.isMultiLine(message)) {
      multiLine = "multi-line";
    }
    const toast = Toastify({
      duration: 8000,
      position,
      gravity: !!bottomOffset ? "bottom" : "",
      className: `${type + " " + multiLine}`,
      offset: {
        y: !!bottomOffset ? bottomOffset : ""
      },
      node: this.createCustomToast(message, buttonText, buttonCallback)
    });
    toast.showToast();
  }
  private showBanner(
    message: string,
    type: string,
    buttonText?: string,
    buttonCallback?: () => void,
    position: string = "center",
    bottomOffset?: string
  ) {
    let multiLine = "";
    if (this.isMultiLine(message)) {
      multiLine = "multi-line";
    }
    const toast = Toastify({
      duration: -1,
      position,
      gravity: "bottom",
      className: `${type + " " + multiLine}`,
      offset: {
        y: 0,
        x: 0
      },
      close: true,
      node: this.createCustomToast(message, buttonText, buttonCallback)
    });
    toast.showToast();
  }

  private createCustomToast(
    message: string,
    buttonText: string,
    buttonCallback: () => void
  ): Element {
    const customElement = document.createElement("div");
    customElement.textContent = message;
    if (this.isMultiLine(message)) {
      customElement.style.whiteSpace = "pre-line";
      customElement.style.marginTop = "5px";
      customElement.style.display = "flex";
      customElement.style.flexDirection = "column";
    }

    if (!!buttonText) {
      const buttonEl = document.createElement("button");
      buttonEl.textContent = buttonText;
      buttonEl.addEventListener("click", buttonCallback);
      customElement.appendChild(buttonEl);
      if (!this.isMultiLine(message)) {
        buttonEl.style.marginLeft = "32px";
      }
    }

    return customElement;
  }
}
