import { TestBed } from '@angular/core/testing';

import { ToastService } from './toast.service';

const mockShowToast = jest.fn();
jest.mock('toastify-js', () => ({
  default: jest.fn(options => ({
    showToast: () => mockShowToast(options)
  }))
}));
import Toastify from 'toastify-js';

describe('ToastService', () => {
  let service: ToastService;

  beforeEach(() => {
    TestBed.configureTestingModule({
      providers: [ToastService]
    });
    service = TestBed.inject(ToastService);
  });
  afterEach(() => {
    mockShowToast.mockClear();
  });

  it('should be created', () => {
    expect(service).toBeTruthy();
  });

  it('should properly display success toast', () => {
    service.success('it works');

    expect(Toastify).toHaveBeenCalledWith({
      position: 'center',
      duration: 8000,
      className: 'success ',
      gravity: '',
      offset: {
        "y": "",
      },
      node: expect.any(Element)
    });
    expect(mockShowToast).toHaveBeenCalled();
    expect(mockShowToast.mock.calls[0][0].node.innerHTML).toContain('it works');
  });

  it('should properly display success toast and correct classname', () => {
    service.success('it works\r\nsecond line');

    expect(Toastify).toHaveBeenCalledWith({
      position: 'center',
      duration: 8000,
      className: 'success multi-line',
      gravity: '',
      offset: {
        "y": "",
      },
      node: expect.any(Element)
    });
    expect(mockShowToast).toHaveBeenCalled();
    expect(mockShowToast.mock.calls[0][0].node.innerHTML).toContain('it works\r\nsecond line');
  });

  it('should properly display success toast and correct classname with correct inline styles', () => {
    service.success('it works\r\nsecond line');

    expect(Toastify).toHaveBeenCalledWith({
      position: 'center',
      duration: 8000,
      className: 'success multi-line',
      gravity: '',
      offset: {
        "y": "",
      },
      node: expect.any(Element)
    });
    expect(mockShowToast).toHaveBeenCalled();
    expect(mockShowToast.mock.calls[0][0].node.style[0]).toContain('white-space');
    expect(mockShowToast.mock.calls[0][0].node.style[1]).toContain('margin-top');
    expect(mockShowToast.mock.calls[0][0].node.style[2]).toContain('display');
    expect(mockShowToast.mock.calls[0][0].node.style[3]).toContain('flex-direction');
  });


  it('should properly display error toast', () => {
    service.error('it doesn\'t work');

    expect(Toastify).toHaveBeenCalledWith({
      position: 'center',
      duration: 8000,
      className: 'error ',
      gravity: '',
      offset: {
        "y": "",
      },
      node: expect.any(Element)
    });
    expect(mockShowToast).toHaveBeenCalled();
    expect(mockShowToast.mock.calls[0][0].node.innerHTML).toContain('it doesn\'t work');
  });

  it('should properly display toast with button', () => {
    const callback = () => { };
    service.error('with button', 'click me', callback);

    expect(Toastify).toHaveBeenCalledWith({
      position: 'center',
      duration: 8000,
      className: 'error ',
      gravity: '',
      offset: {
        "y": "",
      },
      node: expect.any(Element)
    });
    expect(mockShowToast).toHaveBeenCalled();
    expect(mockShowToast.mock.calls[0][0].node.innerHTML).toContain('with button');

    const buttonEl = mockShowToast.mock.calls[0][0].node.querySelector('button');
    expect(buttonEl.textContent).toEqual('click me');
    expect(buttonEl.eventListeners('click')[0]).toBe(callback);
  });

  it('should properly display toast from bottom of screen', () => {
    const callback = () => { };
    service.error('with button', 'click me', callback, 'center', '30px');

    expect(Toastify).toHaveBeenCalledWith({
      position: 'center',
      duration: 8000,
      className: 'error ',
      gravity: 'bottom',
      offset: {
        "y": "30px",
      },
      node: expect.any(Element)
    });
    expect(mockShowToast).toHaveBeenCalled();
  });
});
