import { Injectable } from '@angular/core';
import { BehaviorSubject, Subject } from 'rxjs';
import { ApiService } from "../api/api.service";
import { CoreEnvService } from "../core-env.service";

@Injectable()
export class ScopeService {
  private baseUrl: string;
  private deploymentHistoryUrl: string;
  private allScopes: any;
  private allHierarchyScopes = new Subject<any>();
  private selectedStoresCount = new Subject<any>();
  private allHierarchyScopesValue = [];
  private allHierarchyScopesDefaultValue: string;
  private isScopeModified = new Subject<any>();
  private selectTypeSource = new BehaviorSubject("");
  selectType = this.selectTypeSource.asObservable();

  private selectedStoreCountValue = new BehaviorSubject(0);
  eligibleStoreCount = this.selectedStoreCountValue.asObservable();

  changeStoreValue(checked: number) {
    this.selectedStoreCountValue.next(checked);
  }

  constructor(private api: ApiService) {
    this.baseUrl = CoreEnvService.get('restaurantBaseUrl');
    this.deploymentHistoryUrl = `${this.baseUrl}/restaurant_assets/deployment_history`;
  }

  private getBackendBaseHeaders() {
    return {
        Authorization: "Bearer " + CoreEnvService.get('msalIDToken')
    };
}

  public async getAllScopes(pageOffset?, pageNo?: number): Promise<any> {
    let queryParams = new URLSearchParams();
    if (pageOffset) {
      queryParams.append('offset', pageOffset ? pageOffset.toString() : '');
    }
    if (pageNo) {
      queryParams.append('page', pageNo ? pageNo.toString() : '');
    }

    const scopeList = await this.api.get<any>(`${this.baseUrl}/v1/scopes?${queryParams}`, null, null, this.getBackendBaseHeaders());
    return scopeList;
  }
  /**
   * 
   * @param selectedFilters 
   * @returns restaurants for the deployment group based on filter criteria.
   */
  async getDeploymentHistory(selectedFilters) {
    const urlSearchParams = new URLSearchParams({
      'page[limit]': "-1"
    });
    urlSearchParams.append('include', 'restaurant');
    if (selectedFilters?.deploymentId) {
      urlSearchParams.append('filter[deployment_group.id]', selectedFilters?.deploymentId);
    }
    if (selectedFilters?.applicationName && selectedFilters?.applicationName?.toLowerCase() !== 'all') {
      urlSearchParams.append('filter[applicationName]', selectedFilters?.applicationName);
    }
    if (selectedFilters?.productId && selectedFilters?.productId?.toLowerCase() !== 'all') {
      urlSearchParams.append('filter[productVersions.products.id]', selectedFilters?.productId);
    }

    if (selectedFilters?.status && selectedFilters?.status?.toLowerCase() !== 'all') {
      if (selectedFilters?.status?.toLowerCase() === 'completed') {
        urlSearchParams.append('filter[status]', `${selectedFilters?.status},Already-Present`)
      } else if (selectedFilters?.status?.toLowerCase() === 'inprogress') {
        urlSearchParams.append('filter[status]', 'In-Progress')
      } else {
        urlSearchParams.append('filter[status]', selectedFilters?.status);
      }
    }

    const response = await this.api.get<any>(`${this.deploymentHistoryUrl}?${urlSearchParams.toString()}`, null, null, this.getBackendBaseHeaders());
    return response;
  }

  setSelectType(data) {
    this.selectTypeSource.next(data);
  }

  getSelectType() {
    return this.selectTypeSource.asObservable();
  }

  emitScopesDetails(scopes) {
    this.allScopes = scopes;
  }

  getScopesDetails() {
    return this.allScopes;
  }


  emitAllHierarchyScopes(hierarchyScopes) {
    this.allHierarchyScopes.next(hierarchyScopes);
    this.allHierarchyScopesValue = hierarchyScopes;
    this.checkIfScopeIsModified();
  }

  resetData() {
    this.allHierarchyScopes.next(
      JSON.parse(this.allHierarchyScopesDefaultValue)
    );
    this.allHierarchyScopesValue = JSON.parse(
      this.allHierarchyScopesDefaultValue
    );
    this.selectedStoresCount = new Subject<any>();
  }

  getAllHierarchyScope() {
    return this.allHierarchyScopes;
  }

  getSelectedStoresCount() {
    return this.selectedStoresCount;
  }

  emitSelectedStoresCount(count) {
    this.selectedStoresCount.next(count);
  }

  getAllHierarchyScopesValue() {
    return [...this.allHierarchyScopesValue];
  }

  setAllHierarchyScopesDefaultValue(scopes) {
    this.allHierarchyScopesDefaultValue = JSON.stringify([...scopes]);
    this.allHierarchyScopesValue = [...scopes];
    this.allHierarchyScopes.next(scopes);
  }

  getAllHierarchyScopesDefaultValue() {
    return this.allHierarchyScopesDefaultValue;
  }

  hasModifiedScope() {
    return this.isScopeModified;
  }

  /**
   * Fetches all scopes from API service
   */
  // public async fetchAllScopes() {
  //   const scopes = await this.api.get<Scopes>(
  //     `${this.baseUrl}/v1/scopes`,
  //     null,
  //     null,
  //     this.getBackendBaseHeaders()
  //   );
  //   const hierarchyScopes = this.createScopesHierarchy(scopes.data);
  //   this.emitScopesDetails(scopes.data);
  //   this.setAllHierarchyScopesDefaultValue(hierarchyScopes);
  // }

  /**
   * Contruct hierarchy for the market
   * @param scopes
   * @returns array of hierachy object for each market
   */
  createScopesHierarchy(scopes) {
    return scopes["markets"].map((market) => ({
      ...market,
      marketSelected: false,
      marketSelectedDefault: false,
      regions: scopes["regions"]
        .filter((v) => v["market_scope_id"] === market["id"])
        .map((region) => ({
          ...region,
          regionSelected: false,
          regionSelectedDefault: false,
          coops: scopes["coops"]
            .filter((v) => v["region_scope_id"] === region["id"])
            .map((coop) => ({
              ...coop,
              coopSelected: false,
              coopSelectedDefault: false,
              stores: scopes["stores"]
                .filter((v) => v["coop_scope_id"] === coop["id"])
                .map((store) => ({
                  ...store,
                  storeSelected: false,
                  storeSelectedDefault: false,
                })),
            })),
        })),
      globalStores: {
        stores: scopes["stores"].filter((v) => v["coop_scope_id"] === null),
      },
    }));
  }

  /**
   * Filters selected regions in market
   * @param market object of market hierarchy
   * @returns array of selected regions within market
   */
  getSelectedRegionsInMarket(market, allselected?) {
    return market?.regions?.reduce((acc, region, index) => {
      if (region.regionSelected || allselected) {
        return [
          ...acc,
          {
            ...region,
            hierarchyIndex: index,
          },
        ];
      }
      return acc;
    }, []);
  }

  /**
   * Filters selected coops in region
   * @param region object of region hierarchy
   * @returns array of selected coops within region
   */
  getSelectedCoopsInRegion(region, allselected?) {
    return region?.coops?.reduce((acc, coop, index) => {
      if (coop.coopSelected || allselected) {
        return [
          ...acc,
          {
            ...coop,
            hierarchyIndex: index,
          },
        ];
      }
      return acc;
    }, []);
  }

  /**
   * Filters selected store in coop
   * @param coop object of coop hierarchy
   * @returns array of selected store within coop
   */
  getSelectedStoresInCoop(coop, allselected?) {
    return coop?.stores?.filter((store) => (store.storeSelected || allselected));
  }

  /**
   * Filters hierarchy for only selected scopes
   * @param scopesHierarchy array of market hierarchy
   * @returns filtered array of market with only selected regions and coops
   */
  getSelectedScopesHierarchy(scopesHierarchy, allselected?) {
    return scopesHierarchy?.map((market) => ({
      ...market,
      regions: this.getSelectedRegionsInMarket(market, allselected)?.map((region) => ({
        ...region,
        coops: this.getSelectedCoopsInRegion(region, allselected)?.map((coop) => ({
          ...coop,
          stores: this.getSelectedStoresInCoop(coop, allselected),
        })),
      })),
    }));
  }

  // gets the ids of the default scope so that they can be removed
  editScopeIdsRemoved(defaultAssociationLevel) {
    const scopeIdsRemoved = [];
    this.allHierarchyScopesValue.forEach((market) => {
      if (defaultAssociationLevel === 0 && market.marketSelectedDefault) {
        scopeIdsRemoved.push(market.id);
      } else {
        market.regions?.forEach((region) => {
          if (defaultAssociationLevel === 1 && region.regionSelectedDefault) {
            scopeIdsRemoved.push(region.id);
          } else {
            region.coops?.forEach((coop) => {
              if (defaultAssociationLevel === 2 && coop.coopSelectedDefault) {
                scopeIdsRemoved.push(coop.id);
              } else {
                coop.stores?.forEach((store) => {
                  if (
                    defaultAssociationLevel === 3 &&
                    store.storeSelectedDefault
                  ) {
                    scopeIdsRemoved.push(store.id);
                  }
                });
              }
            });
          }
        });
      }
    });
    return scopeIdsRemoved;
  }

  //gets ids of market when association is edited to market
  getScopeIdsOfMarketsEdit() {
    return this.allHierarchyScopesValue.map((market) => market.id);
  }

  // gets ids of region when association is edited to region
  getScopeIdsOfRegionEdit() {
    const scopeIdsSelected = [];
    this.allHierarchyScopesValue.forEach((market) => {
      market.regions?.forEach((region) => {
        if (region.regionSelected) {
          scopeIdsSelected.push(region.id);
        }
      });
    });
    return scopeIdsSelected;
  }

  //gets ids of coops when association is edited to coops
  getScopeIdsOfCoopEdit() {
    const scopeIdsSelected = [];
    this.allHierarchyScopesValue.forEach((market) => {
      market.regions?.forEach((region) => {
        region.coops?.forEach((coop) => {
          if (coop.coopSelected) {
            scopeIdsSelected.push(coop.id);
          }
        });
      });
    });
    return scopeIdsSelected;
  }

  //gets ids of stores when association is edited to stores
  getScopeIdsOfStoresEdit() {
    const scopeIdsSelected = [];
    this.allHierarchyScopesValue.forEach((market) => {
      market.regions?.forEach((region) => {
        region.coops?.forEach((coop) => {
          coop.stores?.forEach((store) => {
            if (store.storeSelected) {
              scopeIdsSelected.push(store.id);
            }
          });
        });
      });
    });
    return scopeIdsSelected;
  }

  /**
   * Gets scope ids of selected markets
   * @returns selected / removed scope ids
   */
  getScopeIdsOfSelectedMarkets() {
    // Logic needs to get updated for multiple markets
    return {
      scopeIdsSelected: this.allHierarchyScopesValue.map((market) => market.id),
      scopeIdsRemoved: [],
    };
  }

  /**
   * Gets scope ids of selected regions
   * @returns selected / removed scope ids
   */
  getScopeIdsOfSelectedRegions(isEditMode = false) {
    const scopeIdsSelected = [];
    const scopeIdsRemoved = [];
    this.allHierarchyScopesValue.forEach((market) => {
      market.regions?.forEach((region) => {
        if (!isEditMode) {
          if (region.regionSelected) {
            scopeIdsSelected.push(region.id);
          }
        } else {
          if (region.regionSelected && !region.regionSelectedDefault) {
            scopeIdsSelected.push(region.id);
          }
          if (!region.regionSelected && region.regionSelectedDefault) {
            scopeIdsRemoved.push(region.id);
          }
        }
      });
    });
    return {
      scopeIdsSelected,
      scopeIdsRemoved,
    };
  }

  /**
   * Gets scope ids of selected coops
   * @returns selected / removed scope ids
   */
  getScopeIdsOfSelectedCoops(isEditMode = false) {
    const scopeIdsSelected = [];
    const scopeIdsRemoved = [];
    this.allHierarchyScopesValue.forEach((market) => {
      market.regions?.forEach((region) => {
        region.coops?.forEach((coop) => {
          if (!isEditMode) {
            if (coop.coopSelected) {
              scopeIdsSelected.push(coop.id);
            }
          } else {
            if (coop.coopSelected && !coop.coopSelectedDefault) {
              scopeIdsSelected.push(coop.id);
            }
            if (!coop.coopSelected && coop.coopSelectedDefault) {
              scopeIdsRemoved.push(coop.id);
            }
          }
        });
      });
    });
    return {
      scopeIdsSelected,
      scopeIdsRemoved,
    };
  }

  /**
   * Gets scope ids of selected stores
   * @returns selected / removed scope ids
   */
  getScopeIdsOfSelectedStores(isEditMode = false) {
    const scopeIdsSelected = [];
    const scopeIdsRemoved = [];
    const selectedStores = [];
    const selectedRegions = [];
    this.allHierarchyScopesValue.forEach((market) => {
      market.regions?.forEach((region) => {
        if (region.regionSelected) {
          selectedRegions.push(region);
        }
        region.coops?.forEach((coop) => {
          coop.stores?.forEach((store) => {
            if (!isEditMode) {
              if (store.storeSelected) {
                selectedStores.push(store);
                scopeIdsSelected.push(store.id);
              }
            } else {
              if (store.storeSelected && !store.storeSelectedDefault) {
                selectedStores.push(store);
                scopeIdsSelected.push(store.id);
              }
              if (!store.storeSelected && store.storeSelectedDefault) {
                selectedStores.push(store);
                scopeIdsRemoved.push(store.id);
              }
            }
          });
        });
      });
    });
    return {
      scopeIdsSelected,
      scopeIdsRemoved,
      selectedStores,
      selectedRegions
    };
  }

  /**
   * Calls API service to save associated scopes to profile
   * @param profileId
   */
  async associateScopesToProfile(
    profileId: string,
    associationLevel: number,
    defaultAssociationLevel: number,
    isEditMode: boolean,
    dissociateIDs: Array<number> = [],
    onSuccess,
    onFailure
  ) {
    let selections = { scopeIdsSelected: [], scopeIdsRemoved: [] };
    if (isEditMode && associationLevel !== defaultAssociationLevel) {
      selections.scopeIdsRemoved = this.editScopeIdsRemoved(
        defaultAssociationLevel
      );
      switch (associationLevel) {
        case 0: //When Selected Market
          selections.scopeIdsSelected = this.getScopeIdsOfMarketsEdit();
          break;
        case 1: //When Selected Region
          selections.scopeIdsSelected = this.getScopeIdsOfRegionEdit();
          break;
        case 2: //When Selected CO-OP
          selections.scopeIdsSelected = this.getScopeIdsOfCoopEdit();
          break;
        case 3: //When Selected Stores
          selections.scopeIdsSelected = this.getScopeIdsOfStoresEdit();
          break;
      }
    } else {
      switch (associationLevel) {
        case 0: // For Market
          selections = this.getScopeIdsOfSelectedMarkets();
          break;
        case 1: // For Region
          selections = this.getScopeIdsOfSelectedRegions(isEditMode);
          break;
        case 2: // For Co-op
          selections = this.getScopeIdsOfSelectedCoops(isEditMode);
          break;
        default:
          // For Store
          selections = this.getScopeIdsOfSelectedStores(isEditMode);
      }
    }

    if (dissociateIDs.length) {
      selections.scopeIdsRemoved = dissociateIDs;
      selections.scopeIdsSelected = [];
    }
    if (
      selections.scopeIdsSelected.length ||
      selections.scopeIdsRemoved.length
    ) {
      try {
        const payload = {
          associate_scope_ids: selections.scopeIdsSelected,
          remove_associate_scope_ids: selections.scopeIdsRemoved,
        };
        const response = await this.api.post(
          `${this.baseUrl}/v1/profile/${profileId}/scopes`,
          payload,
          dissociateIDs.length
            ? "Scopes Dissociated Successfully"
            : "Scope has been associated successfully",
          dissociateIDs.length
            ? "Cannot Dissociate scopes"
            : "Error while associating scope",
          this.getBackendBaseHeaders()
        );
        if (response["status"]?.toLowerCase() === "success") {
          this.resetData();
          onSuccess();
        } else {
          onFailure();
        }
      } catch (error) {
        onFailure();
      }
    } else {
      this.resetData();
      onSuccess();
    }
  }

  /**
   * Checks if scope has meen modified locally
   */
  checkIfScopeIsModified() {
    // Check if scope is modified
    const isModified =
      this.allHierarchyScopesDefaultValue !==
      JSON.stringify(this.allHierarchyScopesValue);
    this.isScopeModified.next(isModified);
  }
}
