import { Injectable } from "@angular/core";
import { productResponse } from "../../components/add-product/add-product.interface";
import { CoreModule } from "../../core.module";
import { ApiService } from "../api/api.service";
import { CoreEnvService } from "../core-env.service";

@Injectable()

export class ProductsService {
    private baseUrl: string;

    constructor(private api: ApiService) {
        this.baseUrl = CoreEnvService.get('restaurantBaseUrl');
    }

    private getBackendBaseHeaders() {
        return {
            Authorization: "Bearer " + CoreEnvService.get('msalIDToken')
        };
    }

    /**
 * This function makes an API call to fetch list of products
 * @returns get response from products API call
 */
    public async getProduct(): Promise<productResponse[]> {
        return this.api.get(`${this.baseUrl}/restaurant_assets/products?page[limit]=-1`, null, null, this.getBackendBaseHeaders()).then(response => {
            return this.reshapeProducts(response);
        });
    }

    /**
    * This function takes in respone from products API call and reshapes to product name and its id
    * @param res from API call
    * @returns formatted products data
    */
    reshapeProducts(res) {
        return res?.data?.map(item => {
            return {
                label: item?.attributes?.name,
                value: item.id
            }
        })
    }

    /**
   * This function makes an API call to fetch respective product versions
   * @param id of type number as in product id 
   * @returns get response from products version API call
   */
    public async getProductVersions(id: number, pageOffset?: number, pageNo?: number, sortDirection?: string): Promise<any> {
        let queryParams = new URLSearchParams();

        queryParams.append('filter[products.id]', id?.toString());
        if (sortDirection) {
            queryParams.append('sort', sortDirection?.toLowerCase() !== 'asc' ? '-created' : 'created');
        }

        if (!pageOffset && !pageNo) {
            queryParams.append('page[limit]', '-1');
        } else {
            if (pageNo) queryParams.append('page[offset]', ((pageNo - 1) * pageOffset).toString());
            if (pageOffset) queryParams.append('page[limit]', pageOffset.toString());
        }

        return this.api.get(`${this.baseUrl}/restaurant_assets/product_versions?${queryParams}`, null, null, this.getBackendBaseHeaders()).then(versionResponse => {
            return this.reshapeProductVersion(versionResponse);
        });
    }

    /**
    * This functiont akes in response from product version and reshapes it to respective version name
    * and its id
    * @param res from API call
    * @returns formatted product version data
    */
    reshapeProductVersion(ver) {
        return ver?.data?.map(item => {
            return {
                label: item.attributes.versionName,
                value: item.id
            }
        });
    }

    /**
    * This function makes an API call to fetch respective product versions applications
    * @param id of type number as in product version id 
    * @returns get response from products version applications API call
    */
    public async getProductVersionApplications(id: number): Promise<any> {

        return this.api.get(`${this.baseUrl}/restaurant_assets/product_applications?filter[productVersions.id]=${id}&page[limit]=-1`, null, null, this.getBackendBaseHeaders()).then(applicationRes => {
            return this.reshapeApplication(applicationRes);
        })
    }

    /**
 * This function takes in respone from product version API call and reshapes data as requied
 * @param res from API call
 * @returns formatted application version data
 */
    reshapeApplication(res) {
        return res?.data?.map(item => {
            return {
                applicationName: item.attributes.applicationName,
                applicationType: item.attributes.applicationType,
                applicationVersion: item.attributes.applicationVersion,
                created: item.attributes.created,
                id: item.id
            }
        });
    }

}