import { Injectable } from "@angular/core";
import { ApiService } from "../api/api.service";
import { CoreEnvService } from "../core-env.service";

@Injectable()

export class HierarchyApiService {
  private baseUrl: string;
  constructor(private api: ApiService) {
    this.baseUrl = CoreEnvService.get('restaurantBaseUrl');
  }

  private getBackendBaseHeaders() {
    return {
      Authorization: "Bearer " + CoreEnvService.get('msalIDToken')
    };
  }

  /**
 * 
 * @param coop 
 * @param scopes 
 * @returns stores inside a coop.
 */
  filteredStores(coop, scopes) {
    return scopes["stores"].filter((v) => v?.attributes?.coopNodeId === coop).map((store) => ({
      ...store,
      storeSelected: false,
      storeSelectedDefault: false,
    }))
  }

  /**
   * 
   * @param region 
   * @param scopes 
   * @returns Coops inside a region
   */
  filteredCoops(region, scopes) {
    return scopes["coops"]?.filter((v) => v?.regionNodeId === region).map(coop => ({
      ...coop,
      coopSelected: false,
      coopSelectedDefault: false,
      stores: this.filteredStores(coop?.id, scopes)
    }))
  }

  /**
   * 
   * @param market 
   * @param scopes 
   * @returns regions under the market
   */
  filteredRegions(market, scopes) {
    return scopes["regions"]?.filter((v) => v?.marketNodeId === market?.id)?.map(region => ({
      ...region,
      regionSelected: false,
      regionSelectedDefault: false,
      marketName: market?.nodeName,
      coops: this.filteredCoops(region?.id, scopes)
    }))
  }

  /**
       * 
       * @param scopes 
       * @returns scope hierarchy structure
       */
  createScopesHierarchy(scopes) {
    return scopes["markets"].map((market) => ({
      ...market,
      marketSelected: false,
      marketSelectedDefault: false,
      regions: this.filteredRegions(market, scopes),
      globalStores: {
        stores: scopes["stores"].filter((v) => v?.coopNodeId === null),
      },
    }));
  }

  /**
     * 
     * @returns all restaurant data.
     */
  public async getAllRestaurantDetails(restaurantsList?) {
    const urlSearchParams = new URLSearchParams();
    if (restaurantsList) {
      urlSearchParams.append('filter[id]', restaurantsList);
      urlSearchParams.append("fields", 'nodeName,marketName,coopName,regionName,restaurantNo,marketNodeId,coopNodeId,regionNodeId,ipAddress');
    } else {
      urlSearchParams.append("fields", "restaurantNo,regionName,regionNodeId,coopNodeId,nodeName,ipAddress,storeAddress");
    }
    urlSearchParams.append('page[limit]', '-1');
    const response = await this.api.get<any>(`${this.baseUrl}/restaurant_assets/restaurantDetails?${urlSearchParams.toString()}`, null, null, this.getBackendBaseHeaders());
    return response?.data;
  }

  /**
       * 
       * @returns hierarchy values.
       */
  public async getAllHierarchyValues(releaseType) {
    let restaurants;
    if (releaseType?.toLowerCase() !== 'restaurantonboarding') {
      await this.getAllRestaurantDetails().then(res => {
        restaurants = res;
      })
    } else {
      const urlSearchParams = new URLSearchParams();
      urlSearchParams.append("fields", "restaurantNo,regionNodeId,coopNodeId");
      urlSearchParams.append('filter[hierarchyLevel]', '3300');
      urlSearchParams.append('page[limit]', '-1');
      const response = await this.api.get<any>(`${this.baseUrl}/restaurant_assets/bre_hierarchy_nodes?${urlSearchParams.toString()}`, null, null, this.getBackendBaseHeaders());
      restaurants = response?.data;
    }

    let response = await this.api.get<any>(`${this.baseUrl}/store-dashboard/hierarchy-filter`, null, null, this.getBackendBaseHeaders());

    response.data.stores = restaurants;
    return response;
  }

  /**
     * 
     * @returns market , region , coop , stores hierarchy
     */
  public async getAllHierarchyNodeValues(releaseType) {
    const hierArchy = await this.getAllHierarchyValues(releaseType);
    let scopes = this.createScopesHierarchy(hierArchy?.data);
    return scopes;
  }
}
