import { Inject, Injectable } from "@angular/core";
import { HttpClient } from "@angular/common/http";
import { Observable } from "rxjs";
import { ToastService } from "../toast/toast.service";
import { DOCUMENT } from "@angular/common";

@Injectable()
export class ApiService {
  constructor(
    private http: HttpClient,
    private toast: ToastService,
    @Inject(DOCUMENT) private document: any
  ) {}

  get<T>(
    url: string,
    successMessage?: string,
    failureMessage?: string,
    headers?: { [header: string]: string },
    error403Message?: string,
    showToastMessage?: boolean
  ): Promise<T> {
    return this.handleResponse<T>(
      this.http.get<T>(url, { headers }),
      successMessage,
      failureMessage,
      undefined,
      undefined,
      error403Message,
      showToastMessage
    );
  }

  post<T>(
    url: string,
    body: any,
    successMessage?: string,
    failureMessage?: string,
    headers?: { [header: string]: string },
    error403Message?: string,
    showToastMessage?: boolean
  ): Promise<T> {
    return this.handleResponse<T>(
      this.http.post<T>(url, body, { headers }),
      successMessage,
      failureMessage,
      undefined,
      undefined,
      error403Message,
      showToastMessage
    );
  }

  put<T>(
    url: string,
    body: any,
    successMessage?: string,
    failureMessage?: string,
    headers?: { [header: string]: string },
    error403Message?: string,
    showToastMessage?: boolean
  ): Promise<T> {
    return this.handleResponse<T>(
      this.http.put<T>(url, body, { headers }),
      successMessage,
      failureMessage,
      undefined,
      undefined,
      error403Message,
      showToastMessage
    );
  }

  patch<T>(
    url: string,
    body: any,
    successMessage?: string,
    failureMessage?: string,
    headers?: { [header: string]: string },
    position?,
    bottomOffset?,
    error403Message?: string,
    showToastMessage?: boolean
  ): Promise<T> {
    return this.handleResponse<T>(
      this.http.patch<T>(url, body, { headers }),
      successMessage,
      failureMessage,
      position,
      bottomOffset,
      error403Message,
      showToastMessage
    );
  }

  delete<T>(
    url: string,
    successMessage?: string,
    failureMessage?: string,
    headers?: { [header: string]: string },
    error403Message?: string,
    showToastMessage?: boolean
  ): Promise<T> {
    return this.handleResponse<T>(
      this.http.delete<T>(url, { headers }),
      successMessage,
      failureMessage,
      undefined,
      undefined,
      error403Message,
      showToastMessage
    );
  }

  private handleResponse<T>(
    request: Observable<T>,
    successMessage?: string,
    failureMessage?: string,
    position?: string,
    bottomOffset?: string,
    error403Message?: string,
    showToastMessage?: boolean
  ): Promise<T> {
    return new Promise((resolve, reject) => {
      request
        .toPromise()
        .then((value) => {
          if (!!successMessage) {
            this.toast.success(
              successMessage,
              undefined,
              undefined,
              position,
              bottomOffset
            );
          }
          resolve(value);
        })
        .catch((reason) => {
          if (reason.status == "401") {
            this.toast.error("Session Expired! Refreshing the session.");
            setTimeout(() => {
              localStorage.clear();
              this.document.location.href = "/";
            }, 1000);
          } else if (reason.status == "403") {
            localStorage.setItem("error403RequestedStatus", "");
            let error403MessageDetail = "";
            //EM_UAE = New user(OO/McOpco), does not exist in the db
            //EM_URR = User's access request has been rejected
            //EM_URP = User's request is in requested state
            if (reason.error?.errors && reason.error?.errors?.length > 0) {
              if (
                reason.error?.errors[0]?.code === "EM_UAE" ||
                reason.error?.errors[0]?.code === "EM_URR"
              ) {
                error403MessageDetail = "true";
              } else if (reason.error?.errors[0]?.code === "EM_URP") {
                localStorage.setItem("error403RequestedStatus", "true");
              }
            }
            error403MessageDetail
              ? localStorage.setItem(
                  "error403MessageDetail",
                  error403MessageDetail
                )
              : localStorage.setItem("error403MessageDetail", "");
            error403Message
              ? localStorage.setItem("error403Message", error403Message)
              : localStorage.setItem("error403Message", "");
            this.document.location.href = "/403";
          } else if (!showToastMessage) {
            this.toast.error(
              failureMessage || "Error Completing Request. Try Again."
            );
          }
          reject(reason);
        });
    });
  }
}
