import { TestBed } from '@angular/core/testing';
import { HttpClientTestingModule, HttpTestingController } from '@angular/common/http/testing';

import { ApiService } from './api.service';
import { ToastService } from '../toast/toast.service';

const mockSuccessToast = jest.fn();
const mockErrorToast = jest.fn();
class MockToastService {
  success(message) {
    mockSuccessToast(message, 'success');
  }

  error(message) {
    mockErrorToast(message, 'error');
  }
}

describe('ApiService', () => {
  let service: ApiService;
  let httpMock: HttpTestingController;

  beforeEach(() => {
    TestBed.configureTestingModule({
      imports: [HttpClientTestingModule],
      providers: [ApiService, { provide: ToastService, useClass: MockToastService }]
    });
    service = TestBed.inject(ApiService);
    httpMock = TestBed.get(HttpTestingController);
  });
  afterEach(() => {
    httpMock.verify();
    mockSuccessToast.mockClear();
    mockErrorToast.mockClear();
  });

  it('should be created', () => {
    expect(service).toBeTruthy();
  });

  it('should handle GET requests correctly', () => {
    service.get<any>('foo/bar').then(response => {
      expect(response.foo).toEqual('bar');
      expect(response.xyz).toEqual(42);
    });

    const req = httpMock.expectOne('foo/bar');
    expect(req.request.method).toBe('GET');
    expect(req.request.headers.keys()).toStrictEqual([]);
    req.flush({ foo: 'bar', xyz: 42 });
  });

  it('should handle POST requests correctly', () => {
    service.post<any>('foo/bar', { body: 'data' }).then(response => {
      expect(response.success).toEqual(true);
    });

    const req = httpMock.expectOne('foo/bar');
    expect(req.request.method).toBe('POST');
    expect(req.request.body).toEqual({ body: 'data' });
    expect(req.request.headers.keys()).toStrictEqual([]);
    req.flush({ success: true });
  });

  it('should handle PUT requests correctly', () => {
    service.put<any>('foo/bar', { body: 'data' }).then(response => {
      expect(response.success).toEqual(true);
    });

    const req = httpMock.expectOne('foo/bar');
    expect(req.request.method).toBe('PUT');
    expect(req.request.body).toEqual({ body: 'data' });
    expect(req.request.headers.keys()).toStrictEqual([]);
    req.flush({ success: true });
  });

  it('should handle PATCH requests correctly', () => {
    service.patch<any>('foo/bar', { body: 'data' }).then(response => {
      expect(response.success).toEqual(true);
    });

    const req = httpMock.expectOne('foo/bar');
    expect(req.request.method).toBe('PATCH');
    expect(req.request.body).toEqual({ body: 'data' });
    expect(req.request.headers.keys()).toStrictEqual([]);
    req.flush({ success: true });
  });

  it('should handle DELETE requests correctly', () => {
    service.delete<any>('foo/bar').then(response => {
      expect(response.success).toEqual(true);
    });

    const req = httpMock.expectOne('foo/bar');
    expect(req.request.method).toBe('DELETE');
    expect(req.request.headers.keys()).toStrictEqual([]);
    req.flush({ success: true });
  });

  it('should display success message if provided', done => {
    service.get<any>('foo/bar', 'we good', 'oh noes!').then(() => {
      expect(mockSuccessToast).toHaveBeenCalledWith('we good', 'success');
      done();
    });

    const req = httpMock.expectOne('foo/bar');
    req.flush({ success: true });
  });

  it('should handle failed requests correctly', done => {
    service.get<any>('foo/bar', null, 'oh noes!').catch(reason => {
      expect(reason).toBeDefined();
      expect(mockErrorToast).toHaveBeenCalledWith('oh noes!', 'error');
      done();
    });

    const req = httpMock.expectOne('foo/bar');
    req.error(new ErrorEvent('error'));
  });

  it('should add headers correctly', () => {
    service.get<any>('foo/bar', null, null, { foo: 'bar', xyz: '123' });

    const req = httpMock.expectOne('foo/bar');
    expect(req.request.headers.get('foo')).toBe('bar');
    expect(req.request.headers.get('xyz')).toBe('123');
    req.flush({});
  });
});
