import {
  Component,
  Input,
  Output,
  OnInit,
  EventEmitter,
  SimpleChanges
} from "@angular/core";
import { v4 as uuid } from "uuid";
import { FormGroup, FormControl } from "@angular/forms";

@Component({
  selector: "bre-toggle-switch",
  templateUrl: "./toggle-switch.component.html",
  styleUrls: ["./toggle-switch.component.scss"]
})
export class ToggleSwitchComponent implements OnInit {
  @Input() label: string;
  @Input() disabled: boolean = false;
  @Input() parentForm: FormGroup;
  @Input() controlName: string;
  @Input() parentValue: string;
  @Input() initialValue: any;
  @Input() colorChangingLabel: boolean = false;
  @Input() setSwitchFalse: boolean = false;
  @Input() value: any;
  @Output() parentFormChanged = new EventEmitter<any>();
  @Output() toggleClicked = new EventEmitter<any>();

  labelUUID: string;
  componentUUID: string;

  ngOnInit(): void {
    this.labelUUID = uuid();
    this.value = this.value ? JSON.parse(this.value) : this.value;
    if (this.label) {
      const parsedLabel = this.label.replace(/\s+/g, "_").toLowerCase();
      this.componentUUID = `${parsedLabel}_${uuid()}`;
    }
    if (this.parentForm) {
      this.value = this.parentForm.value[this.controlName];
    } else {
      this.parentForm = new FormGroup({
        value: new FormControl({ value: this.value, disabled: this.disabled })
      });
      this.controlName = "value";
    }
    this.subscribeToFormChanges();
  }
  ngOnChanges() {
    this.value = this.value ? JSON.parse(this.value) : this.value;
    this.subscribeToFormChanges();
    if (this.setSwitchFalse) {
      this.parentForm.controls[this.controlName].setValue(false);
    }
  }

  toggleClick($event) {
    $event.stopPropagation();
    this.value = !this.parentForm.controls[this.controlName].value;
    this.parentForm.controls[this.controlName].setValue(this.value);
    this.emitToggleClicked();
    this.checkIfPristineAndEmit(this.value);
  }

  onChange() {
    this.value = !this.value;
    this.checkIfPristineAndEmit(this.value);
  }

  // ngOnChanges(changes: SimpleChanges): void {
  //   if (changes.value) {
  //     if (changes.value.previousValue !== changes.value.currentValue) {
  //       this.checkIfPristineAndEmit(changes.value.currentValue);
  //     }
  //   }
  // }

  private subscribeToFormChanges() {
    this.parentForm?.valueChanges.subscribe((changes) => {
      if (
        changes[this.controlName] !== undefined &&
        changes[this.controlName] !== this.value
      ) {
        this.value = changes[this.controlName];
        this.checkIfPristineAndEmit(this.value);
      }
    });
  }

  private checkIfPristineAndEmit(value) {
    this.markFieldAsPristineIfReset(value);
    //if there's an observer for the parentFormChanged Output, emit and use that to update the value
    if (this.parentFormChanged.observers.length > 0) {
      this.parentFormChanged.emit(value);
    } else {
      //otherwise, set the value directly in the parent form's control
      this.parentForm.controls[this.controlName].setValue(value);
    }
  }

  markFieldAsPristineIfReset(value) {
    if (value === this.initialValue) {
      this.parentForm.get(this.controlName).markAsPristine();
    } else {
      this.parentForm.get(this.controlName).markAsDirty();
    }
  }

  private emitToggleClicked() {
    if (this.toggleClicked.observers.length > 0) {
      this.toggleClicked.emit(this.value);
    }
  }
}
