import { Component, OnInit, ContentChild, ContentChildren, Directive, EventEmitter, Input, Output, QueryList, TemplateRef, ViewEncapsulation, OnChanges, SimpleChanges } from '@angular/core';
import { SelectionModel } from '@angular/cdk/collections';
import { SelectItem } from 'primeng/api';
import { v4 as uuid } from 'uuid';

@Directive({
  selector: 'bre-table-column'
})
export class TableColumnDirective {
  @Input() name: string;
  @Input() label: string;
  @Input() sortable: boolean;
  @Input() checkBox: boolean;
  @Input() checkboxFilter: boolean;
  @Input() dateFilter: boolean;
  @Input() moreInfo: boolean;
  @Input() expandButton: boolean;
  @Input() hidden: boolean = false;
  @Input() showSearch: boolean = false;
  @Input() showSearchAutoSuggest: boolean = false;
  @Input() showSearchAutoSuggestList = [];
  @Input() checkBoxList = [];
  @Input() hideFilterIcon:boolean = false;
  @Input() notEditable: boolean = false;
  @Input() textWrap: boolean = true;


  @ContentChild(TemplateRef) cellTemplate;

  constructor() { }
}

@Component({
  selector: 'bre-table',
  templateUrl: './table.component.html',
  styleUrls: ['./table.component.scss'],
  encapsulation: ViewEncapsulation.None
})
export class TableComponent<T> implements OnInit, OnChanges {
  @Input() data: T[];

  @Input() itemCount: number;
  @Input() showLoadingIndicator: boolean;

  @Input() currentPage: number = 1;
  @Input() currentPageSize: number = PageSizes.TwentyFive;
  @Input() sortBy: string;
  @Input() sortDirection: TableSortDirection;
  @Input() enableCheckBox: boolean;
  @Input() clearSelection: boolean;
  @Input() configPageSizes: any;
  @Input() configEnum: boolean =false;

  selection = new SelectionModel<any>(true, []);

  @Output() onRowClick: EventEmitter<TableRowClickEvent<T>> = new EventEmitter();
  @Output() onPageChange: EventEmitter<TablePageChangeEvent> = new EventEmitter();
  @Output() sortColumn: EventEmitter<any> = new EventEmitter();
  @Output() onSelectChange: EventEmitter<any> = new EventEmitter();

  @ContentChildren(TableColumnDirective) columns: QueryList<TableColumnDirective>;

  paginationOptions: SelectItem[];
  selectedOption: number;
  selectedPage: number;
  maxPages: number;
  pageNumbers: number[];
  prevPagesIndex: number;
  nextPagesIndex: number;
  labelUUID: string;
  currentlyDisplayedData: T[] = [];
  initialPageSize: number;
  
  constructor() {
    this.labelUUID = uuid();
  }

  ngOnInit() {
    const pageSizesArray = Object.values(PageSizes).filter(value => isNaN(Number(value)) === false);
    this.paginationOptions = pageSizesArray.map(x => { return { label: x.toString(), value: x }; });
    this.initialPageSize = this.configEnum ? +Object.keys(this.configPageSizes)[0] : PageSizes.TwentyFive;
    
    if (this.data && !this.itemCount) {
      this.itemCount = this.data.length;
    }
    this.updateSelectedPageSizeOption();
    if (!this.sortColumn.observers.length) {
      this.computePageNumbers();
    }
    if (this.enableCheckBox) {
      this.selection.isSelected = this.isRowChecked.bind(this);
    }
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes.currentPageSize && this.paginationOptions) {
      this.updateSelectedPageSizeOption();
    }
    if (changes.itemCount || changes.currentPageSize || changes.currentPage) {
      this.computePageNumbers();
    }
    if (this.enableCheckBox && changes.clearSelection) {
      this.selection.clear();
    }
    if (this.data && this.data.length) {
      this.currentlyDisplayedData = this.calculateCurrentlyDisplayedData(this.data);
    } else {
      this.currentlyDisplayedData  = [];
    }
  }

  getVisibleColumnsNames(): string[] {
    return this.columns.map(x => x.name);
  }

  calculateCurrentlyDisplayedData(data: T[]): T[] {
    this.computePageNumbers();
    return this.getDataForCurrentPage(data);
  }

  private getDataForCurrentPage(data: T[]): T[] {
    let result: T[];
    if (data.length > this.currentPageSize) {
      const endIndex = (this.currentPage * this.currentPageSize);
      const startIndex = endIndex - this.currentPageSize;
      result = data?.slice(startIndex, endIndex);
    } else {
      result = data;
    }
    return result;
  }

  handleRowClick(itemClicked: T, rowIndex: number): void {
    this.onRowClick.emit({
      item: itemClicked,
      index: rowIndex
    });
  }

  handleSortChange(sortBy: string): void {
    if (!this.sortBy) {
      this.sortBy = sortBy;
      this.sortDirection = TableSortDirection.asc;
    } else if (
      this.sortBy === sortBy &&
      this.sortDirection === TableSortDirection.asc
    ) {
      this.sortDirection = TableSortDirection.desc;
    } else if (
      this.sortBy === sortBy &&
      this.sortDirection === TableSortDirection.desc
    ) {
      this.sortDirection = TableSortDirection.asc;
    } else if (this.sortBy !== sortBy) {
      this.sortBy = sortBy;
      this.sortDirection = TableSortDirection.asc;
    }
    if (this.onPageChange.observers.length) {
      this.emitPageChangeEvent();
    }
    else {
      this.emitSortColumnEvent();
    }

  }

  computePageNumbers() {
    this.selectedPage = this.currentPage || 1;
    this.maxPages = Math.ceil(this.itemCount / this.currentPageSize);
    this.maxPages = Number.isNaN(this.maxPages) ? 0 : this.maxPages;
    if (this.selectedPage > this.maxPages) {
      this.selectedPage = this.maxPages;
    }
    this.pageNumbers = Array(this.maxPages).fill(0).map((x, i) => i + 1);
    if (this.selectedPage > 3) {
      this.prevPagesIndex = this.pageNumbers.findIndex(element => element == this.selectedPage) - 2 > 0 ? this.pageNumbers.findIndex(element => element == this.selectedPage) - 2 : 0;
      this.nextPagesIndex = this.pageNumbers.findIndex(element => element == this.selectedPage) + 2 <= this.maxPages ? this.pageNumbers.findIndex(element => element == this.selectedPage) + 2 : this.maxPages;
    } else {
      this.prevPagesIndex = 0;
      this.nextPagesIndex = 4;
    }
  }

  previousPage(): void {
    this.selection.clear();
    this.handlePageChange((this.selectedPage - 1), this.currentPageSize);
  }

  nextPage(): void {
    this.selection.clear();
    this.handlePageChange((this.selectedPage + 1), this.currentPageSize);
  }

  handlePageChange(pageNumber: number, pageSize: number): void {
    this.currentPage = pageNumber;
    this.currentPageSize = pageSize;
    this.computePageNumbers();
    this.emitPageChangeEvent();
    this.currentlyDisplayedData = this.calculateCurrentlyDisplayedData(this.data);
  }

  emitPageChangeEvent() {
    this.onPageChange.emit({
      page: this.selectedPage,
      pageSize: this.selectedOption,
      sortBy: this.sortBy,
      sortDirection: TableSortDirection[this.sortDirection]
    });
  }

  emitSortColumnEvent() {
    this.sortColumn.emit({
      sortBy: this.sortBy,
      sortDirection: TableSortDirection[this.sortDirection]
    });
  }

  private updateSelectedPageSizeOption() {
    if (this.currentPageSize) {
      this.selectedOption = this.paginationOptions.find(item => item.value == this.currentPageSize).value;
    } else {
      this.selectedOption = this.paginationOptions[0].value;
      this.currentPageSize = this.selectedOption;
    }
  }

  public isAllSelected() {
    const numSelected = this.selection.selected.length;
    const numRows = this.data.length < this.currentPageSize ? this.data.length : this.currentPageSize;
    this.onSelectChange.emit(this.selection.selected)
    return numSelected === numRows;
  }

  public masterToggle() {
    this.isAllSelected() ?
      this.selection.clear() :
      this.data.forEach(row => this.selection.select(row));
  }

  public checkboxLabel(row?: any): string {
    return (!row)
      ? `${this.isAllSelected() ? 'select' : 'deselect'} all`
      : `${this.selection.isSelected(row) ? 'deselect' : 'select'} row ${this.data.indexOf(row) + 1}`;
  }

  public isRowChecked(row) {
    const found = this.selection.selected.find(el => {
      return el._id === row._id
    });
    if (found) {
      return true;
    }
    return false;
  }
}

export interface TableRowClickEvent<T> {
  item: T,
  index: number
}

export interface TablePageChangeEvent {
  page: number,
  pageSize: number,
  sortBy: string,
  sortDirection: TableSortDirection
}

export enum TableSortDirection {
  asc = 'asc',
  desc = 'desc'
}

export enum PageSizes {
  TwentyFive = 25,
  Fifty = 50,
  SeventyFive = 75,
  OneHundred = 100
}