import { Component, EventEmitter, Input, OnInit, Output } from "@angular/core";
import { FormBuilder, FormGroup } from "@angular/forms";
import { ScopeService } from "../../services/scope/scope.service";
import { HierarchyResponse, Region } from "../../interfaces/hierarchy-selection/hierarchy.interface";

@Component({
  selector: 'bre-store-select',
  templateUrl: './store-select.component.html',
  styleUrls: ['./store-select.component.scss']
})
export class StoreSelectComponent implements OnInit {
  searchForm: FormGroup;
  selectedStoresCount: number;
  totalStoresInPage: number;
  selectedStoresInPage: number;
  isAllStoresSelected: boolean;
  totalStoresCount: number;
  filteredSelections;
  activeIndex = {
    region: -1,
    coop: -1,
  };
  hasAllInView: boolean;
  selectStoresText =
    "Deselect restaurant(s) you don't wish to deploy update to";

  allStoresHiearchy: Region[];
  hierarchyRegions: Region[];
  allHierarchyScopes: HierarchyResponse[];
  totalStoresSelected: number;
  hasAllStoresSelectedInView: any;
  selectedStores: number;
  @Input() hierarchyScopeData;
  @Input() segmentedControl: boolean = false;
  @Output() hierarchyScopeDataEmit = new EventEmitter();
  @Output() changeStoreValueEmit = new EventEmitter();
  @Input() selectedIndex = 0;
  @Input() healthyScopeData: HierarchyResponse[];
  @Input() unHealthyScopeData: HierarchyResponse[];
  @Input() showPredeploymentText: boolean = false;

  constructor(private fb: FormBuilder, private scopeService: ScopeService) {
    this.searchForm = this.fb.group({
      searchKeyword: "",
    });
  }

  ngOnInit(): void {
    if(this.hierarchyScopeData) {
      this.scopeService.emitAllHierarchyScopes(this.hierarchyScopeData);
      this.processStoresSelections(this.hierarchyScopeData);
    } 
    this.tabChangeSetData();
    this.scopeService.getAllHierarchyScope().subscribe((storesHierarchy) => {
        this.processStoresSelections(storesHierarchy);
        if(this.totalStoresCount === 0){
          this.totalStoresCount = this.selectedStores;
        }
        this.totalStoresSelected = this.selectedStoresCount;
    }); 
    

    this.searchForm.valueChanges.subscribe(() => {
      this.getAllSelectedRegionsCoops({ keyword: this.searchForm.value.searchKeyword });
      // this.searchFormChange.emit({ keyword: this.searchForm.value.searchKeyword }); Need it for future reference
    });
  }

  tabChangeSetData() {
    if(this.segmentedControl && this.selectedIndex === 0) {
      this.scopeService.emitAllHierarchyScopes(this.healthyScopeData);
      this.processStoresSelections(this.healthyScopeData);
    } else if(this.segmentedControl && this.selectedIndex === 1) {
      this.scopeService.emitAllHierarchyScopes(this.unHealthyScopeData);
      this.processStoresSelections(this.unHealthyScopeData);
    }
  }

      /**
   * This method handles all side effects when stores hierarchy is updated
   * @param storesHierarchy
   */
      processStoresSelections(hierarchy = []) {
        if (hierarchy?.length) {
          this.allHierarchyScopes = hierarchy;
          this.getAllSelectedRegionsCoops({keyword: ""});
    
          if (
            this.activeIndex.region === -1 &&
            this.hierarchyRegions.length &&
            this.hierarchyRegions[0].coops?.length
          ) {
            this.activeIndex = {
              region: 0,
              coop: 0,
            };
          }
    
          this.hasAllStoresSelectedInView = this.checkIfAllSelectedInView(
            this.activeIndex.region,
            this.activeIndex.coop
          );
          this.scopeService.emitSelectedStoresCount(this.selectedStoresCount);
          this.changeStoreValueEmit.emit(this.selectedStoresCount);
          this.scopeService.changeStoreValue(this.selectedStoresCount);
        }
      }

  /**
 * 
 * @param $event 
 * filter based on search keyword.
 */
  getAllSelectedRegionsCoops($event) {
    this.selectedStoresCount = 0;
    this.totalStoresCount = 0;
    const [market] = this.allHierarchyScopes; // To be handled for multiple markets differently
    // // Get selected regions and coops
    this.hierarchyRegions = this.scopeService
      .getSelectedRegionsInMarket(market)?.map((region) => ({
        ...region,
        coops: this.scopeService
          .getSelectedCoopsInRegion(region)?.map((coop) => ({
            ...coop,
            stores: coop.stores?.map((store) => {
              // Count if selected
              this.totalStoresCount += 1;
              if (store.storeSelected) {
                this.selectedStoresCount = this.selectedStoresCount + 1;
              }
              const str = `${store?.attributes?.restaurantNo} - ${store?.attributes?.nodeName}`;
              const keyword = $event?.keyword;
              const isHidden =
                str.toLowerCase().indexOf(keyword?.toLowerCase()) === -1;
              return {
                ...store,
                isHidden,
              };
            }),
          })),
      }));

    this.filteredSelections = [...this.hierarchyRegions]  
      
    this.allStoresHiearchy = [...this.hierarchyRegions];

    this.isAllStoresSelected = this.selectedStoresCount == this.totalStoresCount;
  }

  /**
 * 
 * @param $event 
 * Capture store selection from checkbox event and emit the hierarchy level changes. 
 */
  selectRestaurant(e, regionIndex, coopIndex, storeIndex) {
    const hierarchyIndexOfRegion =
      this.hierarchyRegions[regionIndex].hierarchyIndex;
    const hierarchyIndexOfCoop =
      this.hierarchyRegions[regionIndex].coops[coopIndex].hierarchyIndex;
    if (this.allHierarchyScopes?.length) {
      const [market] = this.allHierarchyScopes; // To be handled for multiple markets differently
      market.regions[hierarchyIndexOfRegion].coops[hierarchyIndexOfCoop].stores[
        storeIndex
      ].storeSelected = e.checkboxChecked;
      if (e.checkboxChecked) {
        this.selectedStoresCount = this.selectedStoresCount + 1;
      } else {
        this.selectedStoresCount = this.selectedStoresCount - 1;
      }
      this.allHierarchyScopes = [market];
      this.scopeService.emitSelectedStoresCount(this.selectedStoresCount);
      this.changeStoreValueEmit.emit(this.selectedStoresCount);
      this.scopeService.changeStoreValue(this.selectedStoresCount);
      this.scopeService.emitAllHierarchyScopes(this.allHierarchyScopes);
      this.hierarchyScopeDataEmit.emit(this.allHierarchyScopes)
      if (this.totalStoresSelected !== this.totalStoresCount) {
        this.isAllStoresSelected = false;
      } else {
        this.isAllStoresSelected = true;
      }
    }
  }

  /**
* Checks if all stores in view are selected for the co-op
* @param regionIndex
* @param coopIndex
* @returns boolean
*/
  checkIfAllSelectedInView(regionIndex, coopIndex) {
    let hasAllSelected = false;
    if (
      this.hierarchyRegions?.length &&
      regionIndex !== -1 &&
      coopIndex !== -1
    ) {
      const hierarchyIndexOfRegion =
        this.hierarchyRegions[regionIndex]?.hierarchyIndex;
      if (this.hierarchyRegions[regionIndex]?.coops?.length && !this.segmentedControl) {
        const hierarchyIndexOfCoop =
          this.hierarchyRegions[regionIndex]?.coops[coopIndex]?.hierarchyIndex;
        if (this.allHierarchyScopes?.length) {
          const [market] = this.allHierarchyScopes; // To be handled for multiple markets differently
          const stores =
            market.regions[hierarchyIndexOfRegion].coops[hierarchyIndexOfCoop]
              .stores;
          const { totalStoresInView, totalSelectedStoresInView } =
            stores.reduce(
              (acc, store, storeIndex) => {
                const isHidden =
                  this.hierarchyRegions[regionIndex].coops[coopIndex].stores[
                    storeIndex
                  ].isHidden;
                return {
                  totalStoresInView: isHidden
                    ? acc.totalStoresInView
                    : acc.totalStoresInView + 1,
                  totalSelectedStoresInView:
                    !isHidden && store.storeSelected
                      ? acc.totalSelectedStoresInView + 1
                      : acc.totalSelectedStoresInView,
                };
              },
              {
                totalStoresInView: 0,
                totalSelectedStoresInView: 0,
              }
            );
          hasAllSelected = totalStoresInView === totalSelectedStoresInView;
          this.selectedStoresInPage = totalSelectedStoresInView;
          this.totalStoresInPage = totalStoresInView;
        }
      } else {
        const hierarchyIndexOfCoop =
        this.hierarchyRegions[regionIndex]?.coops[coopIndex]?.hierarchyIndex;
      if (this.allHierarchyScopes?.length) {
        const [market] = this.allHierarchyScopes; // To be handled for multiple markets differently
        const stores =
          market?.regions[hierarchyIndexOfRegion]?.coops[hierarchyIndexOfCoop]
            .stores;
        let totalStoresInView;
        let totalSelectedStoresInView;
        if(this.selectedIndex === 0) {
          totalStoresInView = stores?.filter(store => !store?.unHealthy)?.length;
          totalSelectedStoresInView = stores?.filter(store => store?.storeSelected && !store?.unHealthy)?.length;
        } else {
          totalStoresInView = stores?.filter(store => store?.unHealthy)?.length;
          totalSelectedStoresInView = stores?.filter(store => store?.storeSelected && store?.unHealthy)?.length;
        }
        hasAllSelected = totalStoresInView === totalSelectedStoresInView;
        this.selectedStoresInPage = totalSelectedStoresInView;
        this.totalStoresInPage = totalStoresInView;
      }
      }
    }
    this.isAllStoresSelected ? hasAllSelected : hasAllSelected = false;
    return hasAllSelected;
  }

  /**
   * 
   * @param $event 
   */
  viewStoresInCoop(regionIndex,coopIndex) {
    this.activeIndex = {
      region: regionIndex,
      coop: coopIndex,
    };
    this.hasAllStoresSelectedInView = this.checkIfAllSelectedInView(
      regionIndex,
      coopIndex
    );
  }

    /**
   * 
   * @param $event 
   * selectAllStoresInCoop and emit the selection changes.
   */
    selectAllCoop(e,regionIndex,coopIndex){
      const hierarchyIndexOfRegion =
        this.hierarchyRegions[regionIndex].hierarchyIndex;
      const hierarchyIndexOfCoop =
        this.hierarchyRegions[regionIndex].coops[coopIndex].hierarchyIndex;
      if (this.allHierarchyScopes?.length) {
        const [market] = this.allHierarchyScopes; // To be handled for multiple markets differently
        const stores =
          market.regions[hierarchyIndexOfRegion].coops[hierarchyIndexOfCoop]
            .stores;
        market.regions[hierarchyIndexOfRegion].coops[
          hierarchyIndexOfCoop
        ].stores = stores.map((store, storeIndex) => {
          // Change only those which are in view (not hidden)
          if (
            !this.hierarchyRegions[regionIndex].coops[coopIndex].stores[
              storeIndex
            ].isHidden
          ) {
            if (e.checkboxChecked) {
              this.selectedStoresCount = this.selectedStoresCount + 1;
            } else {
              this.selectedStoresCount = this.selectedStoresCount - 1;
            }
            return {
              ...store,
              storeSelected: e.checkboxChecked,
            };
          }
          return store;
        });
        this.allHierarchyScopes = [market];
        this.scopeService.emitSelectedStoresCount(this.selectedStoresCount);
        this.changeStoreValueEmit.emit(this.selectedStoresCount);
        this.scopeService.changeStoreValue(this.selectedStoresCount);
        this.scopeService.emitAllHierarchyScopes(this.allHierarchyScopes);
        this.hierarchyScopeDataEmit.emit(this.allHierarchyScopes)
        if(this.totalStoresSelected !== this.totalStoresCount){
          this.isAllStoresSelected = false;
        }else{
          this.isAllStoresSelected = true;
        }
      }
    }



  /**
   * @param $event
   * Select or deselects all stores in the selected regions and coops at once.
   */
  selectAllStores() {
    this.isAllStoresSelected = !this.isAllStoresSelected;
    this.allStoresHiearchy.map(region => {
      return region?.coops.map(coop => {
        return coop.stores?.map(store => {
          store.storeSelected = this.isAllStoresSelected;
          return store;
        })
      })
    })
    if (!this.isAllStoresSelected) {
      this.totalStoresSelected = 0;
    } else {
      this.totalStoresSelected = this.totalStoresCount;
    }
    this.allHierarchyScopes[0].regions = [...this.hierarchyRegions];
    this.scopeService.emitAllHierarchyScopes(this.allHierarchyScopes);
    this.hierarchyScopeDataEmit.emit(this.allHierarchyScopes)
  }

  changeTab(event) {
    if(event) {
      this.selectedIndex = event.index;
      this.tabChangeSetData();
    }
  }
}
