import { Component, EventEmitter, Input, OnInit, Output } from '@angular/core';
import { FormBuilder, FormGroup } from '@angular/forms';
import { HierarchyApiService } from "../../services/hierarchy/hierarchy-api.service";
import { ScopeService } from "../../services/scope/scope.service";
import { HierarchyResponse, HierarchyMultiSelectOption, Region } from "../../interfaces/hierarchy-selection/hierarchy.interface";
import { SelectedRestaurantsDialogComponent } from "../../components/selected-restaurants-dialog/selected-restaurants-dialog.component";
import { DialogService } from "../../services/dialog/dialog.service";

@Component({
  selector: 'bre-select-hierarchy',
  templateUrl: './select-hierarchy.component.html',
  styleUrls: ['./select-hierarchy.component.scss']
})
export class SelectHierarchyComponent implements OnInit {
  @Input() isMultiSelect: boolean = false;
  @Input() releaseType: string;
  isLoading: boolean;
  hierarchyForm: FormGroup;
  regionMultiSelectOptions: HierarchyMultiSelectOption[];
  coopMultiSelectOptions: HierarchyMultiSelectOption[];
  selectedStoreCount: number;
  marketName: string;
  storeSelectionType: string;
  disableCoop: boolean = true;
  public activeIndex = {
    region: -1,
    coop: -1,
  };
  hierarchyRegions;
  restaurantOptions;
  editMode: boolean = false;
  storesHierarchy;
  @Output() nextStep = new EventEmitter();
  @Output() reselectStores = new EventEmitter();
  @Output() storeSelection = new EventEmitter();
  @Output() targetRegion = new EventEmitter();
  @Output() targetCoop = new EventEmitter();
  @Output() targetRestaurant = new EventEmitter();
  storeSelections: HierarchyResponse[];
  selectedStores;
  totalStoresCount: number;
  totalStoresSelected: number;
  selectedStoresCount: number;
  allHierarchyScopes: HierarchyResponse[];
  selectedStoresInPage: number;
  totalStoresInPage: number;
  isAllStoresSelected: boolean;
  hasAllStoresSelectedInView: boolean;
  allStoresHiearchy: Region[];



  constructor(private hierarchyService: HierarchyApiService, private scopeService: ScopeService, private fb: FormBuilder, private dialog: DialogService) {
    this.hierarchyForm = this.fb.group({
      regionSelection: '',
      coopSelection: '',
      restaurantSelection: ''
    });
    this.selectedStoreCount = 0;
  }
  ngOnInit(): void {
    this.isLoading = true;
    this.hierarchyService.getAllHierarchyNodeValues(this.releaseType).then(res => {
      this.storesHierarchy = res;
      this.initializeHierarchyOptions(["region", "coop"]);
      this.isLoading = false;
    }).catch(err => {
      this.isLoading = false;
      console.error(err);
    })
    this.scopeService.getAllHierarchyScope().subscribe((storesHierarchy) => {
      this.storeSelections = storesHierarchy;
      this.processStoresHierarchy(storesHierarchy);
      this.processStoresSelections(storesHierarchy);
      if (this.totalStoresCount === 0) {
        this.totalStoresCount = this.selectedStores;
      }
      this.totalStoresSelected = this.selectedStoresCount;
    });
  }

  /**
* This method handles all side effects when stores hierarchy is updated
* @param storesHierarchy
*/
  processStoresSelections(hierarchy = []) {
    if (hierarchy?.length) {
      this.allHierarchyScopes = hierarchy;
      this.getAllSelectedRegionsCoops({ keyword: "" });

      if (
        this.activeIndex.region === -1 &&
        this.hierarchyRegions.length &&
        this.hierarchyRegions[0].coops?.length
      ) {
        this.activeIndex = {
          region: 0,
          coop: 0,
        };
      }

      this.hasAllStoresSelectedInView = this.checkIfAllSelectedInView(
        this.activeIndex.region,
        this.activeIndex.coop
      );
      this.scopeService.emitSelectedStoresCount(this.selectedStoresCount);
      this.scopeService.changeStoreValue(this.selectedStoresCount);
    }
  }
  /**
   * 
   * @param $event 
   * filter based on search keyword.
   */
  getAllSelectedRegionsCoops($event) {
    this.selectedStoresCount = 0;
    this.totalStoresCount = 0;
    const [market] = this.allHierarchyScopes; // To be handled for multiple markets differently
    // // Get selected regions and coops
    this.hierarchyRegions = this.scopeService
      .getSelectedRegionsInMarket(market)?.map((region) => ({
        ...region,
        coops: this.scopeService
          .getSelectedCoopsInRegion(region)?.map((coop) => ({
            ...coop,
            stores: coop.stores?.map((store) => {
              // Count if selected
              this.totalStoresCount += 1;
              if (store.storeSelected) {
                this.selectedStoresCount = this.selectedStoresCount + 1;
              }
              const str = `${store?.attributes?.restaurantNo} - ${store?.attributes?.nodeName}`;
              const keyword = $event?.keyword;
              const isHidden =
                str.toLowerCase().indexOf(keyword?.toLowerCase()) === -1;
              return {
                ...store,
                isHidden,
              };
            }),
          })),
      }));
    this.allStoresHiearchy = [...this.hierarchyRegions];
    this.isAllStoresSelected = this.selectedStoresCount == this.totalStoresCount;
  }

  /**
* Checks if all stores in view are selected for the co-op
* @param regionIndex
* @param coopIndex
* @returns boolean
*/
  checkIfAllSelectedInView(regionIndex, coopIndex) {
    let hasAllSelected = false;
    if (
      this.hierarchyRegions?.length &&
      regionIndex !== -1 &&
      coopIndex !== -1
    ) {
      const hierarchyIndexOfRegion =
        this.hierarchyRegions[regionIndex]?.hierarchyIndex;
      if (this.hierarchyRegions[regionIndex]?.coops?.length) {
        const hierarchyIndexOfCoop =
          this.hierarchyRegions[regionIndex]?.coops[coopIndex]?.hierarchyIndex;
        if (this.allHierarchyScopes?.length) {
          const [market] = this.allHierarchyScopes; // To be handled for multiple markets differently
          const stores =
            market.regions[hierarchyIndexOfRegion].coops[hierarchyIndexOfCoop]
              .stores;
          const { totalStoresInView, totalSelectedStoresInView } =
            stores.reduce(
              (acc, store, storeIndex) => {
                const isHidden =
                  this.hierarchyRegions[regionIndex].coops[coopIndex].stores[
                    storeIndex
                  ].isHidden;
                return {
                  totalStoresInView: isHidden
                    ? acc.totalStoresInView
                    : acc.totalStoresInView + 1,
                  totalSelectedStoresInView:
                    !isHidden && store.storeSelected
                      ? acc.totalSelectedStoresInView + 1
                      : acc.totalSelectedStoresInView,
                };
              },
              {
                totalStoresInView: 0,
                totalSelectedStoresInView: 0,
              }
            );
          hasAllSelected = totalStoresInView === totalSelectedStoresInView;
          this.selectedStoresInPage = totalSelectedStoresInView;
          this.totalStoresInPage = totalStoresInView;
        }
      }
    }
    this.isAllStoresSelected ? hasAllSelected : hasAllSelected = false;
    return hasAllSelected;
  }


  /**
* This method handles all side effects when stores hierarchy is updated
* @param storesHierarchy
*/
  processStoresHierarchy(storesHierarchy) {
    this.storesHierarchy = storesHierarchy;
    if (storesHierarchy?.length) {
      // Get name of market (to be changed while handling multiple markets)
      this.marketName = storesHierarchy[0].name;
    }
    this.initializeHierarchyOptions(["region", "coop"]);
    this.scopeService.emitSelectedStoresCount(this.selectedStoresCount);
    this.scopeService.changeStoreValue(this.selectedStoresCount);
  }

  /**
  * 
  * @param hierarchyLevels 
  * Initializes region and coopoptions for dropdown.
  */
  initializeHierarchyOptions(hierarchyLevels) {
    if (hierarchyLevels.indexOf("region") > -1) {
      this.regionMultiSelectOptions = this.storesHierarchy?.reduce(
        (acc, market, marketIndex) => {
          let regionOptions: any[] = [];
          market['regions']?.forEach((region, regionIndex) => {
            if (this.releaseType !== 'software') {
              if (region?.nodeName?.toLowerCase() !== 'cloud') {
                regionOptions.push({
                  label: region?.nodeName,
                  value: region?.nodeName,
                  checked: region.regionSelected,
                  regionIndex,
                  marketIndex,
                });
              }
            } else {
              regionOptions.push({
                label: region?.nodeName,
                value: region?.nodeName,
                checked: region.regionSelected,
                regionIndex,
                marketIndex,
              });
            }

          });
          return [...acc, ...regionOptions];
        },
        []
      );
    }
    if (hierarchyLevels.indexOf("coop") > -1) {
      this.selectedStoreCount = 0;
      this.coopMultiSelectOptions = this.storesHierarchy?.reduce(
        (acc, market, marketIndex) => {
          let coopOptions: any[] = [];
          market.regions?.forEach((region, regionIndex) => {
            if (region.regionSelected) {
              region.coops?.forEach((coop, coopIndex) => {
                coopOptions.push({
                  label: coop?.nodeName,
                  value: coop?.nodeName,
                  checked: coop.coopSelected,
                  coopIndex,
                  regionIndex,
                  marketIndex,
                });
                if (coop.coopSelected && coop.stores) {
                  this.selectedStoreCount =
                    this.selectedStoreCount + coop.stores.length;
                }
              });
            }
          });
          return [...acc, ...coopOptions];
        },
        []
      );
    }
  }



  /**
   * Resets all selections in hierarchy dropdowns (to unselected)
   */
  resetHierarchySelection() {
    if (!this.isMultiSelect) {
      this.hierarchyForm.patchValue({
        regionSelection: '',
        coopSelection: '',
        restaurantSelection: ''
      })
    } else {
      this.scopeService.setSelectType('');
      this.storeSelectionType = '';
      if (this.selectedStoreCount) {
        this.regionMultiSelectOptions?.forEach(regionOption => {
          this.updateSelectedRegionsInHierarchy({ ...regionOption, checked: false })
        })
        this.scopeService.emitAllHierarchyScopes(this.storesHierarchy);
      }
    }
  }

  /**
   * 
   * @param $event 
   * This function catches the emit and the emits back the data to parent component
   */
  nextStepEmit($event) {
    this.nextStep.emit($event);
  }

  /**
   * 
   * @param $event 
   * This function catches the emit and the emits back the data to parent component
   */
  reselectStoresEmit($event) {
    this.reselectStores.emit($event);
  }

  /**
 * Select/unselect that region with all coops and stores inside it.
 */
  updateSelectedRegionsInHierarchy(option: any) {
    const storesHierarchy = [...this.storesHierarchy];
    const { marketIndex, regionIndex, checked } = option;
    storesHierarchy[marketIndex].regions[regionIndex].regionSelected = checked;
    storesHierarchy[marketIndex].regions[regionIndex].coops = storesHierarchy[
      marketIndex
    ].regions[regionIndex].coops?.map((coop) => ({
      ...coop,
      coopSelected: checked,
      stores: coop.stores?.map((store) => ({
        ...store,
        storeSelected: checked,
      })),
    }));
    this.storesHierarchy = storesHierarchy;
    this.scopeService.emitAllHierarchyScopes(this.storesHierarchy);
  }

  /**
   * Select/unselect that coop with all stores inside it.
   * @param option object for checked/unchecked option
   */
  updateSelectedCoopsInHierarchy(option: any) {
    const storesHierarchy = [...this.storesHierarchy];
    const { marketIndex, regionIndex, coopIndex, checked } = option;
    storesHierarchy[marketIndex].regions[regionIndex].coops[
      coopIndex
    ].coopSelected = checked;
    storesHierarchy[marketIndex].regions[regionIndex].coops[coopIndex].stores =
      storesHierarchy[marketIndex].regions[regionIndex].coops[
        coopIndex
      ].stores?.map((store) => ({
        ...store,
        storeSelected: checked,
      }));
    this.storesHierarchy = storesHierarchy;
    this.scopeService.emitAllHierarchyScopes(this.storesHierarchy);
  }
  /**
   * 
   * @param option 
   * @param hierarchyLevel 
   * Triggered when hierarchy options are changed and updates the selection state.
   */
  onHierarchyOptionChange(option, hierarchyLevel) {

    switch (hierarchyLevel) {
      case "region":
        this.hierarchyForm?.patchValue({ regionSelection: option?.value }, { emitEvent: false, onlySelf: true });
        if (option !== "All") {
          if (!this.isMultiSelect) {
            let previouslySelectedRegions = this.regionMultiSelectOptions?.filter(reg => {
              return reg.checked
            })

            if (previouslySelectedRegions?.length) {
              this.storesHierarchy[0].regions = this.storesHierarchy[0]?.regions?.map(regOption => {
                regOption.regionSelected = false;
                return regOption;
              })
            }

            let selectedRegion = this.regionMultiSelectOptions?.find(region => {
              return region?.label?.toLowerCase() === option?.label?.toLowerCase()
            })

            if (selectedRegion) {
              selectedRegion.checked = true;
              this.updateSelectedRegionsInHierarchy(selectedRegion);
            }

            let regionSelection = this.storesHierarchy[0]?.regions[selectedRegion?.regionIndex];
            this.targetRegion.emit(regionSelection);

          } else {
            this.updateSelectedRegionsInHierarchy(option);
          }
        } else {
          this.regionMultiSelectOptions?.forEach((regionOption) => {
            this.updateSelectedRegionsInHierarchy(regionOption);
          });
        }
        let checkedArray = this.regionMultiSelectOptions?.filter(option => {
          return option.checked === true;
        })

        this.disableCoop = checkedArray?.length === 0 ? true : false;
        break;
      case "coop":
        this.hierarchyForm?.patchValue({ coopSelection: option?.value }, { emitEvent: false, onlySelf: true });
        if (option !== "All") {
          if (!this.isMultiSelect) {
            if (this.coopMultiSelectOptions?.length) {
              let selectedCoop = this.coopMultiSelectOptions?.find(coop => {
                return coop?.label?.toLowerCase() === option?.label?.toLowerCase()
              })

              let filteredCoop = this.storesHierarchy[0]?.regions[selectedCoop?.regionIndex]?.coops[selectedCoop?.coopIndex];
              this.targetCoop.emit(filteredCoop);

              this.restaurantOptions = filteredCoop?.stores?.map((store, index) => {
                return {
                  label: store?.attributes?.restaurantNo,
                  value: store?.attributes?.restaurantNo,
                  restaurantIndex: index,
                  regionIndex: selectedCoop?.regionIndex,
                  coopIndex: selectedCoop?.coopIndex
                }
              })
            }
          } else {
            this.updateSelectedCoopsInHierarchy(option);
          }

        } else {
          this.coopMultiSelectOptions?.forEach((coopOption) => {
            this.updateSelectedCoopsInHierarchy(coopOption);
          });
        }
        break;

      case "restaurant":
        let selectedRestaurantOption = this.restaurantOptions?.find(res => {
          return res?.value === option.value
        })
        this.hierarchyForm?.patchValue({ restaurantSelection: option?.value }, { emitEvent: false, onlySelf: true });

        let selectedRestaurant = this.storesHierarchy[0]?.regions[selectedRestaurantOption?.regionIndex]?.coops[selectedRestaurantOption?.coopIndex]?.stores[selectedRestaurantOption?.restaurantIndex];
        this.targetRestaurant.emit(selectedRestaurant);
        // Update hierarchy selection state to service
        this.scopeService.emitAllHierarchyScopes(this.storesHierarchy);
    }
  }
}