import { async, ComponentFixture, TestBed } from '@angular/core/testing';
import { FormsModule, FormGroup, FormControl } from '@angular/forms';
import { AutoCompleteModule } from 'primeng/autocomplete';
import { SearchableSelectComponent } from './searchable-select.component';
import { InputErrorMessageComponent } from '../input-error-message/input-error-message.component';
import { MatMenuModule } from '@angular/material/menu';
import { PopoverMenuButtonComponent } from '../popover-menu-button/popover-menu-button.component';
import { IconButtonComponent } from '../icon-button/icon-button.component';
import { StatusPillComponent } from '../status-pill/status-pill.component';
import { ControlLabelComponent } from '../control-label/control-label.component';
import { TooltipDirective } from '../tooltip/tooltip.directive';
import { ErrorIconComponent } from '../error-icon/error-icon.component';
import { SimpleChange, EventEmitter } from '@angular/core';

describe('SearchableSelectComponent', () => {
  let component: SearchableSelectComponent;
  let fixture: ComponentFixture<SearchableSelectComponent>;

  beforeEach(async(() => {
    TestBed.configureTestingModule({
      imports: [AutoCompleteModule, FormsModule, MatMenuModule],
      declarations: [SearchableSelectComponent, InputErrorMessageComponent, StatusPillComponent, PopoverMenuButtonComponent, IconButtonComponent, ControlLabelComponent, TooltipDirective, ErrorIconComponent]
    })
      .compileComponents();
  }));

  beforeEach(() => {
    fixture = TestBed.createComponent(SearchableSelectComponent);
    component = fixture.componentInstance;
    fixture.detectChanges();
  });

  it('should create', () => {
    expect(component).toBeTruthy();
  });

  it('should display a label if one has been declared', () => {
    component.label = "Test";
    component.ngOnInit();
    fixture.detectChanges();
    expect(component.componentUUID).toContain('test');
  });

  it('should display a status pill if configured to do so', () => {
    component.isSelectable = true;
    component.statusPillHidden = false;
    component.statusPillText = "Test";
    component.ngOnInit();
    fixture.detectChanges();

    const pill = fixture.nativeElement.querySelector('bre-status-pill');

    expect(pill).toBeDefined();
    expect(component.componentUUID).toContain('test');
  });

  it('should lazy-load options if configured to do so', () => {
    component.filteredOptions = Array(1000).map(x => ({ label: '', value: '' }));
    component.lazyLoadOptions = true;

    component.loadOptions();
    expect(component.lazyLoadedOptions).toEqual(component.filteredOptions.slice(0, 99));
  });

  it('should mark the form field as pristine if reset to its initial value and dirty if set to another value', () => {
    component.initialValue = 1;
    component.controlName = 'testControl';
    component.parentForm = new FormGroup({ 'testControl': new FormControl([''])});
    const pristineSpy = spyOn(component.parentForm.get(component.controlName), 'markAsPristine');
    component.onSelect({ value: 1 });

    expect(pristineSpy).toHaveBeenCalled();

    const dirtySpy = spyOn(component.parentForm.get(component.controlName), 'markAsDirty');
    component.onSelect({ value: 2 });

    expect(dirtySpy).toHaveBeenCalled();
  });

  it('should force an item selection on blur if configured to do so by resetting to the previous value', () => {
    const item1 = {label: 'test item 1', value: 1, searchText: '1'};
    const item2 = {label: 'test item 2', value: 2, searchText: '2'};
    component.forceSelection = true;
    component.value = 1;
    component.options = [item1, item2];
    component.selectedItem = null;

    component.onBlur({});

    expect(component.selectedItem).toEqual(item1);
  });

  it('should emit a search event on complete', () => {
    component.options = [];
    const spy = spyOn(component.searchQuery, 'emit');
    const event = {query: 'test'};
    component.onComplete(event);
    expect(spy).toHaveBeenCalledWith({query: event.query, filteredOptions: []});
  });

  it('should lazy load options on complete if configured to do so', () => {
    component.options = [];
    component.filteredOptions = Array(1000).map(x => ({ label: '', value: '' }));
    component.lazyLoadOptions = true;
    const spy = spyOn(component, 'loadOptions');
    const event = {query: 'test'};
    component.onComplete(event);
    expect(spy).toHaveBeenCalled();
  });

  it('should emit a select event on click of the status pill if configured to do so', () => {
    component.selectDropdown = new EventEmitter<any>();
    component.statusPillText = "Test";

    const spy = spyOn(component.selectDropdown, 'emit');
    component.handleClick();
    
    expect(spy).toHaveBeenCalledWith(component.statusPillText);
  });
});
