import {
  Component,
  EventEmitter,
  ElementRef,
  Input,
  OnChanges,
  OnInit,
  Output,
  ViewChild,
  ViewEncapsulation
} from "@angular/core";
import { SelectionModel } from "@angular/cdk/collections";
import { InputComponent } from "../input/input.component";

@Component({
  selector: 'bre-searchable-multiselect-core',
  templateUrl: './searchable-multiselect.component.html',
  styleUrls: ['./searchable-multiselect.component.scss'],
  encapsulation: ViewEncapsulation.None
})
export class SearchableMultiselectComponent implements OnInit {
  @Input() columnData: { [key: string]: string }[] = [];
  @Input() customFilterOptions: { [key: string]: string }[];
  @Input() label: string;
  @Input() chipsEnabled: boolean;
  @Input() disable: boolean = false;
  filteredList: { [key: string]: any }[] = [];
  @Output() updateCheckboxFilters = new EventEmitter<{
    [key: string]: { [key: string]: string }[];
  }>();
  @Output() updatedCheckboxSearchField = new EventEmitter<any>();
  @Output() checkboxListScroll = new EventEmitter<any>();
  placeholderText: string;

  @ViewChild("searchField")
  private searchField: InputComponent;

  dedupedColumnData: { [key: string]: string }[];
  currentSearch: string = "";
  unselection: SelectionModel<{ [key: string]: string }> = new SelectionModel(
    true,
    []
  );
  selection: SelectionModel<{ [key: string]: string }> = new SelectionModel(
    true,
    []
  );
  showPanel: boolean = false;
  option: boolean = true;
  selectedItem = [];
  selectedItemCopy: { [key: string]: any }[] = [];
  showDropDownElement: boolean = false;
  wasInside = false;

  constructor(private _eref: ElementRef) { }

  ngOnInit(): void {
    if (this.columnData) {
      this.dedupedColumnData = this.dedupe([...this.columnData]);
      this.filteredList = this.customFilterOptions
        ? this.customFilterOptions
        : this.dedupedColumnData;
      this.toggleOptionValue();
    }
  }

  /**
     *
     * @param event click event
     * this function is called when ever user clicks on input field
     */
  showDropdown(event) {
    this.showDropDownElement = true;
  }
  /**
   * This function is called at many placed to set the option values based
   * on changes happening in the component
   */
  toggleOptionValue() {
    this.filteredList.filter((option, index) => {
      return this.filteredList[index].checked === false;
    }).length === 0
      ? (this.option = false)
      : (this.option = true);
    this.setPlaceholder();
  }

  ngOnChanges(): void {
    if (this.columnData) {
      this.dedupedColumnData = this.dedupe([...this.columnData]);
      this.filteredList = this.customFilterOptions
        ? this.customFilterOptions
        : this.dedupedColumnData;
      this.toggleOptionValue();
    }
  }
  /**
   *
   * @param event  input event
   * This is called when ever user enters some value
   * in to the input field and sets the filtered list
   */
  onListSearchChange(event): void {
    this.currentSearch = event;
    if (this.selectedItemCopy.length === 0) {
      this.selectedItemCopy = this.filteredList;
    } else {
      this.filteredList = this.selectedItemCopy;
    }
    this.filteredList = this.filteredList.filter((option, index) => {
      return this.filteredList[index].label
        .toLowerCase()
        .includes(event.value.toLowerCase());
    });
  }
  /**
   *
   * @param event scroll event
   * Called when user scrolls the muti select drop down valued
   */
  onFilterItemListScroll(event): void {
    if (!this.customFilterOptions) {
      this.checkboxListScroll.emit(event);
    }
  }
  /**
   *
   * @param list drop down lists
   * @returns lists by removing duplicate copies of data
   */
  private dedupe(
    list: { [key: string]: string }[]
  ): { [key: string]: string }[] {
    return list.filter((object, index, self) => {
      return (
        self.findIndex(
          (o) => o.label === object.label && o.value === object.value
        ) === index
      );
    });
  }

  toggleAndEmit(item): void {
    item.checked = !item.checked;
    this.toggleOptionValue();
    this.emitUpdatedSelection(item);
  }

  isRowChecked(item?): boolean {
    return !!this.selection.selected.find(
      (el) => item.label === el.label && item.value === el.value
    );
  }

  /**
   * Called when user checks and unchecks All option
   */
  selectOrDeselectAll(item): void {
    const data = this.filteredList;
    !!this.option
      ? data?.forEach((option, index) => (data[index].checked = true))
      : data?.forEach((option, index) => (data[index].checked = false)),
      this.unselection.select(...data);
    this.option = !this.option;
    this.emitUpdatedSelection(item);
  }

  /**
   *
   * @param item contains unselection items, selection items and current value
   * event is emit when ever selection or deselction is performed on the drop down values
   */
  emitUpdatedSelection(item): void {
    this.updateCheckboxFilters.emit({
      unselection: this.selectedUnselectedItems(false),
      selection: this.selectedUnselectedItems(true),
      value: item
    });
    this.selectedItem = this.selection.selected;
    this.toggleOptionValue();
  }

  /**
   * 
   * @param flag 
   * @returns selection state
   */
  selectedUnselectedItems(flag: boolean) {
    const val = this.filteredList.filter((option, index) => {
      return this.filteredList[index].checked === flag;
    });
    return val;
  }

  /**
   * sets the serach field to empty value
   */
  clearSearchField(): void {
    this.currentSearch = "";
    this.searchField.parentForm.reset();
  }

  /**
   * clear search field
   */
  clear(): void {
    this.clearSearchField();
    this.emitUpdatedSelection("Clear");
  }

  /**
   * If any of the drop down is open this function closes them
   */
  close() {
    this.showDropDownElement = false;
  }

  /**
   * This function is called at multiple places to set the
   * placeholder value for multi select drop down
   */
  setPlaceholder() {
    let uncheckedList = this.filteredList.filter((option, index) => {
      return this.filteredList[index].checked === false;
    }).length;
    let checkedList = this.filteredList.filter((option, index) => {
      return this.filteredList[index].checked === true;
    }).length;
    if (checkedList === 0) {
      this.placeholderText = "Select";
    } else {
      this.placeholderText = uncheckedList === 0 ? "All" : "Selected";
    }
  }

}
