import { Component, AfterViewInit, OnInit, ContentChildren, EventEmitter, Input, Output, QueryList, ViewEncapsulation, OnChanges, SimpleChanges } from '@angular/core';
import { SelectionModel } from '@angular/cdk/collections';
import { SelectItem } from 'primeng/api';
import { v4 as uuid } from 'uuid';
import { TableSortDirection, TablePageChangeEvent, TableRowClickEvent, PageSizes } from './quick-jump-navigation.interface';
import { TableColumnDirective } from '../table/table.component';

@Component({
    selector: 'bre-quick-jump-nav',
    templateUrl: './quick-jump-navigation.component.html',
    styleUrls: ['./quick-jump-navigation.component.scss'],
    encapsulation: ViewEncapsulation.None
})

export class QuickJumpNavigationComponent<T> implements OnInit, OnChanges, AfterViewInit {
    @Input() data: T[];
    @Input() type: string = "Profiles";
    @Input() itemCount: number;
    @Input() showLoadingIndicator: boolean;

    @Input() currentPage: number = 1;
    @Input() currentPageSize: number  = PageSizes.TwentyFive;
    @Input() sortBy: string;
    @Input() sortDirection: TableSortDirection;
    @Input() clearSelection: boolean;
    @Input() configPageSizes:any;
    @Input() configEnum:boolean;
    @Input() selectRowNumber:number;
    @Input() setFocusOnLoad:boolean = false;
    
    selection = new SelectionModel<any>(true, []);

    @Output() rowClick: EventEmitter<TableRowClickEvent<T>> = new EventEmitter();
    @Output() pageChange: EventEmitter<TablePageChangeEvent> = new EventEmitter();
    @Output() sortColumn: EventEmitter<any> = new EventEmitter();
    @Output() selectChange: EventEmitter<any> = new EventEmitter();

    @ContentChildren(TableColumnDirective) columns: QueryList<TableColumnDirective>;

    paginationOptions: SelectItem[];
    selectedOption: number;
    selectedPage: number;
    maxPages: number;
    pageNumbers: number[];
    prevPagesIndex: number;
    nextPagesIndex: number;
    labelUUID: string;
    selectedRowIndex: number;
    initialPageSize: number;
   
    constructor() {
    this.labelUUID = uuid();
    }

    ngOnInit() {
        const pageSizesArray = Object.values(this.configEnum?this.configPageSizes:PageSizes).filter(value => isNaN(Number(value)) === false);
        this.paginationOptions = pageSizesArray.map(x => {
            return {label: x.toString(), value: x}; 
        });  
        this.initialPageSize = this.configEnum ? +Object.keys(this.configPageSizes)[0] : PageSizes.TwentyFive;
        
        this.selectedRowIndex = this.selectRowNumber ? this.selectRowNumber : 0;
        
        if (this.data && !this.itemCount) {
            this.itemCount = this.data.length;
        }
        this.updateSelectedPageSizeOption();
        if (!this.sortColumn.observers.length) {
            this.computePageNumbers();
        }
    }

    ngAfterViewInit(){      
        if(this.selectRowNumber && this.setFocusOnLoad) {
            setTimeout(() => {
                var rows = document.querySelectorAll('.quick-jump-navigation__table tr');
                // line is zero-based
                // line is the row number that you want to see into view after scroll    
                rows[this.selectRowNumber].scrollIntoView({
                    behavior: 'smooth',
                    block: 'center'
                });
            }, 1000)
        }    
    }

    ngOnChanges(changes: SimpleChanges): void {
        if (changes.currentPageSize && this.paginationOptions) {
            this.updateSelectedPageSizeOption();
        }
        if (changes.itemCount || changes.currentPageSize || changes.currentPage) {
            this.computePageNumbers();
        }

    }

    getVisibleColumnsNames(): string[] {
        return this.columns.map(x => x.name);
    }

    // to handle actions when any profile is clicked
    handleRowClick(itemClicked: T, rowIndex: number): void {
        this.selectedRowIndex = rowIndex;
        this.rowClick.emit({
            item: itemClicked,
            index: rowIndex
        });
    }

    // handle change in sorting
    handleSortChange(): void {
        this.selectedRowIndex = 0;
        this.sortDirection = this.sortDirection === TableSortDirection.asc ? TableSortDirection.desc : TableSortDirection.asc;
        if (this.pageChange.observers.length) {
            this.emitPageChangeEvent();
        }
        else {
            this.emitSortColumnEvent();
        }
    }

    //compute page numbers based on items per list and total items
    computePageNumbers() {
        this.selectedPage = this.currentPage || 1;
        this.maxPages = Math.ceil(this.itemCount / this.currentPageSize);
        if (this.selectedPage > this.maxPages) {
            this.selectedPage = this.maxPages;
        }
        this.pageNumbers = Array(this.maxPages).fill(0).map((x, i) => i + 1);
        if (this.selectedPage > 3) {
            this.prevPagesIndex = this.pageNumbers.findIndex(element => element == this.selectedPage) - 2 > 0 ? this.pageNumbers.findIndex(element => element == this.selectedPage) - 2 : 0;
            this.nextPagesIndex = this.pageNumbers.findIndex(element => element == this.selectedPage) + 2 <= this.maxPages ? this.pageNumbers.findIndex(element => element == this.selectedPage) + 2 : this.maxPages;
        } else {
            this.prevPagesIndex = 0;
            this.nextPagesIndex = 4;
        }
    }

    previousPage(): void {
        this.selection.clear();
        this.handlePageChange((this.selectedPage - 1), this.currentPageSize);
    }

    nextPage(): void {
        this.selection.clear();
        this.handlePageChange((this.selectedPage + 1), this.currentPageSize);
    }

    // to handle page change event
    handlePageChange(pageNumber: number, pageSize: number): void {
        this.selectedRowIndex = 0;
        this.currentPage = pageNumber;
        this.currentPageSize = pageSize;
        this.computePageNumbers();
        this.emitPageChangeEvent();
    }

    emitPageChangeEvent() {
        this.pageChange.emit({
            page: this.selectedPage,
            pageSize: this.selectedOption,
            sortBy: this.sortBy,
            sortDirection: TableSortDirection[this.sortDirection]
        });
    }

    emitSortColumnEvent() {
        this.sortColumn.emit({
            sortBy: this.sortBy,
            sortDirection: TableSortDirection[this.sortDirection]
        });
    }

    // to get initial page size
    private updateSelectedPageSizeOption() {
        if (this.currentPageSize)  {
            this.selectedOption = this.paginationOptions.find(item => item.value === this.currentPageSize).value;
        } else {
            this.selectedOption =  this.paginationOptions[0].value;
            this.currentPageSize = this.selectedOption;          
        }
    }


}