import { Component, OnInit, Input, OnChanges, Output, EventEmitter } from '@angular/core';
import { SelectItem } from 'primeng/api';
import { v4 as uuid } from 'uuid';
import { FormGroup, FormControl } from '@angular/forms';
import { MenuItem } from '../popover-menu-button/popover-menu-button.component';
import { arraysAreEqual } from '../../utilities/general-utils';

@Component({
  selector: 'bre-multi-select',
  templateUrl: './multi-select.component.html',
  styleUrls: ['./multi-select.component.scss']
})
export class MultiSelectComponent implements OnInit {
  @Input() items: SelectItem[];
  @Input() parentForm: FormGroup;
  @Input() controlName: string;
  @Input() label: string;
  @Input() disabled: boolean = false;
  @Input() error: boolean;
  @Input() required: boolean;
  @Input() placeholder: string = 'Select';
  @Input() errorMessage: string;
  @Input() hasPopover: boolean = false;
  @Input() popoverMenuItems: MenuItem[] = [];
  @Input() parentValue;
  @Input() initialValue: any;
  @Input() colorChangingLabel: boolean = false;

  @Input() value: any;
  @Output() parentFormChanged = new EventEmitter<any>();

  labelUUID: string;
  componentUUID: string;

  ngOnInit(): void {
    this.labelUUID = uuid();
    if(this.label) {
      const parsedLabel = this.label.replace(/\s+/g, '_').toLowerCase();
      this.componentUUID = `${parsedLabel}_${uuid()}`;
    }
    if (this.parentForm) {
      this.value = this.parentForm.value[this.controlName];
    } else {
      this.parentForm = new FormGroup({
        value: new FormControl({ value: this.value, disabled: this.disabled })
      });
      this.controlName = 'value';
    }
    this.subscribeToFormChanges();
  }

  getItemLabel(itemValue) {
    return this.items.find(x => x.value === itemValue)?.label || itemValue;
  }

  unselectItem($event, itemValue) {
    if (!this.parentForm.disabled) {
      $event.stopPropagation();
      const newValue = this.parentForm.get(this.controlName).value.filter(x => x !== itemValue);
      this.parentForm.patchValue({ [this.controlName]: newValue });
      this.value = [...newValue];
      this.checkIfPristineAndEmit(this.value);
    }
  }

  changeSelections($event) {
    this.value = $event.value;
    this.checkIfPristineAndEmit(this.value);
  }

  private subscribeToFormChanges() {
    this.parentForm?.valueChanges.subscribe(changes => {
      if (changes[this.controlName] !== undefined && changes[this.controlName] !== this.value) {
        this.value = changes[this.controlName];
        this.checkIfPristineAndEmit(this.value);
      }
    });
  }

  private checkIfPristineAndEmit(value) {
    this.markFieldAsPristineIfReset(value);
    //if there's an observer for the parentFormChanged Output, emit and use that to update the value
    if(this.parentFormChanged.observers.length > 0) {
      this.parentFormChanged.emit(value);
    } else {
      //otherwise, set the value directly in the parent form's control
      this.parentForm.controls[this.controlName].setValue(value);
    }
  }
  
  markFieldAsPristineIfReset(value) {
    if (arraysAreEqual(value, this.initialValue)) {
      this.parentForm.get(this.controlName).markAsPristine();
    } else {
      this.parentForm.get(this.controlName).markAsDirty();
    }
  }
}