import {
  Component,
  OnInit,
  Input,
  EventEmitter,
  Output,
  SimpleChanges,
} from "@angular/core";
import { v4 as uuid } from "uuid";
import { FormGroup, FormControl } from "@angular/forms";
import { filter, debounceTime, distinctUntilChanged } from "rxjs/operators";

@Component({
  selector: "bre-input",
  templateUrl: "./input.component.html",
  styleUrls: ["./input.component.scss"],
})
export class InputComponent implements OnInit {
  @Input() inputType: string = "text";
  @Input() label: string;
  @Input() tooltip: string;
  @Input() parentForm: FormGroup;
  @Input() controlName: string;
  @Input() disabled: boolean = false;
  @Input() keyFilterRegex: RegExp = DefaultKeyFilters.any; // by default, exclude nothing
  @Input() maxLength: number;
  @Input() error: boolean;
  @Input() emptyInput: boolean;
  @Input() required: boolean;
  @Input() placeholder: string = "Enter";
  @Input() errorMessage: string;
  @Input() parentValue;
  @Input() initialValue: any = "";
  @Input() colorChangingLabel: boolean = false;
  @Input() fontWeight: string = "normal";
  @Input() isSelected: boolean = false;
  @Input() readOnly: boolean = false;
  @Input() updateOnFocusOut: boolean = false;
  @Input() showErrorTooltip: boolean;
  @Input() tooltipText: string;
  @Input() toolTipTextArray: string[];
  @Input() isFilter: boolean = false;
  @Input() enableDebounce: boolean = false;

  @Input() value: any = null;
  @Output() parentFormChanged = new EventEmitter<any>();
  @Output() inputFocusedOut = new EventEmitter<any>();
  @Output() enterKeyEvent = new EventEmitter<any>();
  @Input() mealLineupChanges: boolean;
  @Input() showClearButton: boolean = false;
  emptyInputErrorMessage: string = "Field cannot be empty";
  @Input() showSearchIcon: boolean = false;

  labelUUID: string;
  componentUUID: string;
  isScript: boolean;

  ngOnInit(): void {
    this.labelUUID = uuid();
    if (this.label) {
      const parsedLabel = this.label.replace(/\s+/g, "_").toLowerCase();
      this.componentUUID = `${parsedLabel}_${uuid()}`;
    }
    if (this.parentForm) {
      this.value = this.parentForm.value[this.controlName];
    } else {
      this.parentForm = new FormGroup({
        value: new FormControl({ value: this.value, disabled: this.disabled }),
      });
      this.controlName = "value";
    }
    this.subscribeToFormChanges();
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (this.readOnly) {
      setTimeout(() => this.parentForm.get(this.controlName).disable(), 200);
    } else {
      if (changes.value) {
        if (
          changes.value.previousValue !== changes.value.currentValue &&
          !this.updateOnFocusOut
        ) {
          setTimeout(() => {
            this.checkIfPristineAndEmit(
              this.initialValue,
              changes.value.currentValue,
              null,
              changes.label
            );
          });
        }
      }
    }
  }

  private subscribeToFormChanges() {
    if (this.enableDebounce) {
      this.parentForm?.valueChanges
        .pipe(filter(Boolean), debounceTime(1500), distinctUntilChanged())
        .subscribe((changes) => {
          if (
            changes[this.controlName] !== undefined &&
            changes[this.controlName] !== this.value
          ) {
            this.value = changes[this.controlName];
            this.checkIfPristineAndEmit(
              this.initialValue,
              this.value,
              null,
              this.label
            );
          }
        });
    } else {
      this.parentForm?.valueChanges.subscribe((changes) => {
        if (
          changes[this.controlName] !== undefined &&
          changes[this.controlName] !== this.value
        ) {
          this.value = changes[this.controlName];
          this.checkIfPristineAndEmit(
            this.initialValue,
            this.value,
            null,
            this.label
          );
        }
      });
    }
  }

  private checkIfPristineAndEmit(initialValue, value, currentIndex, label) {
    this.markFieldAsPristineIfReset(value);
    const isPristine = initialValue === value;
    //if there's an observer for the parentFormChanged Output, emit and use that to update the value
    if (this.parentFormChanged.observers.length > 0 && !this.updateOnFocusOut) {
      this.parentFormChanged.emit({
        isPristine,
        controlName: this.controlName,
        value,
        currentIndex,
        label,
      });
    } else {
      //otherwise, set the value directly in the parent form's control
      this.parentForm.controls[this.controlName].setValue(value);
    }
  }

  markFieldAsPristineIfReset(value) {
    if (value === this.initialValue) {
      this.parentForm.get(this.controlName).markAsPristine();
    }
  }
  /**
   * Check for script tags in input field and clear the field if found.
   */
  checkForScripts(value) {
    if (
      value.match(
        /(<|%3C)script[\s\S]*?(>|%3E)[\s\S]*?(<|%3C)(\/|%2F)script[\s\S]*?(>|%3E)/gi
      )
    ) {
      this.isScript = true;
    } else {
      this.error = false;
      this.errorMessage = "";
      this.isScript = false;
    }
  }

  /**
   * This function clears the input value
   */
  clearInput() {
    this.parentForm.controls[this.controlName].setValue(null);
  }

  /**
   * Check for errors, if found throw error and error message
   */
  checkForErrors(value) {
    if (this.emptyInput && !value) {
      this.parentForm.controls[this.controlName].setValue(null);
    }
  }

  verifyInput(value) {
    //this.checkForScripts(value);
    this.checkForErrors(value);
    if (this.inputFocusedOut.observers.length > 0) {
      this.inputFocusedOut.emit({
        controlName: this.controlName,
        value,
        currentIndex: null,
        label: this.label,
      });
    }
  }

  handleKeyup(event) {
    if (event.key == "Enter") {
      this.enterKeyEvent.emit(event);
      event.target.blur();
    }
  }
}

export const DefaultKeyFilters = {
  any: /[^]/,
  num: /[\d\-\.]/,
  alpha: /[a-z_]/i,
  alphanum: /[a-z0-9_]/i,
};
