import { async, ComponentFixture, TestBed } from '@angular/core/testing';
import { ReactiveFormsModule, FormGroup, FormControl } from '@angular/forms';
import { InputTextModule } from 'primeng/inputtext';
import { KeyFilterModule } from 'primeng/keyfilter';
import { TooltipModule } from 'primeng/tooltip';
import { InputComponent, DefaultKeyFilters } from './input.component';
import { InputErrorMessageComponent } from '../input-error-message/input-error-message.component'
import { ControlLabelComponent } from '../control-label/control-label.component';
import { TooltipDirective } from '../tooltip/tooltip.directive';
import { PopoverMenuButtonComponent } from '../popover-menu-button/popover-menu-button.component'
import { IconButtonComponent } from '../icon-button/icon-button.component';
import { MatMenuModule } from '@angular/material/menu';
import { ErrorIconComponent } from '../error-icon/error-icon.component';
import { SimpleChange } from '@angular/core';


describe('InputComponent', () => {
  let component: InputComponent;
  let fixture: ComponentFixture<InputComponent>;

  beforeEach(async(() => {
    TestBed.configureTestingModule({
      imports: [ReactiveFormsModule, InputTextModule, KeyFilterModule, TooltipModule, MatMenuModule],
      declarations: [InputComponent, TooltipDirective, InputErrorMessageComponent, ControlLabelComponent, TooltipDirective, PopoverMenuButtonComponent, IconButtonComponent, ErrorIconComponent]
    })
      .compileComponents();
  }));

  beforeEach(() => {
    fixture = TestBed.createComponent(InputComponent);
    component = fixture.componentInstance;
    fixture.detectChanges();
  });

  it('should create', () => {
    expect(component).toBeTruthy();
  });
  
  it('should have a configurable type', () => {
    let compiled = fixture.nativeElement;

    let inputElement = compiled.querySelector('input');
    expect(inputElement.type).toBe('text');

    component.inputType = 'number';
    fixture.detectChanges();
    compiled = fixture.nativeElement;
    inputElement = compiled.querySelector('input');
    expect(inputElement.type).toBe('number');

  });

  it('should display an error message for the input if in an error state', () => {
    let compiled = fixture.nativeElement;

    let errorMessageElement = compiled.querySelector('.error-message none');
    expect(errorMessageElement).toBeFalsy();

    component.error = true;
    component.errorMessage = 'You messed up!';
    fixture.detectChanges();
    compiled = fixture.nativeElement;
    errorMessageElement = compiled.querySelector('.error-message');
    expect(errorMessageElement).toBeTruthy();
    expect(errorMessageElement.textContent).toBe('You messed up!');
  });

  it('should conditionally allow a maximum number of characters based upon component input', () => {
    let compiled = fixture.nativeElement;

    let inputElement = compiled.querySelector('input');
    expect(inputElement.attributes['maxlength']).toBeUndefined();

    component.maxLength = 9;
    fixture.detectChanges();
    compiled = fixture.nativeElement;
    inputElement = compiled.querySelector('input');
    expect(inputElement.attributes['maxlength'].value).toBe(component.maxLength.toString());
  });

  it('should conditionally filter characters using a regex pattern based upon component input', () => {
    let compiled = fixture.nativeElement;

    let inputElement = compiled.querySelector('input');
    expect(inputElement.attributes['ng-reflect-pattern'].value).toBe(DefaultKeyFilters.any.toString());

    component.keyFilterRegex = DefaultKeyFilters.alphanum;
    fixture.detectChanges();
    compiled = fixture.nativeElement;
    inputElement = compiled.querySelector('input');
    expect(inputElement.attributes['ng-reflect-pattern'].value).toBe(DefaultKeyFilters.alphanum.toString());

    const blockSpaces = /[^\s]/;
    component.keyFilterRegex = blockSpaces;
    fixture.detectChanges();
    compiled = fixture.nativeElement;
    inputElement = compiled.querySelector('input');
    expect(inputElement.attributes['ng-reflect-pattern'].value).toBe(blockSpaces.toString());
  });

  it('should mark the form field as pristine if reset to its initial value', () => {
    component.initialValue = '1';
    component.controlName = 'testControl';
    component.parentForm = new FormGroup({ 'testControl': new FormControl([''])});
    const pristineSpy = spyOn(component.parentForm.get(component.controlName), 'markAsPristine');
    component.markFieldAsPristineIfReset('1');

    expect(pristineSpy).toHaveBeenCalled();
  });
});
