import {
  Component,
  OnInit,
  ContentChildren,
  EventEmitter,
  Input,
  Output,
  QueryList,
  OnChanges,
  SimpleChanges,
  ViewChildren,
} from "@angular/core";
import { SelectionModel } from "@angular/cdk/collections";
import { SelectItem } from "primeng/api";
import { v4 as uuid } from "uuid";
import * as _ from "lodash";
import { FilterOption } from "../filter-group/filter.interface";
import {
  TableColumnDirective,
  TableRowClickEvent,
  TableSortDirection,
  PageSizes,
} from "../table/table.component";
import { ColumnFilterDropdownComponent } from "../column-filter-dropdown/column-filter-dropdown.component";
import { FormBuilder, FormGroup } from "@angular/forms";
import { TablePageChangeEvent } from "../table/table.component";
import { of } from "rxjs";

@Component({
  selector: "bre-filter-table",
  templateUrl: "./filter-table.component.html",
  styleUrls: ["./filter-table.component.scss"],
})
export class FilterTableComponent<T> implements OnInit, OnChanges {
  @Input() originalData: T[] = [];

  @Input() itemCount: number;
  @Input() showLoadingIndicator: boolean;

  @Input() currentPage = 1;
  @Input() currentPageSize: number = PageSizes.TwentyFive;
  @Input() sortBy: string;
  @Input() sortDirection: TableSortDirection;
  @Input() enableCheckBox: boolean;
  @Input() clearSelection: boolean;
  @Input() checkboxKey: string;
  @Input() editable: boolean = false;
  @Input() expandable: boolean = false;
  @Input() showSelect: boolean = true;
  @Input() selectedColumnName: string;
  @Input() prefilteredColumnName: string;
  @Input() prefilledFilters: ColumnFilters;
  @Input() filterLabel = "Filter";
  @Input() configReset = true;
  @Input() configEnum: boolean = false;
  @Input() configPageSizes: any;
  @Input() showCountMessage: string = "";
  @Input() initialColumnFilters: [InitialColumnFilters];
  @Input() emptyInput: string;
  @Input() error: boolean = false;
  @Input() errorMessage: string;
  @Input() showErrorTooltip: boolean = false;
  @Input() tooltipText: string = "";
  @Input() toolTipTextArray: string[];
  @Input() isOriginalDataChange: boolean = false;
  @Input() confirmDisabled: boolean = false;
  @Input() hideEditingCrossIcon: boolean = false;
  selection: SelectionModel<any>;

  @Output() onRowClick: EventEmitter<TableRowClickEvent<T>> =
    new EventEmitter();
  @Output() sortColumn: EventEmitter<any> = new EventEmitter();
  @Output() onSelectChange: EventEmitter<any> = new EventEmitter();
  @Output() onFilterChange: EventEmitter<any> = new EventEmitter();
  @Output() filtersCleared: EventEmitter<boolean> = new EventEmitter();
  @Output() moreInfoClicked: EventEmitter<any> = new EventEmitter();
  @Output() updateConfirmed: EventEmitter<any> = new EventEmitter();
  @Output() onPageChange: EventEmitter<TablePageChangeEvent> =
    new EventEmitter();
  @Output() updateApplyClicked: EventEmitter<any> = new EventEmitter();
  @Output() getItemCount: EventEmitter<any> = new EventEmitter();
  @Output() validateInputEvent: EventEmitter<any> = new EventEmitter();
  @ContentChildren(TableColumnDirective)
  columns: QueryList<TableColumnDirective>;
  @ViewChildren(ColumnFilterDropdownComponent)
  filterDropdowns!: QueryList<ColumnFilterDropdownComponent>;

  paginationOptions: SelectItem[];
  selectedOption: number;
  selectedPage: number;
  maxPages: number;
  pageNumbers: number[];
  prevPagesIndex: number;
  nextPagesIndex: number;
  labelUUID: string;
  initialSortBy: string;
  initialSortDirection: TableSortDirection;
  allCurrentFilters: { [columnName: string]: ColumnFilters } = {};
  perColumnData: { [columnName: string]: string[] } = {};
  filteredData: T[] = null;
  currentlyDisplayedData: T[] = [];
  editing: boolean[];
  moreInfoHidden: boolean[] = [];
  expandedRows: T[] = [];
  visibleColumnsLength: number;
  editFormGroup: FormGroup;
  shouldClearFilters = false;
  initialPageSize: number;
  revertToInitialSort: boolean = false;
  initialOriginalData: T[] = [];
  //animation
  animationHeight = "180px";
  animationWidth = "180px";
  constructor(private fb: FormBuilder) {
    this.labelUUID = uuid();
  }

  /**
   * Initializing page size sort by column and sort direction
   * initial sort filters have been applied in this function
   */
  ngOnInit() {
    const pageSizesArray = Object.values(
      this.configEnum ? this.configPageSizes : PageSizes
    ).filter((value) => isNaN(Number(value)) === false);
    this.paginationOptions = pageSizesArray.map((x) => ({
      label: x.toString(),
      value: x,
    }));
    this.initialPageSize = this.configEnum
      ? +Object.keys(this.configPageSizes)[0]
      : PageSizes.TwentyFive;
    this.selection = new SelectionModel<any>(true, []);

    if (this.filteredData && !this.itemCount) {
      this.itemCount = this.filteredData.length;
    }
    this.updateSelectedPageSizeOption();

    if (!this.sortColumn.observers.length) {
      this.computePageNumbers();
    }

    if (this.enableCheckBox) {
      this.selection.isSelected = this.isRowChecked.bind(this);
    }

    // copying unsorted and unfiltered original data to an array
    this.initialOriginalData = this.initialOriginalData.concat(
      this.originalData
    );
    //BRMS-2430 This loop iterates through all the multiple initial filters upon load. Added this because prefilledFilters allows for one initial filter
    if (this.initialColumnFilters) {
      for (let i of this.initialColumnFilters) {
        this.onFilterUpdate([i.columnName, i.columnFilter, "", false]);
      }
    }
  }

  /**
   *
   * @param changes : previous and current values
   * tracks size of the page and change in the page number
   * also triggers on clearing the selections of the checkboxes.
   */
  ngOnChanges(changes: SimpleChanges): void {
    if (changes.currentPageSize && this.paginationOptions) {
      this.updateSelectedPageSizeOption();
    }
    if (changes.itemCount || changes.currentPageSize || changes.currentPage) {
      this.computePageNumbers();
    }
    if (this.enableCheckBox && changes.clearSelection) {
      if (this.selection) {
        this.selection.clear();
      }
    }

    if (
      (!this.filteredData && this.originalData && this.originalData.length) ||
      changes.originalData
    ) {
      this.initializeData(this.originalData);
      if (changes.originalData) {
        this.initialSortBy = this.sortBy;
        this.initialSortDirection = this.sortDirection;
        this.clearFilters();
      }
    }
  }

  /**
   *
   * @param originalData : unfiltered data
   * mapping the data in the columns of filter table
   * only if original data has data this function would run
   */
  private initializeData(originalData): void {
    if (originalData.length) {
      this.filteredData = [];
      this.perColumnData = this.buildPerColumnData(originalData);
      this.filteredData = this.filteredData.concat(originalData);
      this.currentlyDisplayedData = this.calculateCurrentlyDisplayedData(
        this.filteredData
      );
    }
  }
  validateInput($event) {
    this.validateInputEvent.emit($event);
  }
  /**
   *
   * @param data : filtered data
   * @returns re compute the page numbers and return the data for the current page selected
   */
  calculateCurrentlyDisplayedData(data: T[]): T[] {
    this.computePageNumbers();
    return this.getDataForCurrentPage(data);
  }

  /**
   *
   * @param length : length of the page
   * if the data is editable then reset the data with false values
   */
  resetEditableArray(length) {
    this.editing = new Array();
    for (let i = 0; i < length; i++) {
      this.editing.push(false);
    }
  }
  /**
   * to reset all the filters and sortings
   * building column data again with original data and reverting all the sortings to initial sort
   */
  clearFilters() {
    if (this.filterDropdowns) {
      this.filterDropdowns.forEach((dropdown) => dropdown.clearFilters());
    }
    this.filtersCleared.emit();
    this.revertToOriginalSort();
    if (this.isOriginalDataChange) {
      this.initializeData(this.originalData);
      this.emitFilteredData(this.originalData.length, this.originalData);
      return;
    }
    if (this.initialSortBy) {
      const sortedData = this.sortTableByColumn(
        this.initialSortBy,
        this.initialSortDirection,
        this.initialOriginalData
      );
      this.initializeData(sortedData);
      this.emitFilteredData(sortedData.length, sortedData);
    } else {
      this.initializeData(this.initialOriginalData);
      this.emitFilteredData(
        this.initialOriginalData.length,
        this.initialOriginalData
      );
    }
  }

  /**
   *
   * @param data: data is the filter table data
   * @returns data for the current page that user has selected based on page size
   */
  private getDataForCurrentPage(data: T[]): T[] {
    const endIndex = this.currentPage * this.currentPageSize;
    const startIndex = endIndex - this.currentPageSize;
    let result = data.slice(startIndex, endIndex);
    if (this.editable) {
      this.resetEditableArray(result.length);
    }
    if (this.expandable && result.length > 0 && this.columns) {
      this.isHiddenDataNull(result);
    }
    return result;
  }

  /**
   *
   * @param rowData : complete data on the selected page
   * @returns if column is hidden and column name doesnt exist or is null then
   *  hide moreInfo column, if expandable property is true
   */
  isHiddenDataNull(rowData: T[]) {
    this.moreInfoHidden = new Array();
    let isHiddenData = true;
    rowData.forEach((row) => {
      isHiddenData = true;
      this.columns
        .filter((column) => column.hidden)
        .forEach((column) => {
          if (row[column.name] && row[column.name] !== "null - null") {
            isHiddenData = false;
          }
        });
      if (isHiddenData) {
        this.moreInfoHidden.push(true);
      } else {
        this.moreInfoHidden.push(false);
      }
    });
    return isHiddenData;
  }

  /**
   *
   * @returns returns the header name of all the columns except the hidden column
   */
  getVisibleColumnsNames(): string[] {
    let result = this.columns.filter((x) => !x.hidden).map((x) => x.name);
    if (result.length !== this.visibleColumnsLength) {
      this.visibleColumnsLength = result.length;
    }
    return result;
  }

  /**
   *
   * @param data table data
   * @returns map the data in the columns and push label and value pairs
   */
  buildPerColumnData(data: T[]): { [columnName: string]: string[] } {
    return data.reduce((keyedByColumn, row) => {
      Object.entries(row).forEach(([columnLabel, columnValue]) => {
        if (_.get(keyedByColumn, columnLabel)) {
          keyedByColumn[columnLabel].push({
            label: columnValue,
            value: columnValue,
          } as FilterOption);
        } else {
          _.set(keyedByColumn, columnLabel, [
            { label: columnValue, value: columnValue } as FilterOption,
          ]);
        }
      });
      return keyedByColumn;
    }, {});
  }

  /**
   *
   * @param param0 : array of the following variables
   * columnName: name of the column where filter has to be applied
   * filtersForColumn: filters associated with the selected column
   * filterAction: action to be performed on filter
   * cle
   */
  onFilterUpdate([columnName, filtersForColumn, filterAction, clearCols]: [
    string,
    ColumnFilters,
    string,
    boolean
  ]): void {
    if (filtersForColumn?.sortDirection) {
      this.clearPreviousSortFromCurrentFilters(
        columnName,
        filtersForColumn.sortDirection
      );
    }
    this.revertToInitialSort = false;
    this.allCurrentFilters[columnName] = { ...filtersForColumn };

    const updatedFilteredData: T[] = Object.entries(
      this.allCurrentFilters
    ).reduce((acc: T[], [columnName, filters]) => {
      if (filters?.checkboxFilters) {
        acc = this.applyCheckboxFilters(
          filters.checkboxFilters,
          columnName,
          acc
        );
      }
      if (filters?.timeFilters) {
        acc = this.applyDateTimeFilters(filters.timeFilters, columnName, acc);
      }
      if (filters?.sortDirection) {
        acc = this.sortTableByColumn(columnName, filters.sortDirection, acc);
      }
      if (filters?.searchValue) {
        acc = this.applySearchFilters(columnName, filters.searchValue, acc);
      }
      if (clearCols) {
        if (columnName === this.initialSortBy && this.initialSortDirection) {
          this.sortTableByColumn(columnName, this.initialSortDirection, acc);
        }
      }
      return acc;
    }, this.originalData);
    if (this.shouldRevertToOriginalSort()) {
      this.revertToOriginalSort();
    }

    if (clearCols) {
      this.perColumnData = this.buildPerColumnData(updatedFilteredData);
    }

    this.itemCount = updatedFilteredData.length;
    this.filteredData = updatedFilteredData;
    if (filtersForColumn?.checkboxFilters) {
      let selectedFilterData = this.perColumnData[columnName];
      let dataForCheckboxFilter = this.buildPerColumnData(this.filteredData);
      this.perColumnData = dataForCheckboxFilter;
      this.perColumnData[columnName] = selectedFilterData;
    }
    this.currentPage = 1;
    this.currentlyDisplayedData =
      this.calculateCurrentlyDisplayedData(updatedFilteredData);
    this.updateApplyClicked.emit([
      columnName,
      filtersForColumn.searchValue,
      filterAction,
    ]);
    //BRMS-2142
    this.emitFilteredData(this.itemCount, this.filteredData);
  }

  emitFilteredData(itemcount, fdata) {
    this.onFilterChange.emit({
      filteredDataLength: itemcount,
      filteredData: fdata,
    });
    this.getItemCount.emit(itemcount);
  }

  /**
   *
   * @param column column name where sorting has been applied
   * @param direction sort direction - ascending or descending
   * @param data column data to be sorted
   * @returns sorted data of the column by the direction selected
   */
  sortTableByColumn(column, direction: TableSortDirection, data: T[]): T[] {
    this.sortBy = column;
    this.sortDirection = TableSortDirection[direction];
    return data.sort((a, b) => {
      let rowA =
        typeof a[column] === "string"
          ? a[column].toLowerCase().trim()
          : a[column];
      let rowB =
        typeof b[column] === "string"
          ? b[column].toLowerCase().trim()
          : b[column];
      if (!rowA) {
        rowA = "";
      }
      if (!rowB) {
        rowB = "";
      }
      let dateFlag: boolean = false;

      var dateReg = /^\d{2}([./-])\d{2}\1\d{4}$/;
      if (rowA.toString().match(dateReg) && rowB.toString().match(dateReg)) {
        dateFlag = true;
      } else {
        dateFlag = false;
      }
      return this.sortComparison(rowA, rowB, direction, dateFlag);
    });
  }

  /**
   *
   * @param rowA first row
   * @param rowB second row
   * @param direction direction of sorting
   * @param dateFlag if the data is in date format
   * @returns sorted column data especially for date formats
   */
  private sortComparison(
    rowA: T,
    rowB: T,
    direction: TableSortDirection,
    dateFlag?: boolean
  ): number {
    if (TableSortDirection[direction] === TableSortDirection["asc"]) {
      if (dateFlag) {
        let date1 = new Date(rowA.toString());
        let date2 = new Date(rowB.toString());
        if (date1 < date2) return -1;
        if (date1 > date2) return 1;
        return 0;
      } else {
        if (rowA < rowB) return -1;
        if (rowA > rowB) return 1;
        return 0;
      }
    }
    if (TableSortDirection[direction] === TableSortDirection["desc"]) {
      if (dateFlag) {
        let date1 = new Date(rowA.toString());
        let date2 = new Date(rowB.toString());
        if (date1 > date2) return -1;
        if (date1 < date2) return 1;
        return 0;
      } else {
        if (rowA > rowB) return -1;
        if (rowA < rowB) return 1;
        return 0;
      }
    }
  }

  /**
   * function included to filter the search values when showSearch value is true in table-column
   * @param columnName column where filters applied
   * @param searchVal search value entered
   * @param data data in the selected
   * @returns if the search value matches the data then returns true or false
   */
  applySearchFilters(columnName: string, searchVal: string, data: T[]): T[] {
    return data.filter((row) => {
      let match = false;
      if (
        _.get(row, columnName, "")
          .toString()
          .toLowerCase()
          .includes(searchVal.toLowerCase())
      ) {
        match = true;
      }
      return match;
    });
  }

  /**
   *
   * @param checkboxFilters data list of checkbox filters
   * @param columnName column name where checkbox filters have been applied
   * @param data data of the column
   * @returns if selected checkbox filter matches with the data then returns true
   */
  applyCheckboxFilters(
    checkboxFilters: { [key: string]: string }[],
    columnName: string,
    data: T[]
  ): T[] {
    return data.filter((row) => {
      let match = false;
      checkboxFilters.forEach((filter) => {
        if (
          _.get(row, columnName, "").toString().toLowerCase() ===
            filter?.label?.toString().toLowerCase() ||
          (filter &&
            !filter.label &&
            _.get(row, columnName, "").toString().toLowerCase() === "")
        ) {
          match = true;
        }
      });
      return match;
    });
  }

  /**
   *
   * @param timeFilters date range or hours selected in the filters
   * @param columnName column name where filters have been applied
   * @param data data in the column
   * @returns filtered data
   */
  applyDateTimeFilters(
    timeFilters: TimeFilters,
    columnName: string,
    data: T[]
  ): T[] {
    let filteredData = data;
    const dateRange = timeFilters?.dateRange;
    const selectedHours = timeFilters?.selectedHours;

    filteredData = filteredData.filter((rowData) => {
      let match = true;
      let dateMatch = true;
      const regex = /(^\d{4}-\d{2}-\d{2})\s(\d{2})\:(\d{2})\:(\d{2})\s/;
      const regexResults = regex.exec(rowData[columnName]);
      if (dateRange) {
        const lowerBound = dateRange[0];
        const upperBound = dateRange[1];
        const rowDate = regexResults?.[1];
        if (upperBound) {
          match = this.checkIfDateInRange(lowerBound, upperBound, rowDate);
          dateMatch = match;
        } else {
          match = this.checkIfDatesAreEqual(lowerBound, rowDate);
          dateMatch = match;
        }
      }
      if (selectedHours) {
        match = this.checkIfTimeInRange(selectedHours, regexResults[2]);
      }
      return match && dateMatch;
    });
    return filteredData;
  }

  /**
   * when date range is selected
   * checks for filter match in each row
   * @param lowerBound lower bound date
   * @param upperBound upper bound date
   * @param rowDate date in the row
   * @returns true if the date in the row falls within the date range selected
   */
  private checkIfDateInRange(
    lowerBound: string,
    upperBound: string,
    rowDate: string
  ): boolean {
    const rowDateObject = new Date(rowDate).setHours(0, 0, 0, 0);
    const lowerBoundDate = new Date(lowerBound).setHours(0, 0, 0, 0);
    const upperBoundDate = new Date(upperBound).setHours(0, 0, 0, 0);
    return rowDateObject >= lowerBoundDate && rowDateObject <= upperBoundDate;
  }

  /**
   * when only one date is selected
   * @param lowerBound lower bound date
   * @param rowDate date in the row which is getting checked
   * @returns true if date in the row is equals to the lower bound date
   */
  private checkIfDatesAreEqual(lowerBound: string, rowDate: string): boolean {
    return lowerBound == rowDate;
  }

  /**
   *
   * @param selectedHours hours selected in the filters
   * @param hour if the date contains hours then fetching that hour
   * @returns true if selected hours contains hour of the row data
   */
  private checkIfTimeInRange(selectedHours, hour) {
    return selectedHours.toString() === "24" && hour.toString() === "00"
      ? true
      : selectedHours.includes(hour.toString());
  }

  /**
   *
   * @param columnName name of the column
   * @param sortDirection selected direction of sorting
   */
  clearPreviousSortFromCurrentFilters(columnName, sortDirection) {
    if (this.newSortDiffersFromPreviousSort(columnName, sortDirection)) {
      this.allCurrentFilters[this.sortBy] =
        this.getAllCurrentFiltersWithOldSortRemoved();
    }
  }

  /**
   *
   * @returns column with original data (sorting removed)
   */
  private getAllCurrentFiltersWithOldSortRemoved(): ColumnFilters {
    const { sortDirection, ...filtersWithSortRemoved } =
      this.allCurrentFilters[this.sortBy];
    return filtersWithSortRemoved;
  }

  /**
   *
   * @param newColumnName : name of the column
   * @param newSortDirection : sort direction selected
   * @returns true if the sorting is already selected
   */
  private newSortDiffersFromPreviousSort(
    newColumnName: string,
    newSortDirection: TableSortDirection
  ): boolean {
    if (newColumnName !== this.sortBy) {
      return true;
    } else if (
      newColumnName === this.sortBy &&
      TableSortDirection[this.sortDirection] !==
        TableSortDirection[newSortDirection]
    ) {
      return true;
    }
    return false;
  }

  /**
   *
   * @returns the opposite sort direction if sorting has been applied already
   */
  private shouldRevertToOriginalSort(): boolean {
    const sortDirection = _.get(
      this.allCurrentFilters,
      `${this.sortBy}.sortDirection`,
      null
    );
    return !sortDirection;
  }

  /**
   * revert to the initial sort direction before filters were applied
   */
  private revertToOriginalSort(): void {
    this.sortBy = this.initialSortBy;
    this.sortDirection = TableSortDirection[this.initialSortDirection];
    _.set(
      this.allCurrentFilters,
      `${this.initialSortBy}.sortDirection`,
      TableSortDirection[this.initialSortDirection]
    );
    this.revertToInitialSort = true;
  }

  /**
   * since we are getting all the data at once we are using this function to compute page numbers
   * based on the page size selected.
   */
  private computePageNumbers(): void {
    this.selectedPage = this.currentPage || 1;
    this.maxPages = Math.ceil(this.itemCount / this.currentPageSize);
    if (this.selectedPage > this.maxPages) {
      this.selectedPage = this.maxPages;
    }
    this.maxPages = Number.isNaN(this.maxPages) ? 0 : this.maxPages;
    this.pageNumbers = Array(this.maxPages)
      .fill(0)
      .map((x, i) => i + 1);
    if (this.selectedPage > 3) {
      this.prevPagesIndex =
        this.pageNumbers.findIndex((element) => element == this.selectedPage) -
          2 >
        0
          ? this.pageNumbers.findIndex(
              (element) => element == this.selectedPage
            ) - 2
          : 0;
      this.nextPagesIndex =
        this.pageNumbers.findIndex((element) => element == this.selectedPage) +
          2 <=
        this.maxPages
          ? this.pageNumbers.findIndex(
              (element) => element == this.selectedPage
            ) + 2
          : this.maxPages;
    } else {
      this.prevPagesIndex = 0;
      this.nextPagesIndex = 4;
    }
  }

  previousPage(): void {
    this.handlePageChange(this.selectedPage - 1, this.currentPageSize);
  }

  nextPage(): void {
    this.handlePageChange(this.selectedPage + 1, this.currentPageSize);
  }

  handlePageChange(pageNumber: number, pageSize: number): void {
    this.currentPage = pageNumber;
    this.currentPageSize = pageSize;
    this.computePageNumbers();
    this.emitPageChangeEvent();
    this.currentlyDisplayedData = this.calculateCurrentlyDisplayedData(
      this.filteredData
    );
  }
  /**
   * Gets triggered when ever there is a page change
   */
  emitPageChangeEvent() {
    this.onPageChange.emit({
      page: this.selectedPage,
      pageSize: this.selectedOption,
      sortBy: this.sortBy,
      sortDirection: TableSortDirection[this.sortDirection],
    });
  }

  /**
   * get selected page size when changed
   */
  private updateSelectedPageSizeOption(): void {
    if (this.currentPageSize) {
      this.selectedOption = this.paginationOptions.find(
        (item) => item.value == this.currentPageSize
      ).value;
    } else {
      this.selectedOption = this.paginationOptions[0].value;
      this.currentPageSize = this.selectedOption;
    }
  }

  public isAllSelected(): boolean {
    const numSelected = this.selection.selected.length;
    const numRows = this.originalData.length;
    this.onSelectChange.emit(this.selection.selected);
    return numSelected === numRows;
  }

  public masterToggle(): void {
    this.isAllSelected()
      ? this.selection.clear()
      : this.filteredData.forEach((row) => this.selection.select(row));
  }

  public checkboxLabel(row?: any): string {
    return !row
      ? `${this.isAllSelected() ? "select" : "deselect"} all`
      : `${this.selection.isSelected(row) ? "deselect" : "select"} row ${
          this.filteredData.indexOf(row) + 1
        }`;
  }

  public toggleRow(event, row): void {
    let rowToToggle;
    const checkedRow = this.selection.selected.find(
      (rowInSelection) => rowInSelection._id === row._id
    );
    if (checkedRow?.id === row.id) {
      rowToToggle = checkedRow;
    } else {
      rowToToggle = row;
    }
    event ? this.selection.toggle(rowToToggle) : null;
  }

  public isRowChecked(row): boolean {
    return this.selection.selected.find((selectedRow, index) => {
      const isMatch = selectedRow[this.checkboxKey] === row[this.checkboxKey];
      if (isMatch && selectedRow._id !== row._id) {
        this.selection.selected[index]._id = row._id;
      }
      return isMatch;
    });
  }

  moreInfoItemClicked(event, index) {
    if (this.optionShouldTriggerEdits(event) && !this.editing.includes(true)) {
      this.editing[index] = !this.editing[index];
      this.editFormGroup = this.fb.group(this.currentlyDisplayedData[index]);
    }
    this.moreInfoClicked.emit(event);
  }

  optionShouldTriggerEdits(event): boolean {
    return (
      event.label === TableEditOption.update ||
      event.label === TableEditOption.manage
    );
  }

  /**
   * @param element more info element
   * check if it is in expanded or collapsed state
   */
  isExpanded(element: T) {
    let result = false;
    if (this.expandedRows) {
      this.expandedRows.forEach((row) => {
        if (row === element) {
          result = true;
          return result;
        }
      });
    }
    return result;
  }

  /**
   *
   * @param rowClicked row where expand is clicked
   */
  expandClicked(rowClicked: T) {
    if (this.expandedRows.indexOf(rowClicked) === -1) {
      this.expandedRows.push(rowClicked);
    }
  }

  /**
   *
   * @param rowClicked row where collapse is clicked
   */
  collapseClicked(rowClicked: T) {
    let index = this.expandedRows.indexOf(rowClicked);
    if (index !== -1) {
      this.expandedRows.splice(index, 1);
    }
  }

  /**
   *
   * @param index row where confirm is clicked
   */
  confirmClicked(event, index) {
    //do not confirm if errors
    //condition to check if any specific pattern based error is present
    if (
      !this.editFormGroup?.controls[this.emptyInput]?.errors?.required &&
      !this.error[index]
    ) {
      this.updateConfirmed.emit(this.editFormGroup.value);
      this.currentlyDisplayedData[index] = this.editFormGroup.value;
      this.editing[index] = false;
    }
  }
  /**
   * emits value and row index on input focused out
   * @param event value entered in input
   * @param index index of the row where value gets added
   */
  focusedOut(event, index) {
    this.validateInputEvent.emit({ value: event.value, index: index });
  }
  cancelClicked(event, index) {
    this.editing[index] = false;
  }

  /**
   * emits item and row info on click
   * @param itemClicked item that has been clicked
   * @param rowIndex index of the row where item gets clicked
   */
  handleRowClick(itemClicked: T, rowIndex: number): void {
    this.onRowClick.emit({
      item: itemClicked,
      index: rowIndex,
    });
  }
}

export enum TableEditOption {
  update = "Update",
  delete = "Delete",
  cofirm = "Confirm",
  cancel = "Cancel",
  manage = "Manage",
}

export interface ColumnFilters {
  sortDirection?: TableSortDirection;
  searchFilters?: { [key: string]: string }[];
  checkboxFilters?: { [key: string]: string }[];
  timeFilters?: TimeFilters;
  searchValue?: string;
  searchAutoSuggestValues?: { label: string; checked: boolean }[];
}

export interface InitialColumnFilters {
  columnFilter?: ColumnFilters;
  columnName?: string;
}

export interface TimeFilters {
  dateRange?: string[];
  selectedHours?: string[];
}
