import {
  Component,
  OnInit,
  ContentChildren,
  EventEmitter,
  Input,
  Output,
  QueryList,
  OnChanges,
  SimpleChanges,
  ViewChildren,
} from "@angular/core";
import { SelectionModel } from "@angular/cdk/collections";
import { SelectItem } from "primeng/api";
import { v4 as uuid } from "uuid";
import * as _ from "lodash";
import { FilterOption } from "../filter-group/filter.interface";
import {
  TableColumnDirective,
  TableRowClickEvent,
  TableSortDirection,
  PageSizes,
} from "../table/table.component";
import { ColumnFilterDropdownComponent } from "../column-filter-dropdown/column-filter-dropdown.component";
import { FormBuilder, FormGroup } from "@angular/forms";
import { TablePageChangeEvent } from "../table/table.component";
import { of } from "rxjs";

@Component({
  selector: "bre-filter-table-update",
  templateUrl: "./filter-table-update.component.html",
  styleUrls: ["./filter-table-update.component.scss"],
})
export class FilterTableUpdateComponent<T> implements OnInit, OnChanges {
  @Input() originalData: T[] = [];

  @Input() itemCount: number;
  @Input() showLoadingIndicator: boolean;

  @Input() currentPage = 1;
  @Input() currentPageSize: number = PageSizes.TwentyFive;
  @Input() sortBy: string;
  @Input() sortDirection: TableSortDirection;
  @Input() enableCheckBox: boolean;
  @Input() clearSelection: boolean;
  @Input() checkboxKey: string;
  @Input() editable: boolean = false;
  @Input() expandable: boolean = false;
  @Input() showSelect: boolean = true;
  @Input() selectedColumnName: string;
  @Input() prefilteredColumnName: string;
  @Input() prefilledFilters: ColumnFilters;
  @Input() filterLabel = "Filter";
  @Input() configReset = true;
  @Input() configEnum: boolean = false;
  @Input() configPageSizes: any;
  @Input() showCountMessage: string = "";
  @Input() initialColumnFilters: [InitialColumnFilters];
  @Input() emptyInput: string;
  selection: SelectionModel<any>;

  @Output() onRowClick: EventEmitter<TableRowClickEvent<T>> =
    new EventEmitter();
  @Output() sortColumn: EventEmitter<any> = new EventEmitter();
  @Output() onSelectChange: EventEmitter<any> = new EventEmitter();
  @Output() onFilterChange: EventEmitter<any> = new EventEmitter();
  @Output() filtersCleared: EventEmitter<boolean> = new EventEmitter();
  @Output() moreInfoClicked: EventEmitter<any> = new EventEmitter();
  @Output() updateConfirmed: EventEmitter<any> = new EventEmitter();
  @Output() onPageChange: EventEmitter<TablePageChangeEvent> =
    new EventEmitter();
  @Output() updateApplyClicked: EventEmitter<any> = new EventEmitter();
  @Output() getItemCount: EventEmitter<any> = new EventEmitter();
  @Output() autoSuggestSearchInputChanged: EventEmitter<any> =
    new EventEmitter();
  @ContentChildren(TableColumnDirective)
  columns: QueryList<TableColumnDirective>;
  @ViewChildren(ColumnFilterDropdownComponent)
  filterDropdowns!: QueryList<ColumnFilterDropdownComponent>;

  paginationOptions: SelectItem[];
  selectedOption: number;
  selectedPage: number;
  maxPages: number;
  pageNumbers: number[];
  prevPagesIndex: number;
  nextPagesIndex: number;
  labelUUID: string;
  initialSortBy: string;
  initialSortDirection: TableSortDirection;
  allCurrentFilters: { [columnName: string]: ColumnFiltersUpdate } = {};
  perColumnData: { [columnName: string]: string[] } = {};
  filteredData: T[] = null;
  currentlyDisplayedData: T[] = [];
  editing: boolean[];
  moreInfoHidden: boolean[] = [];
  expandedRows: T[] = [];
  visibleColumnsLength: number;
  editFormGroup: FormGroup;
  shouldClearFilters = false;
  initialPageSize: number;
  revertToInitialSort: boolean = false;
  //animation
  animationHeight = "180px";
  animationWidth = "180px";
  currentFilterType: any;
  clearAllFilters: boolean;
  constructor(private fb: FormBuilder) {
    this.labelUUID = uuid();
  }

  ngOnInit() {
    const pageSizesArray = Object.values(
      this.configEnum ? this.configPageSizes : PageSizes
    ).filter((value) => isNaN(Number(value)) === false);
    this.paginationOptions = pageSizesArray.map((x) => ({
      label: x.toString(),
      value: x,
    }));
    this.initialPageSize = this.configEnum
      ? +Object.keys(this.configPageSizes)[0]
      : PageSizes.TwentyFive;
    this.selection = new SelectionModel<any>(true, []);

    if (this.filteredData && !this.itemCount) {
      this.itemCount = this.filteredData.length;
    }
    this.updateSelectedPageSizeOption();

    if (!this.sortColumn.observers.length) {
      this.computePageNumbers();
    }

    if (this.enableCheckBox) {
      this.selection.isSelected = this.isRowChecked.bind(this);
    }

    this.initialSortBy = this.sortBy;

    this.initialSortDirection = this.sortDirection;

    if (this.prefilteredColumnName && this.prefilledFilters) {
      this.onFilterUpdate([
        this.prefilteredColumnName,
        this.prefilledFilters,
        "",
        false,
      ]);
      this.selection.clear();
    }

    //BRMS-2430 This loop iterates through all the multiple initial filters upon load. Added this because prefilledFilters allows for one initial filter
    if (this.initialColumnFilters) {
      for (let i = 0; i < this.initialColumnFilters.length; i++) {
        this.onFilterUpdate([
          this.initialColumnFilters[i].columnName,
          this.initialColumnFilters[i].columnFilter,
          "",
          false,
        ]);
        this.selection.clear();
      }
    }
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes.currentPageSize && this.paginationOptions) {
      this.updateSelectedPageSizeOption();
    }
    if (changes.itemCount || changes.currentPageSize || changes.currentPage) {
      this.computePageNumbers();
    }
    if (this.enableCheckBox && changes.clearSelection) {
      if (this.selection) {
        this.selection.clear();
      }
    }

    if (
      (!this.filteredData && this.originalData && this.originalData.length) ||
      changes.originalData
    ) {
      this.initializeData(this.originalData);
    }
  }

  private initializeData(originalData): void {
    this.perColumnData = this.buildPerColumnData(this.originalData);
    this.filteredData = this.originalData;
    this.computePageNumbers();
    this.currentlyDisplayedData = this.originalData;
  }

  resetEditableArray(length) {
    this.editing = new Array();
    for (let i = 0; i < length; i++) {
      this.editing.push(false);
    }
  }

  clearFilters() {
    this.clearAllFilters = true;
    if (this.filterDropdowns) {
      this.filterDropdowns.forEach((dropdown) => dropdown.clearFilters());
    }
    this.filtersCleared.emit();
    this.clearAllFilters = false;
  }

  isHiddenDataNull(rowData: T[]) {
    this.moreInfoHidden = new Array();
    let isHiddenData = true;
    rowData.forEach((row) => {
      isHiddenData = true;
      this.columns
        .filter((column) => column.hidden)
        .forEach((column) => {
          if (row[column.name] && row[column.name] !== "null - null") {
            isHiddenData = false;
          }
        });
      if (isHiddenData) {
        this.moreInfoHidden.push(true);
      } else {
        this.moreInfoHidden.push(false);
      }
    });
    return isHiddenData;
  }

  getVisibleColumnsNames(): string[] {
    let result = this.columns.filter((x) => !x.hidden).map((x) => x.name);
    if (result.length !== this.visibleColumnsLength) {
      this.visibleColumnsLength = result.length;
    }
    return result;
  }

  buildPerColumnData(data: T[]): { [columnName: string]: string[] } {
    return data.reduce((keyedByColumn, row) => {
      Object.entries(row).forEach(([columnLabel, columnValue]) => {
        if (_.get(keyedByColumn, columnLabel)) {
          keyedByColumn[columnLabel].push({
            label: columnValue,
            value: columnValue,
          } as FilterOption);
        } else {
          _.set(keyedByColumn, columnLabel, [
            { label: columnValue, value: columnValue } as FilterOption,
          ]);
        }
      });
      return keyedByColumn;
    }, {});
  }

  onFilterUpdate([columnName, filtersForColumn, filterAction, clearCols]: [
    string,
    ColumnFilters,
    string,
    boolean
  ]): void {
    let unappliedFilters = [];
    if (filtersForColumn?.sortDirection) {
      this.clearPreviousSortFromCurrentFilters(
        columnName,
        filtersForColumn.sortDirection
      );
    }
    this.revertToInitialSort = false;
    this.allCurrentFilters[columnName] = { ...filtersForColumn };

    if (filtersForColumn?.timeFilters) {
      unappliedFilters.push({
        columnName: columnName,
        items: filtersForColumn.timeFilters,
        filterAction: filterAction,
        type: "timeFilters",
        clearCols: clearCols,
      });
    }
    if (filtersForColumn?.sortDirection) {
      this.sortBy = columnName;
      this.sortDirection = TableSortDirection[filtersForColumn.sortDirection];
      unappliedFilters.push({
        columnName: columnName,
        items: filtersForColumn.sortDirection,
        filterAction: filterAction,
        type: "sortDirection",
        clearCols: clearCols,
      });
    }

    this.currentPage = 1;
    if (filtersForColumn?.searchAutoSuggestValues) {
      const sugestionValues = filtersForColumn.searchAutoSuggestValues
        .filter((item) => item.checked)
        .map((item) => item.label);
      unappliedFilters.push({
        columnName: columnName,
        items: sugestionValues,
        filterAction: filterAction,
        type: "autoSuggest",
        clearCols: clearCols,
      });
    }
    if (filtersForColumn?.checkboxFilters) {
      unappliedFilters.push({
        columnName: columnName,
        items: filtersForColumn.checkboxFilters,
        filterAction: filterAction,
        type: "checkboxFilters",
        clearCols: clearCols,
      });
    }
    if (unappliedFilters.length) {
      this.updateApplyClicked.emit(unappliedFilters);
    } else if (clearCols && !this.clearAllFilters) {
      this.updateApplyClicked.emit([
        {
          columnName: columnName,
          clearCols: clearCols,
          type: this.currentFilterType,
        },
      ]);
    }

    //BRMS-2142
    this.onFilterChange.emit({
      filteredDataLength: this.itemCount,
      filteredData: this.filteredData,
    });
    this.getItemCount.emit(this.itemCount);
  }

  clearPreviousSortFromCurrentFilters(columnName, sortDirection) {
    if (this.newSortDiffersFromPreviousSort(columnName, sortDirection)) {
      this.allCurrentFilters[this.sortBy] =
        this.getAllCurrentFiltersWithOldSortRemoved();
    }
  }

  private getAllCurrentFiltersWithOldSortRemoved(): ColumnFilters {
    const { sortDirection, ...filtersWithSortRemoved } =
      this.allCurrentFilters[this.sortBy];
    return filtersWithSortRemoved;
  }

  private newSortDiffersFromPreviousSort(
    newColumnName: string,
    newSortDirection: TableSortDirection
  ): boolean {
    if (newColumnName !== this.sortBy) {
      return true;
    } else if (
      newColumnName === this.sortBy &&
      TableSortDirection[this.sortDirection] !==
        TableSortDirection[newSortDirection]
    ) {
      return true;
    }
    return false;
  }

  private computePageNumbers(): void {
    this.selectedPage = this.currentPage || 1;
    this.maxPages = Math.ceil(this.itemCount / this.currentPageSize);
    if (this.selectedPage > this.maxPages) {
      this.selectedPage = this.maxPages;
    }
    this.maxPages = Number.isNaN(this.maxPages) ? 0 : this.maxPages;
    this.pageNumbers = Array(this.maxPages)
      .fill(0)
      .map((x, i) => i + 1);
    if (this.selectedPage > 3) {
      this.prevPagesIndex =
        this.pageNumbers.findIndex((element) => element == this.selectedPage) -
          2 >
        0
          ? this.pageNumbers.findIndex(
              (element) => element == this.selectedPage
            ) - 2
          : 0;
      this.nextPagesIndex =
        this.pageNumbers.findIndex((element) => element == this.selectedPage) +
          2 <=
        this.maxPages
          ? this.pageNumbers.findIndex(
              (element) => element == this.selectedPage
            ) + 2
          : this.maxPages;
    } else {
      this.prevPagesIndex = 0;
      this.nextPagesIndex = 4;
    }
  }

  previousPage(): void {
    this.handlePageChange(this.selectedPage - 1, this.currentPageSize);
  }

  nextPage(): void {
    this.handlePageChange(this.selectedPage + 1, this.currentPageSize);
  }

  handlePageChange(pageNumber: number, pageSize: number): void {
    this.currentPage = pageNumber;
    this.currentPageSize = pageSize;
    this.computePageNumbers();
    this.emitPageChangeEvent();
    this.currentlyDisplayedData = this.originalData;
  }
  /**
   * Gets triggered when ever there is a page change
   */
  emitPageChangeEvent() {
    this.onPageChange.emit({
      page: this.selectedPage,
      pageSize: this.selectedOption,
      sortBy: this.sortBy,
      sortDirection: TableSortDirection[this.sortDirection],
    });
  }

  private updateSelectedPageSizeOption(): void {
    if (this.currentPageSize) {
      this.selectedOption = this.paginationOptions.find(
        (item) => item.value == this.currentPageSize
      ).value;
    } else {
      this.selectedOption = this.paginationOptions[0].value;
      this.currentPageSize = this.selectedOption;
    }
  }

  public isAllSelected(): boolean {
    const numSelected = this.selection.selected.length;
    const numRows = this.originalData.length;
    this.onSelectChange.emit(this.selection.selected);
    return numSelected === numRows;
  }

  public masterToggle(): void {
    this.isAllSelected()
      ? this.selection.clear()
      : this.filteredData.forEach((row) => this.selection.select(row));
  }

  public checkboxLabel(row?: any): string {
    return !row
      ? `${this.isAllSelected() ? "select" : "deselect"} all`
      : `${this.selection.isSelected(row) ? "deselect" : "select"} row ${
          this.filteredData.indexOf(row) + 1
        }`;
  }

  public toggleRow(event, row): void {
    let rowToToggle;
    const checkedRow = this.selection.selected.find(
      (rowInSelection) => rowInSelection._id === row._id
    );
    if (checkedRow?.id === row.id) {
      rowToToggle = checkedRow;
    } else {
      rowToToggle = row;
    }
    event ? this.selection.toggle(rowToToggle) : null;
  }

  public isRowChecked(row): boolean {
    return this.selection.selected.find((selectedRow, index) => {
      const isMatch = selectedRow[this.checkboxKey] === row[this.checkboxKey];
      if (isMatch && selectedRow._id !== row._id) {
        this.selection.selected[index]._id = row._id;
      }
      return isMatch;
    });
  }

  moreInfoItemClicked(event, index) {
    if (this.optionShouldTriggerEdits(event) && !this.editing.includes(true)) {
      this.editing[index] = !this.editing[index];
      this.editFormGroup = this.fb.group(
        this.currentlyDisplayedData[
          index + (this.currentPage - 1) * this.currentPageSize
        ]
      );
    }
    this.moreInfoClicked.emit(event);
  }

  optionShouldTriggerEdits(event): boolean {
    return (
      event.label === TableEditOption.update ||
      event.label === TableEditOption.manage
    );
  }

  whenIsExpanded(i, element) {
    let result = false;
    if (this.expandedRows) {
      this.expandedRows.forEach((row) => {
        if (row === element) {
          result = true;
          return result;
        }
      });
    }
    return result;
  }

  isExpanded(i: number, element: T) {
    let result = false;
    if (this.expandedRows) {
      this.expandedRows.forEach((row) => {
        if (row === element) {
          result = true;
          return result;
        }
      });
    }
    return result;
  }

  expandClicked(rowClicked: T) {
    if (this.expandedRows.indexOf(rowClicked) === -1) {
      this.expandedRows.push(rowClicked);
    }
  }

  collapseClicked(rowClicked: T) {
    let index = this.expandedRows.indexOf(rowClicked);
    if (index !== -1) {
      this.expandedRows.splice(index, 1);
    }
  }

  confirmClicked(event, index) {
    //do not confirm if errors
    if (!this.editFormGroup?.controls[this.emptyInput]?.errors?.required) {
      this.updateConfirmed.emit(this.editFormGroup.value);
      this.currentlyDisplayedData[
        index + (this.currentPage - 1) * this.currentPageSize
      ] = this.editFormGroup.value;
      this.editing[index] = false;
    }
  }

  cancelClicked(event, index) {
    this.editing[index] = false;
  }

  handleRowClick(itemClicked: T, rowIndex: number): void {
    this.onRowClick.emit({
      item: itemClicked,
      index: rowIndex,
    });
  }

  onAutoSuggestInputChange(event) {
    this.autoSuggestSearchInputChanged.emit(event);
  }

  setCurrentColumnType(event) {
    this.currentFilterType = event.filterType;
  }
}

export enum TableEditOption {
  update = "Update",
  delete = "Delete",
  cofirm = "Confirm",
  cancel = "Cancel",
  manage = "Manage",
}

export interface ColumnFilters {
  filterType?: string;
  sortDirection?: TableSortDirection;
  searchFilters?: { [key: string]: string }[];
  checkboxFilters?: { [key: string]: string }[];
  timeFilters?: TimeFilters;
  searchValue?: string;
  searchAutoSuggestValues?: { label: string; checked: boolean }[];
}

export interface ColumnFiltersUpdate {
  checkboxFilters?: { [key: string]: string }[];
  sortDirection?: TableSortDirection;
  timeFilters?: TimeFilters;
  searchAutoSuggestValues?: { label: string; checked: boolean }[];
  hideFilterIcon?: boolean;
}

export interface InitialColumnFilters {
  columnFilter?: ColumnFilters;
  columnName?: string;
}

export interface TimeFilters {
  dateRange?: string[];
  selectedHours?: string[];
}
