import { CdkTableModule } from '@angular/cdk/table';
import { async, ComponentFixture, TestBed } from '@angular/core/testing';
import { FormsModule, ReactiveFormsModule, FormGroup, FormControl } from '@angular/forms';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { MatMenuModule } from '@angular/material/menu';
import { AutoCompleteModule } from 'primeng/autocomplete';
import { ButtonModule } from 'primeng/button';
import { CalendarModule } from 'primeng/calendar';
import { DropdownModule } from 'primeng/dropdown';
import { InputTextModule } from 'primeng/inputtext';
import { KeyFilterModule } from 'primeng/keyfilter';
import { MultiSelectModule } from 'primeng/multiselect';
import { OverlayPanelModule } from 'primeng/overlaypanel';
import { ProgressSpinnerModule } from 'primeng/progressspinner';
import { AutoCompleteComponent } from '../auto-complete/auto-complete.component';
import { ButtonComponent } from '../button/button.component';
import { CalendarComponent } from '../calendar/calendar.component';
import { ColumnCheckboxFilterComponent } from '../column-checkbox-filter/column-checkbox-filter.component';
import { ColumnDateFilterComponent } from '../column-date-filter/column-date-filter.component';
import { ColumnFilterDropdownComponent } from '../column-filter-dropdown/column-filter-dropdown.component';
import { ColumnSearchFilterComponent } from '../column-search-filter/column-search-filter.component';
import { ColumnSortFilterComponent } from '../column-sort-filter/column-sort-filter.component';
import { ControlLabelComponent } from '../control-label/control-label.component';
import { IconButtonComponent } from '../icon-button/icon-button.component';
import { InputErrorMessageComponent } from '../input-error-message/input-error-message.component';
import { InputComponent } from '../input/input.component';
import { MultiSelectComponent } from '../multi-select/multi-select.component';
import { PopoverMenuButtonComponent } from '../popover-menu-button/popover-menu-button.component';
import { StatusPillComponent } from '../status-pill/status-pill.component';
import { TableRowClickEvent, TablePageChangeEvent, TableComponent, TableColumnDirective, TableSortDirection } from '../table/table.component';
import { TooltipDirective } from '../tooltip/tooltip.directive';
import { ErrorIconComponent } from '../error-icon/error-icon.component';

import { FilterTableUpdateComponent, ColumnFilters, TableEditOption } from './filter-table-update.component';

describe('FilterTableComponent', () => {
  let component: FilterTableUpdateComponent<any>;
  let fixture: ComponentFixture<FilterTableUpdateComponent<any>>;
  let dataByColumn = {
    "column1": [
      {
        "label": "row1 item1",
        "value": "row1 item1",
      },
      {
        "label": "row2 item3",
        "value": "row2 item3",
      },
      {
        "label": "row3 item2",
        "value": "row3 item2",
      },
    ],
    "column2": [
      {
        "label": "row1 item2",
        "value": "row1 item2",
      },
      {
        "label": "row2 item1",
        "value": "row2 item1",
      },
      {
        "label": "row3 item3",
        "value": "row3 item3",
      },
    ],
    "column3": [
      {
        "label": "row1 item3",
        "value": "row1 item3",
      },
      {
        "label": "row2 item2",
        "value": "row2 item2",
      },
      {
        "label": "row3 item1",
        "value": "row3 item1",
      },
    ],
  };

  beforeEach(async(() => {
    TestBed.configureTestingModule({
      imports: [
        FormsModule,
        CdkTableModule,
        DropdownModule,
        ButtonModule,
        MatCheckboxModule,
        OverlayPanelModule,
        ButtonModule,
        ProgressSpinnerModule,
        ReactiveFormsModule,
        MultiSelectModule,
        MatMenuModule,
        AutoCompleteModule,
        KeyFilterModule,
        InputTextModule,
        CalendarModule
      ],
      declarations: [
        TableComponent,
        TableColumnDirective,
        ColumnFilterDropdownComponent,
        IconButtonComponent,
        ButtonComponent,
        ColumnSortFilterComponent,
        MultiSelectComponent,
        ControlLabelComponent,
        InputErrorMessageComponent,
        TooltipDirective,
        PopoverMenuButtonComponent,
        AutoCompleteComponent,
        ColumnSearchFilterComponent,
        ColumnDateFilterComponent,
        InputComponent,
        CalendarComponent,
        ColumnCheckboxFilterComponent,
        StatusPillComponent,
        FilterTableUpdateComponent,
        ErrorIconComponent
      ]
    })
      .compileComponents();
  }));

  const originalData = [
    {column1: 'row1 item1', column2: 'row1 item2', column3: 'row1 item3'},
    {column1: 'row2 item3', column2: 'row2 item1', column3: 'row2 item2'},
    {column1: 'row3 item2', column2: 'row3 item3', column3: 'row3 item1'},
  ];

  beforeEach(() => {
    fixture = TestBed.createComponent(FilterTableUpdateComponent);
    component = fixture.componentInstance;
    component.originalData = originalData;
    component.itemCount = originalData.length;
    component.sortBy = 'column1';
    component.sortDirection = TableSortDirection['asc'];
    fixture.detectChanges();
  });

  it('should create', () => {
    expect(component).toBeTruthy();
  });

  it('reorganizes the data into an object keyed by column using buildPerColumnData', () => {
    const expected = component.buildPerColumnData(component.originalData);
    expect(expected).toEqual(dataByColumn);
  });

  it('should handle row clicking correctly with calculateCurrentlyDisplayedData', () => {
    const spy = spyOn(component.onRowClick, 'emit');
    const item = {column1: 'column1'};
    const index = 1;
    component.handleRowClick(item, index)
    expect(spy).toHaveBeenCalledWith({index,item});
  });  

  it('should emit a change event when updating filters for a column', () => {
    const spy = spyOn(component.onFilterChange, 'emit');
    const columnString = 'column3';
    const filters: ColumnFilters = { searchValue: 'item', timeFilters: {selectedHours: undefined, dateRange: ['2021-04-20']}, checkboxFilters: [{label:'row1 item2', value: 'row1 item2'}]};
    component.onFilterUpdate([columnString, filters, 'test',false])
    expect(spy).toHaveBeenCalled();
  });  

  it('should emit an event when the more info button is clicked', () => {
    component.currentlyDisplayedData = originalData;
    component.currentPage = 1;
    component.currentPageSize = 10;
    component.editing = [false];
    const event = { label: TableEditOption.manage };
    const spy = spyOn(component.moreInfoClicked, 'emit');
    component.moreInfoItemClicked(event, 0);
    expect(spy).toHaveBeenCalledWith(event);
  }); 
  it('should emit a page change event when moving to a previous or next page', () => {
    component.filteredData = [];
    const spy = spyOn(component.onPageChange, 'emit');
    component.nextPage();
    expect(spy).toHaveBeenCalled();

    spy.calls.reset();

    component.previousPage();
    expect(spy).toHaveBeenCalled();
  });

  it('should return a list of dates that match the given date filters in applyDateTimeFilters', () => {
    const timeFiltersSingleDate = {selectedHours: undefined, dateRange: ['2021-04-20']};
    const timeFiltersRange = {selectedHours: undefined, dateRange: ['2021-04-20', '2021-04-21']};
    const timeFiltersRangeAndTimes = {selectedHours: ['21'], dateRange: ['2021-04-20', '2021-04-21']};
    const dateData = [
      {column1: 'row1 item1', column2: 'row1 item2', column3: 'row1 item3', column4: '2021-04-19 19:15:43 GMT'},
      {column1: 'row2 item3', column2: 'row2 item1', column3: 'row2 item2', column4: '2021-04-20 20:15:43 GMT'},
      {column1: 'row3 item2', column2: 'row3 item3', column3: 'row3 item1', column4: '2021-04-21 21:15:43 GMT'},
    ];
  });

  it('should emit an update event and turn off editing upon confirmation', () => {
    component.currentlyDisplayedData = originalData;
    component.currentPage = 1;
    component.currentPageSize = 10;
    component.editing = [true];
    component.editFormGroup = new FormGroup({'example': new FormControl(['test'])});
    const event = {};
    const spy = spyOn(component.updateConfirmed, 'emit');
    component.confirmClicked(event, 0);
    expect(spy).toHaveBeenCalledWith(component.editFormGroup.value);
    expect(component.editing[0]).toEqual(false);
  });

  it('should turn off editing upon cancel', () => {
    component.editing = [true];
    const event = {};
    component.cancelClicked(event, 0);
    expect(component.editing[0]).toEqual(false);
  });
});