import { async, ComponentFixture, TestBed } from "@angular/core/testing";
import { ActivatedRoute, Router } from "@angular/router";
import { RouterTestingModule } from "@angular/router/testing";
import { Component } from "@angular/core";
import { FilterGroupComponent } from "./filter-group.component";
import {
  Filter,
  FilterType,
  FilterOption,
  FiltersChangeEvent
} from "../../../public-api";
import { AutoCompleteModule } from "primeng/autocomplete";
import { ButtonModule } from "primeng/button";
import { ButtonComponent } from "../button/button.component";
import { AutoCompleteComponent } from "../auto-complete/auto-complete.component";
import { ProgressSpinnerModule } from "primeng/progressspinner";
import { FormsModule } from "@angular/forms";
import { MultiSelectModule } from "primeng/multiselect";
import { AdvancedFilterComponent } from "./advanced-filter/advanced-filter.component";
import { CalendarComponent } from "../calendar/calendar.component";
import { CalendarModule } from "primeng/calendar";
import { ReactiveFormsModule } from "@angular/forms";
import { ControlLabelComponent } from "../control-label/control-label.component";
import { TooltipDirective } from "../tooltip/tooltip.directive";
import { PopoverMenuButtonComponent } from "../popover-menu-button/popover-menu-button.component";
import { IconButtonComponent } from "../icon-button/icon-button.component";
import { MatMenuModule } from "@angular/material/menu";
import { StatusPillComponent } from "../status-pill/status-pill.component";
import { InputComponent } from "../input/input.component";
import { InputErrorMessageComponent } from "../input-error-message/input-error-message.component";
import { InputTextModule } from "primeng/inputtext";
import { DropdownComponent } from "../dropdown/dropdown.component";
import { DropdownModule } from "primeng/dropdown";
import { KeyFilterModule } from "primeng/keyfilter";
import { TooltipModule } from "primeng/tooltip";
import { ErrorIconComponent } from "../error-icon/error-icon.component";

@Component({
  selector: "bre-mock",
  template: `
    <bre-filter-group
      [filters]="filters"
      [selectedFilters]="selectedFilters"
      (filterDidChange)="filterData($event)"
    >
    </bre-filter-group>
  `
})
class MockComponent {
  public filters: Filter[];
  private autoCompleteOptions: FilterOption[] = [
    { label: "1 - Hamburger", value: "hamburger" },
    { label: "2 - Fries", value: "fries" },
    { label: "3 - Cheeseburger", value: "cheeseburger" },
    { label: "4 - Salt", value: "salt" }
  ];
  public selectedFilters: any;

  constructor(private route: ActivatedRoute, private router: Router) {}

  ngOnInit(): void {
    let initialParams = {};
    this.route.snapshot.queryParamMap.keys.forEach((key) => {
      initialParams = {
        ...initialParams,
        [key]: this.route.snapshot.queryParamMap
          .getAll(key)
          .map((el) => el.toLowerCase())
      };
    });
    this.selectedFilters = initialParams; //temporary, this should select item based on url params
    this.filters = [
      {
        name: "inputPlaceholder",
        type: FilterType.input,
        placeholder: "Search Product Code or Name",
        options: [],
        loadOptions: async (searchTerm: string) =>
          this.loadAutoCompleteOptions(searchTerm),
        isAdvanced: false
      },
      {
        name: "productCodeName",
        type: FilterType.autocomplete,
        placeholder: "Search Product Code or Name",
        options: this.autoCompleteOptions.filter((el) =>
          this.selectedFilters.productCodeName
            ? this.selectedFilters.productCodeName.includes(
                el.value.toString().toLowerCase()
              )
            : ""
        ),
        loadOptions: async (searchTerm: string) =>
          this.loadAutoCompleteOptions(searchTerm),
        isAdvanced: false
      },
      {
        name: "dropdown",
        type: FilterType.dropdown,
        placeholder: "Dropdown",
        options: [
          { label: "Item 1", value: "1" },
          { label: "Item 2", value: "2" }
        ],
        isAdvanced: false
      },
      {
        name: "cellLocation",
        type: FilterType.multiselect,
        placeholder: "Cell Location",
        options: [
          { label: "coffee", value: "coffee" },
          { label: "prepline", value: "prepline" },
          { label: "potato fryer", value: "potatofryer" }
        ],
        isAdvanced: false
      },
      {
        name: "dateRange",
        type: FilterType.calendar,
        placeholder: "Calendar",
        isAdvanced: false
      }
    ];
  }

  loadAutoCompleteOptions(searchTerm: string): void {
    this.filters[0].options = this.autoCompleteOptions.filter((x) =>
      x.value
        .toString()
        .toLowerCase()
        .includes(searchTerm.toLowerCase())
    );
  }

  filterData(event: FiltersChangeEvent): void {
    const queryParams = event.filters;
    this.router.navigate([], {
      relativeTo: this.route,
      queryParams: queryParams
    });
  }
}

describe("FilterGroupComponent", () => {
  let component: MockComponent;
  let fixture: ComponentFixture<MockComponent>;

  beforeEach(async(() => {
    TestBed.configureTestingModule({
      imports: [
        RouterTestingModule,
        FormsModule,
        ReactiveFormsModule,
        AutoCompleteModule,
        MultiSelectModule,
        ButtonModule,
        ProgressSpinnerModule,
        CalendarModule,
        MatMenuModule,
        InputTextModule,
        KeyFilterModule,
        TooltipModule,
        MatMenuModule,
        DropdownModule
      ],
      declarations: [
        MockComponent,
        FilterGroupComponent,
        AdvancedFilterComponent,
        ButtonComponent,
        AutoCompleteComponent,
        CalendarComponent,
        ControlLabelComponent,
        PopoverMenuButtonComponent,
        IconButtonComponent,
        TooltipDirective,
        StatusPillComponent,
        InputComponent,
        DropdownComponent,
        InputErrorMessageComponent,
        ErrorIconComponent
      ]
    }).compileComponents();
  }));

  beforeEach(() => {
    fixture = TestBed.createComponent(MockComponent);
    component = fixture.componentInstance;
    fixture.detectChanges();
  });

  it("should create", () => {
    expect(component).toBeTruthy();
  });

  it("should have an input placeholder", () => {
    const compiled = fixture.nativeElement;
    const placeholder = compiled
      .querySelector("input.is-filter")
      .getAttribute("placeholder");
    expect(placeholder).toBe("Search Product Code or Name");
  });

  it("should have an auto-complete placeholder", () => {
    const compiled = fixture.nativeElement;
    const placeholder = compiled
      .querySelector(
        "span.ui-fluid p-autocomplete:first-child span ul li input"
      )
      .getAttribute("placeholder");
    expect(placeholder).toBe("Search Product Code or Name");
  });

  it("should have a multi select placeholder", () => {
    const compiled = fixture.nativeElement;
    const placeholder = compiled.querySelector(
      ".ui-multiselect-label-container .ui-multiselect-label .ui-multiselected-empty-token"
    ).innerHTML;
    expect(placeholder).toBe("Cell Location");
  });

  it("should have a calendar placeholder", () => {
    const compiled = fixture.nativeElement;
    const placeholder = compiled.querySelector(".ui-calendar input")
      .placeholder;
    expect(placeholder).toBe("Calendar");
  });
  it("should have a dropdown placeholder", () => {
    const compiled = fixture.nativeElement;
    const placeholder = compiled.querySelector(".ui-dropdown label").innerHTML;
    expect(placeholder).toBe("Dropdown");
  });
});
