import { Component, OnInit, Input, Output, EventEmitter } from "@angular/core";
import { SelectItem } from "primeng/api";
import { v4 as uuid } from "uuid";
import { FormGroup, FormControl } from "@angular/forms";
import { MenuItem } from "../popover-menu-button/popover-menu-button.component";

@Component({
  selector: "bre-dropdown",
  templateUrl: "./dropdown.component.html",
  styleUrls: ["./dropdown.component.scss"],
})
export class DropdownComponent implements OnInit {
  @Input() items: SelectItem[];
  @Input() parentForm: FormGroup;
  @Input() controlName: string;
  @Input() label: string;
  @Input() placeholder: string = "Select";
  @Input() disabled: boolean = false;
  @Input() error: boolean;
  @Input() required: boolean;
  @Input() hasPopover: boolean = false;
  @Input() popoverMenuItems: MenuItem[] = [];
  @Input() errorMessage: string;
  @Input() parentValue: string;
  @Input() initialValue: string = "";
  @Input() colorChangingLabel: boolean = false;
  @Input() readOnly: boolean = false;
  @Input() virtualScroll: boolean = false;
  @Input() itemSize: number;
  @Input() splitDropdown: boolean = false;
  @Input() showSelectedValTick:boolean = false;

  labelUUID: string;
  componentUUID: string;
  @Input() value: any;
  @Output() parentFormChanged = new EventEmitter<any>();

  ngOnInit(): void {
    if (this.label) {
      this.labelUUID = uuid();
      const parsedLabel = this.label.replace(/\s+/g, "_").toLowerCase();
      this.componentUUID = `${parsedLabel}_${uuid()}`;
    }
    if (this.parentForm) {
      this.value = this.parentForm.value[this.controlName];
    } else {
      this.parentForm = new FormGroup({
        value: new FormControl({ value: this.value, disabled: this.disabled }),
      });
      this.controlName = "value";
    }
    this.subscribeToFormChanges();
  }

  ngOnChanges(changes) {
    this.subscribeToFormChanges();

    if (this.readOnly) {
      setTimeout(() => this.parentForm.get(this.controlName).disable(), 200);
    } else {
      if (changes.value && this.parentForm) {
        if (changes.value.previousValue !== changes.value.currentValue) {
          this.checkIfPristineAndEmit(
            this.initialValue,
            changes.value.currentValue,
            this.checkCurrentIndex(),
            changes.label
          );
        }
      }
    }
  }

  checkCurrentIndex() {
    let currentIndex = null;
    if (this.parentForm.parent && this.parentForm.parent.value) {
      if (
        this.parentForm.parent.value.spokenNameElementSequenceNumber !== null
      ) {
        currentIndex =
          this.parentForm.parent.value.spokenNameElementSequenceNumber - 1;
      } else if (this.parentForm.parent.value.index !== null) {
        currentIndex = this.parentForm.parent.value.index;
      }
    }
    return currentIndex;
  }

  private subscribeToFormChanges() {
    this.parentForm?.valueChanges.subscribe((changes) => {
      if (
        changes[this.controlName] !== undefined &&
        changes[this.controlName] !== this.value
      ) {
        this.value = changes[this.controlName];
        this.checkIfPristineAndEmit(
          this.initialValue,
          this.value,
          this.checkCurrentIndex(),
          this.label
        );
      }
    });
  }

  private checkIfPristineAndEmit(initialValue, value, currentIndex, label) {
    this.markFieldAsPristineIfReset(value);
    const isPristine = initialValue === value;
    //if there's an observer for the parentFormChanged Output, emit and use that to update the value
    if (this.parentFormChanged.observers.length > 0) {
      this.parentFormChanged.emit({
        isPristine,
        controlName: this.controlName,
        value,
        currentIndex,
        label,
      });
    } else {
      //otherwise, set the value directly in the parent form's control
      this.parentForm.controls[this.controlName].setValue(value);
    }
  }

  markFieldAsPristineIfReset(value) {
    if (value === this.initialValue || value === undefined) {
      this.parentForm.get(this.controlName).markAsPristine();
    } else {
      this.parentForm.get(this.controlName).markAsDirty();
    }
  }
}
