import {
  Component,
  EventEmitter,
  Input,
  OnChanges,
  OnInit,
  Output,
  ViewChild,
} from "@angular/core";
import { IconButtonComponent } from "../icon-button/icon-button.component";
import { TableSortDirection } from "../table/table.component";
import { ColumnFilters } from "../filter-table/filter-table.component";
import { get } from "lodash";
import { ColumnCheckboxFilterComponent } from "../column-checkbox-filter/column-checkbox-filter.component";
import { ColumnDateFilterComponent } from "../column-date-filter/column-date-filter.component";
import { ColumnSearchFilterComponent } from "../column-search-filter/column-search-filter.component";
import { ColumnSortFilterComponent } from "../column-sort-filter/column-sort-filter.component";
import { InputComponent } from "../input/input.component";

@Component({
  selector: "bre-column-filter-dropdown",
  templateUrl: "./column-filter-dropdown.component.html",
  styleUrls: ["./column-filter-dropdown.component.scss"],
})
export class ColumnFilterDropdownComponent
  extends IconButtonComponent
  implements OnInit, OnChanges
{
  @Input() columnData: { [key: string]: string }[] = [];
  @Input() columnName: string;
  @Input() sortable = false;
  @Input() searchFilter = false;
  @Input() dateFilter = false;
  @Input() checkboxFilter = false;
  @Input() sortBy: string;
  @Input() sortDirection: string;
  @Input() currentFilters: ColumnFilters;
  @Input() shouldClear = false;
  @Input() showSearch = false;
  @Input() showSearchAutoSuggest = false;
  @Input() showSearchAutoSuggestList;
  @Input() filterLabel: string;
  @Input() checkBoxList: { [key: string]: string }[] = [];
  @Input() revertToInitialSort: boolean;
  @Input() sortOnApply: boolean = false; //when this variable is set to true sort happens only when user clicks on apply
  @Input() hideDateCheckboxes: boolean = false;
  @Input() hideCheckboxInput: boolean = false;
  @Input() hideFilterIcon: boolean = false;
  @Output() filterUpdate = new EventEmitter<
    [string, ColumnFilters, string, boolean]
  >();
  @Output() checkboxListScroll = new EventEmitter<any>();
  @Output() updatedCheckboxSearchField = new EventEmitter<any>();
  @Output() columnFilterDropdownClick = new EventEmitter<any>();
  @Output() resetShouldClear = new EventEmitter<boolean>();
  @Output() updatedAutoSuggestSearchField = new EventEmitter<any>();
  @Output() clearColumnClicked = new EventEmitter<{ filterType: any }>();
  @ViewChild(ColumnSortFilterComponent)
  private sortFilterComponent: ColumnSortFilterComponent;

  @ViewChild(ColumnCheckboxFilterComponent)
  private checkboxFilterComponent: ColumnCheckboxFilterComponent;

  @ViewChild(ColumnDateFilterComponent)
  private dateFilterComponent: ColumnDateFilterComponent;

  @ViewChild(ColumnSearchFilterComponent)
  private searchFilterComponent: ColumnSearchFilterComponent;

  @ViewChild(InputComponent)
  public searchField: InputComponent;

  @ViewChild(InputComponent)
  public searchFieldAutoSuggest: InputComponent;

  @ViewChild(ColumnCheckboxFilterComponent)
  public typeField: ColumnCheckboxFilterComponent;

  unappliedFilters: ColumnFilters = {};
  originalSortColumn = false;
  originalSortDirection: TableSortDirection = null;
  activeClass = "";
  activeSort = "";
  searchColumnName: string;
  clearSort: boolean = false;
  showSuggestList: boolean = false;
  selectedAutoSuggestList = [];
  constructor() {
    super();
  }

  ngOnInit() {
    if (this.isActiveSortColumn()) {
      this.originalSortColumn = true;
      this.originalSortDirection = TableSortDirection[this.sortDirection];
    }
    this.markIfActiveSortColumn();
  }

  ngOnChanges(changes) {
    const { activeFilters, activeSortDirection } = this.hasActiveFilters();
    if (activeFilters) {
      this.activeClass = "active-column-filters";
    } else {
      this.activeClass = "";
    }

    //Conditions to not to show default sort icon
    if (
      changes?.sortDirection?.previousValue ||
      changes?.sortBy?.currentValue !== changes?.sortBy?.previousValue ||
      changes?.currentFilters?.previousValue !== undefined
    ) {
      if (activeSortDirection) {
        this.activeSort = activeSortDirection;
        this.clearSort = false;
      } else {
        this.activeSort = "";
      }
    }
    if (
      this.originalSortColumn &&
      (this.originalSortDirection == changes?.sortDirection?.currentValue ||
        this.originalSortDirection ==
          changes?.currentFilters?.currentValue?.sortDirection)
    ) {
      this.activeSort = activeSortDirection;
      this.originalSortColumn = true;
    }
    //On clear remove the sort icon
    if (this.revertToInitialSort || this.clearSort) {
      if (this.originalSortColumn) {
        this.activeSort = activeSortDirection;
      } else {
        this.activeSort = "";
      }
    }
  }

  private hasActiveFilters(): {
    activeFilters: boolean;
    activeSortDirection: string;
  } {
    let activeFilters = false;
    let activeSortDirection = ""; //to show sort icon
    if (this.currentFilters) {
      Object.entries(this.currentFilters).forEach(([filterType, filters]) => {
        if (
          !activeFilters &&
          filterType !== "sortDirection" &&
          filterType !== "checkboxFilters"
        ) {
          activeFilters = !!filters;
        } else if (!activeSortDirection && filterType === "sortDirection") {
          activeSortDirection = filters;
        } else if (!activeFilters && filterType === "checkboxFilters") {
          if (filters && this.checkBoxList.length === filters.length) {
            activeFilters = false;
          } else {
            activeFilters = !!filters;
          }
        }
      });
    }
    const activeObj = {
      activeFilters: activeFilters,
      activeSortDirection: activeSortDirection,
    };
    return activeObj;
  }

  markIfActiveSortColumn(): void {
    if (this.isActiveSortColumn()) {
      this.unappliedFilters = {
        sortDirection: TableSortDirection[this.sortDirection],
      };
      setTimeout(() => {
        // avoids an Angular "changed after checked" error
        this.applyFilters();
      });
    }
  }

  isActiveSortColumn(): boolean {
    return (
      this.columnName === this.sortBy &&
      !!TableSortDirection[this.sortDirection]
    );
  }

  onFilterClick(event) {
    const { activeFilters, activeSortDirection } = this.hasActiveFilters();
    this.columnFilterDropdownClick.emit(this.columnName);
    if (this.searchFieldAutoSuggest) {
      this.searchFieldAutoSuggest.parentForm.controls["value"].setValue("");
      this.showSuggestList = false;
      this.showSearchAutoSuggestList = null;
      if (!activeFilters) {
        this.selectedAutoSuggestList = [];
      }
    }
  }

  onCheckboxFilterChange(event) {
    this.unappliedFilters.checkboxFilters = event.selection;
  }

  onDateFilterChange(event) {
    this.unappliedFilters.timeFilters = event;
  }

  onSortDirectionChange(direction): void {
    this.unappliedFilters.sortDirection = direction;
    if (!this.sortOnApply) {
      this.applyFilters();
    }
  }

  onSearchInput(event) {
    this.unappliedFilters.searchValue = event.value;
  }

  onSearchAutoSuggestInput(event) {
    this.showSuggestList = true;
    this.updatedAutoSuggestSearchField.emit({
      columnName: this.columnName,
      query: event.value,
    });
  }

  onCheckboxListScroll(event) {
    this.checkboxListScroll.emit([
      this.columnName,
      CheckboxEventTypes.SCROLL,
      event,
    ]);
  }

  onUpdatedCheckboxSearchField(event) {
    this.updatedCheckboxSearchField.emit([
      this.columnName,
      CheckboxEventTypes.SEARCH,
      event,
    ]);
  }

  applyFilters(): void {
    if (!get(this.unappliedFilters, "sortDirection", null)) {
      this.unappliedFilters.sortDirection = this.currentFilters?.sortDirection;
    }
    if (!get(this.unappliedFilters, "checkboxFilters", null)) {
      this.unappliedFilters.checkboxFilters =
        this.currentFilters?.checkboxFilters;
    }

    if (!get(this.unappliedFilters, "timeFilters", null)) {
      this.unappliedFilters.timeFilters = this.currentFilters?.timeFilters;
    }

    if (!get(this.unappliedFilters, "searchAutoSuggestValues", null)) {
      this.unappliedFilters.searchAutoSuggestValues =
        this.currentFilters?.searchAutoSuggestValues;
    }

    this.filterUpdate.emit([
      this.columnName,
      this.unappliedFilters,
      "apply",
      false,
    ]);
    this.unappliedFilters = {};
    this.clearSort = false;
  }

  clearFilters(): void {
    let filterType;
    if (this.sortFilterComponent && this.shouldClearSort()) {
      filterType = "sortDirection";
      this.sortDirection = this.originalSortDirection;
      this.unappliedFilters.sortDirection = this.originalSortDirection;
      this.sortFilterComponent.clear();
    }
    if (this.checkboxFilterComponent) {
      filterType = "checkboxFilters";
      this.checkboxFilterComponent.clear();
    }
    if (this.dateFilterComponent) {
      filterType = "timeFilters";
      this.dateFilterComponent.clear();
    }
    if (this.searchField) {
      filterType = "searchField";
      this.searchField.parentForm.reset();
    }
    if (this.searchFieldAutoSuggest) {
      filterType = "autoSuggest";
      this.searchFieldAutoSuggest.parentForm.reset();
    }
    this.selectedAutoSuggestList = [];
    this.showSearchAutoSuggestList = [];
    this.unappliedFilters = {};
    this.clearSort = true;
    this.clearColumnClicked.emit({
      filterType: filterType,
    });

    if (!(this.sortFilterComponent && this.shouldClearSort())) {
      this.filterUpdate.emit([
        this.columnName,
        this.unappliedFilters,
        "apply",
        true,
      ]);
    } else {
      this.filterUpdate.emit([
        this.columnName,
        this.unappliedFilters,
        "apply",
        false,
      ]);
    }
  }

  shouldClearSort(): boolean {
    return this.sortDirection != this.originalSortDirection;
  }

  selectSuggestionList(item) {
    this.showSuggestList = false;
    this.selectedAutoSuggestList.push({ label: item, checked: true });
    this.unappliedFilters.searchAutoSuggestValues =
      this.selectedAutoSuggestList;
  }

  toggleAndEmit(item): void {
    item.checked = !item.checked;
    if (this.unappliedFilters.searchAutoSuggestValues)
      this.unappliedFilters.searchAutoSuggestValues =
        this.unappliedFilters.searchAutoSuggestValues.filter(
          (item) => item.checked
        );
  }
}

enum CheckboxEventTypes {
  SCROLL = "scroll",
  SEARCH = "search",
}
