import { Component, EventEmitter, Input, OnInit, Output, ViewChild } from '@angular/core';
import { AbstractControl, FormBuilder, FormGroup, ValidatorFn } from '@angular/forms';
import { TimeFilters } from '../../components/filter-table/filter-table.component';
import { set, get } from 'lodash';
import { CalendarComponent } from '../calendar/calendar.component';
import { ColumnCheckboxFilterComponent } from '../column-checkbox-filter/column-checkbox-filter.component';

@Component({
  selector: 'bre-column-date-filter',
  templateUrl: './column-date-filter.component.html',
  styleUrls: ['./column-date-filter.component.scss']
})
export class ColumnDateFilterComponent implements OnInit {
  @Input() columnData: {[key: string]: string}[];
  @Input() hideDateCheckboxes:boolean = false;
  @Output() updateDateFilters = new EventEmitter<TimeFilters>();
  @Output() checkboxListScroll = new EventEmitter<any>();
  @Output() updatedCheckboxSearchField = new EventEmitter<any>();

  @ViewChild(CalendarComponent) calendar: CalendarComponent;
  @ViewChild(ColumnCheckboxFilterComponent) checkboxFilter: ColumnCheckboxFilterComponent;

  form: FormGroup;
  currentSelection: TimeFilters = {};
  availableTimes: {[key: string]: string};
  timeOptions: {[key: string]: string}[] = [
    {label: '1 AM', value:'01'},
    {label: '2 AM', value:'02'},
    {label: '3 AM', value:'03'},
    {label: '4 AM', value:'04'},
    {label: '5 AM', value:'05'},
    {label: '6 AM', value:'06'},
    {label: '7 AM', value:'07'},
    {label: '8 AM', value:'08'},
    {label: '9 AM', value:'09'},
    {label: '10 AM', value: '10'},
    {label: '11 AM', value: '11'},
    {label: '12 PM', value: '12'},
    {label: '1 PM', value: '13'},
    {label: '2 PM', value: '14'},
    {label: '3 PM', value: '15'},
    {label: '4 PM', value: '16'},
    {label: '5 PM', value: '17'},
    {label: '6 PM', value: '18'},
    {label: '7 PM', value: '19'},
    {label: '8 PM', value: '20'},
    {label: '9 PM', value: '21'},
    {label: '10 PM', value: '22'},
    {label: '11 PM', value: '23'},
    {label: '12 AM', value: '24'},
  ];

  constructor(private fb: FormBuilder) { }

  ngOnInit(): void {
    this.form = this.fb.group({
      datePicker: this.fb.control({value: null, disabled: false}),
    });
    this.subscribeToChanges();

    if (this.columnData && !this.hideDateCheckboxes) {
      this.availableTimes = this.convertDataToAvailableTimes(this.columnData);
    }
  }

  ngOnChanges(): void {
    if (this.columnData && !this.hideDateCheckboxes) {
      this.availableTimes = this.convertDataToAvailableTimes(this.columnData);
    }
  }

  private convertDataToAvailableTimes(data) {
    let availableTimes = data.reduce((acc, rowData): {[key: string]: string}[] => {
      const regex = /^\d{4}-\d{2}-\d{2}\s(\d{2})\:\d{2}\:\d{2}\s/;
      const hour = parseInt(regex.exec(rowData.label)[1]);
      (hour - 1 >= 0) ? acc.push(this.timeOptions[hour - 1]) : acc.push(this.timeOptions[23]);
      return acc;
    }, []);

    return availableTimes.sort((a, b) => {
      if (a.value < b.value) return -1;
      if (a.value > b.value) return 1;
      return 0;
    })
  }

  private subscribeToChanges(): void {
    this.form.get('datePicker').valueChanges.subscribe(val => {
      this.onCalendarValueChange(val);
    });
  }

  onCalendarValueChange(event) {
    if (event) {
      const parsedDates = event.map(date => {
        if (date) {
          return this.parseDateForRequest(date);
        }
      })
      set(this.currentSelection, 'dateRange', parsedDates);
    } else {
      set(this.currentSelection, 'dateRange', null);
    }
    this.emitDateFilters();
  }

  parseDateForRequest(date) {
    const jsDate = new Date(date);
    const oneIndexedMonth = jsDate.getMonth() + 1;
    let month = this.addLeadingZero(oneIndexedMonth.toString());
    let day = this.addLeadingZero(jsDate.getDate().toString());
    return `${jsDate.getFullYear()}-${month}-${day}`
  }

  private addLeadingZero(number: string): string {
    if (number.length === 1) {
      return `0${number}`;
    } else {
      return number;
    }
  }

  closeOverlay() {
    this.calendar.hideOverlay();
  }

  onCheckboxListScroll(event) {
    this.checkboxListScroll.emit(event);
  }

  onUpdatedCheckboxSearchField(event) {
    this.updatedCheckboxSearchField.emit(event);
  }

  onCheckboxFilterChange(event) {
    if (get(event, 'selection', null)) {
      this.currentSelection.selectedHours = event.selection.reduce((acc, timeOption) => {
        acc.push(timeOption.value);
        return acc;
      }, []);
      this.emitDateFilters();
    }
  }

  emitDateFilters() {
    this.updateDateFilters.emit(this.currentSelection);
  }

  clear(): void {
    this.form.reset();
    if(!this.hideDateCheckboxes) {
      this.checkboxFilter.clear();
    }
    this.currentSelection = {};
    this.emitDateFilters();
  }

}

