import { SelectionModel } from "@angular/cdk/collections";
import {
  Component,
  EventEmitter,
  Input,
  OnInit,
  OnChanges,
  Output,
  ViewChild,
} from "@angular/core";
import { InputComponent } from "../input/input.component";
import { ColumnFilters } from "../filter-table/filter-table.component";

@Component({
  selector: "bre-column-checkbox-filter",
  templateUrl: "./column-checkbox-filter.component.html",
  styleUrls: ["./column-checkbox-filter.component.scss"],
})
export class ColumnCheckboxFilterComponent implements OnInit, OnChanges {
  @Input() columnData: { [key: string]: string }[] = [];
  @Input() customFilterOptions: { [key: string]: string }[] = null;
  @Input() filteredList: { [key: string]: string }[] = [];
  @Input() filterLabel = "Filter";
  @Input() currentFilters: ColumnFilters;
  @Input() hideCheckboxInput: boolean = false;
  @Output() updateCheckboxFilters = new EventEmitter<{
    [key: string]: { [key: string]: string }[];
  }>();
  @Output() updatedCheckboxSearchField = new EventEmitter<any>();
  @Output() checkboxListScroll = new EventEmitter<any>();

  @ViewChild("searchField")
  private searchField: InputComponent;

  dedupedColumnData: { [key: string]: string }[];
  currentSearch = "";
  selection: SelectionModel<{ [key: string]: string }> = new SelectionModel(
    true,
    []
  );

  constructor() {}

  ngOnInit(): void {
    if (this.columnData) {
      this.dedupedColumnData = this.dedupe([...this.columnData]);
      this.dedupedColumnData = this.anchorNoneOption(this.dedupedColumnData);
      this.filteredList =
        this.customFilterOptions && this.customFilterOptions.length
          ? this.customFilterOptions
          : this.dedupedColumnData;
      this.selection.select(...this.filteredList);
    }
    if (this.currentFilters?.checkboxFilters) {
      this.selection.clear();
      this.selection.select(...this.currentFilters?.checkboxFilters);
      this.updateCheckboxFilters.emit({
        selection: this.selection.selected,
      });
    }
  }

  ngOnChanges(changes): void {
    if (this.columnData && changes.columnData) {
      this.dedupedColumnData = this.dedupe([...this.columnData]);
      this.dedupedColumnData = this.anchorNoneOption(this.dedupedColumnData);
      this.filteredList =
        this.customFilterOptions && this.customFilterOptions.length
          ? this.customFilterOptions
          : this.dedupedColumnData;
      this.selection.clear();
      this.selection.select(...this.filteredList);
    }
    if (this.currentFilters?.checkboxFilters) {
      this.selection.clear();
      this.selection.select(...this.currentFilters?.checkboxFilters);
      this.updateCheckboxFilters.emit({
        selection: this.selection.selected,
      });
    }
  }

  /**
   * This method is used to anchor the None option to the top if present
   * @param dedupedArr
   * @returns array
   */
  anchorNoneOption(dedupedArr: { [key: string]: string }[]) {
    if (dedupedArr) {
      const indexOfNone = dedupedArr.findIndex((x) => !x.value || !x.label);
      if (indexOfNone > -1) {
        const noneElement = dedupedArr[indexOfNone];
        dedupedArr.splice(indexOfNone, 1);
        dedupedArr.unshift(noneElement);
      }
      return dedupedArr;
    }
    return dedupedArr;
  }

  onListSearchChange(event): void {
    this.currentSearch = event.value;
    let customFilterOptionsVal =
      this.customFilterOptions && this.customFilterOptions.length
        ? true
        : false;
    if (!customFilterOptionsVal && event.value) {
      this.filteredList = this.dedupedColumnData.filter((option) => {
        // to handle undefined filter value
        if (!option.label || !option.value) {
          return /^[None]/.test(this.currentSearch.toLowerCase());
        }
        return option.label
          .toString()
          .toLowerCase()
          .includes(event.value.toLowerCase());
      });
    } else if (customFilterOptionsVal && event.value) {
      this.filteredList = this.customFilterOptions.filter((option) => {
        return option.label.toLowerCase().includes(event.value.toLowerCase());
      });
    } else {
      this.filteredList =
        this.customFilterOptions && this.customFilterOptions.length
          ? this.customFilterOptions
          : this.dedupedColumnData;
    }

    if (this.currentFilters?.checkboxFilters) {
      this.selection.clear();
      this.selection.select(...this.currentFilters?.checkboxFilters);
      this.updateCheckboxFilters.emit({
        selection: this.selection.selected,
      });
    } else {
      if (
        this.filteredList.every((elem) =>
          this.selection.selected.includes(elem)
        )
      )
        this.updateCheckboxFilters.emit({ selection: this.filteredList });
    }
  }

  onFilterItemListScroll(event): void {
    let customFilterOptionsVal =
      this.customFilterOptions && this.customFilterOptions.length
        ? true
        : false;
    if (!customFilterOptionsVal) {
      this.checkboxListScroll.emit(event);
    }
  }

  private dedupe(
    list: { [key: string]: string }[]
  ): { [key: string]: string }[] {
    return list.filter((object, index, self) => {
      return (
        self.findIndex(
          (o) => o.label === object.label && o.value === object.value
        ) === index
      );
    });
  }

  isRowChecked(item): boolean {
    return !!this.selection.selected.find(
      (el) => item.label === el.label && item.value === el.value
    );
  }

  allSelected(): boolean {
    if(this.customFilterOptions && this.customFilterOptions.length > 0) {
      return this.selection.selected.length === this.customFilterOptions.length
    } else if (this.dedupedColumnData) {
      return this.selection.selected.length === this.dedupedColumnData.length;
    } 
  }

  /**
   *
   * @param item item which needs to be toggled from selected to deselected and vice versa
   * for items like boolean values, 'item' is not matching with existing values in this.selection
   * so, toggle adds duplicate values in the selected array.
   * Checking for duplicate values and deselecting them.
   * Emitting the updated selection
   */
  toggleAndEmit(item): void {
    this.selection.toggle(item);
    const selectedSet = this.selection.selected.filter(
      (sel) => sel.value === item.value
    );
    if (selectedSet.length > 1) {
      this.selection.deselect(...selectedSet);
    }
    this.emitUpdatedSelection();
  }

  selectOrDeselectAll(option: boolean): void {
    const data =
      this.customFilterOptions && this.customFilterOptions.length
        ? this.customFilterOptions
        : this.filteredList;
    !!option ? this.selection.select(...data) : this.selection.clear();
    this.emitUpdatedSelection();
  }

  emitUpdatedSelection(): void {
    this.updateCheckboxFilters.emit({
      selection: this.selection.selected,
    });
  }

  clearSearchField(): void {
    this.currentSearch = "";
    if (this.searchField) this.searchField.parentForm.reset();
  }

  clear(): void {
    this.selectOrDeselectAll(true);
    this.clearSearchField();
    this.emitUpdatedSelection();
    if (this.dedupedColumnData) {
      this.selection.select(...this.dedupedColumnData);
    }
  }
}
