import { Input } from "@angular/core";
import {
  async,
  ComponentFixture,
  ComponentFixtureAutoDetect,
  TestBed,
} from "@angular/core/testing";
import { FormsModule, ReactiveFormsModule } from "@angular/forms";
import { MatCheckboxModule } from "@angular/material/checkbox";
import { MatMenuModule } from "@angular/material/menu";
import { ButtonModule } from "primeng/button";
import { InputTextModule } from "primeng/inputtext";
import { KeyFilterModule } from "primeng/keyfilter";
import { ProgressSpinnerModule } from "primeng/progressspinner";
import { ButtonComponent } from "../button/button.component";
import { ControlLabelComponent } from "../control-label/control-label.component";
import { ErrorIconComponent } from "../error-icon/error-icon.component";
import { IconButtonComponent } from "../icon-button/icon-button.component";
import { InputErrorMessageComponent } from "../input-error-message/input-error-message.component";
import { InputComponent } from "../input/input.component";
import { PopoverMenuButtonComponent } from "../popover-menu-button/popover-menu-button.component";
import { TooltipDirective } from "../tooltip/tooltip.directive";

import { ColumnCheckboxFilterComponent } from "./column-checkbox-filter.component";

describe("ColumnCheckboxFilterComponent", () => {
  let component: ColumnCheckboxFilterComponent;
  let fixture: ComponentFixture<ColumnCheckboxFilterComponent>;

  beforeEach(async(() => {
    TestBed.configureTestingModule({
      imports: [
        MatCheckboxModule,
        ReactiveFormsModule,
        FormsModule,
        InputTextModule,
        KeyFilterModule,
        ProgressSpinnerModule,
        MatMenuModule,
        ButtonModule,
      ],
      declarations: [
        ColumnCheckboxFilterComponent,
        InputComponent,
        ControlLabelComponent,
        InputErrorMessageComponent,
        ErrorIconComponent,
        ButtonComponent,
        ControlLabelComponent,
        IconButtonComponent,
        PopoverMenuButtonComponent,
        TooltipDirective,
      ],
    }).compileComponents();
  }));
  beforeEach(() => {
    fixture = TestBed.createComponent(ColumnCheckboxFilterComponent);
    component = fixture.componentInstance;
    component.columnData = [
      { label: "label1", value: "value1" },
      { label: "label2", value: "value2" },
    ];
    fixture.detectChanges();
  });

  it("should create", () => {
    expect(component).toBeTruthy();
  });

  it("selects or deselects all options", () => {
    component.selectOrDeselectAll(true);
    fixture.detectChanges();
    expect(component.selection.selected).toEqual(component.dedupedColumnData);
    component.selectOrDeselectAll(false);
    fixture.detectChanges();
    expect(component.selection.selected).toEqual([]);
  });

  it("correctly checks if a row was checked", () => {
    component.selection.clear();
    component.selection.select({ label: "label1", value: "value1" });
    fixture.detectChanges();
    const resultTrue = component.isRowChecked({
      label: "label1",
      value: "value1",
    });
    const resultFalse = component.isRowChecked({
      label: "label2",
      value: "value2",
    });
    expect(resultTrue).toEqual(true);
    expect(resultFalse).toEqual(false);
  });

  it("correctly checks if all the items are selected", () => {
    component.selection.select(component.columnData[0]);
    component.selection.select(component.columnData[1]);
    fixture.detectChanges();
    const resultFalse = component.allSelected();
    component.selection.clear();
    fixture.detectChanges();
    const resultTrue = component.allSelected();
    expect(resultTrue).toEqual(false);
    expect(resultFalse).toEqual(true);
  });

  it("clear calls selectAll and emits an update", () => {
    spyOn(component, "selectOrDeselectAll");
    spyOn(component, "emitUpdatedSelection");
    component.clear();
    fixture.detectChanges();
    expect(component.selectOrDeselectAll).toHaveBeenCalledWith(true);
    expect(component.emitUpdatedSelection).toHaveBeenCalled();
  });

  it("updates the current search value and filtered list on list search change", () => {
    const event1 = { value: "test" };
    component.onListSearchChange(event1);
    expect(component.currentSearch).toEqual(event1.value);

    const event2 = { value: "label1" };
    component.customFilterOptions = [
      { label: "label1", value: "value1" },
      { label: "label2", value: "value2" },
    ];
    component.onListSearchChange(event2);
    expect(component.filteredList).toEqual([
      { label: "label1", value: "value1" },
    ]);

    component.onListSearchChange({});
    expect(component.filteredList).toEqual(component.customFilterOptions);
  });

  it("should emit an event on scroll if there are no custom filter options", () => {
    const event = { value: "test" };
    const spy = spyOn(component.checkboxListScroll, "emit");
    component.onFilterItemListScroll(event);
    expect(spy).toHaveBeenCalledWith(event);
  });

  it("should emit an event on selection toggle", () => {
    const event = { label: "label3", value: "value3" };

    const spy = spyOn(component.updateCheckboxFilters, "emit");
    component.toggleAndEmit(event);

    expect(spy).toHaveBeenCalledWith({
      selection: component.columnData.concat([event]),
    });

    expect(spy).toHaveBeenCalledWith({
      selection: component.columnData.concat([event]),
    });
  });
});
