import { SelectionModel } from "@angular/cdk/collections";
import {
  Component,
  Input,
  Output,
  EventEmitter,
  OnChanges,
} from "@angular/core";

@Component({
  selector: "bre-checkbox-dropdown-with-pills",
  templateUrl: "./checkbox-dropdown-with-pills.component.html",
  styleUrls: ["./checkbox-dropdown-with-pills.component.scss"],
})
export class CheckboxDropdownWithPillsComponent implements OnChanges {
  @Input() placeholderTextOuter: string;
  @Input() labelTextOuter: string;
  @Input() placeholderText: string;
  @Input() labelText: string;
  @Input() checkboxList;
  @Input() loadingSearchedList: boolean = false;
  @Input() clearFilteredList: boolean = false;
  @Output() seachStr: EventEmitter<string> = new EventEmitter();
  @Output() onApplyClicked: EventEmitter<any> = new EventEmitter();
  @Output() selectedCount: EventEmitter<number> = new EventEmitter();
  @Output() cancelClicked: EventEmitter<any> = new EventEmitter();
  showCheckboxPanel: boolean = false;
  showPillsPanel: boolean = false;
  selectedData: SelectionModel<any> = new SelectionModel(true, []);
  showDropDownElement: boolean = false;

  /**
   *
   * @param changes changes in checkboxList after search happen
   * fetching the checkboxlist after either search results gets changed
   * or any other change from the component
   */
  ngOnChanges(changes) {
    this.checkboxList = changes.checkboxList?.currentValue
      ? changes.checkboxList.currentValue
      : [];
    if (this.checkboxList.length > 0) {
      this.checkboxList
        .filter((list) => this.selectedData.selected.includes(list))
        .map((item) => (item.checked = true));
      this.showSelectedCheckboxPanel();
    } else {
      this.hidePanels();
    }
    this.loadingSearchedList = changes.loadingSearchedList?.currentValue
      ? true
      : false;
    this.setPlaceholder();
    if (changes.clearFilteredList?.currentValue) {
      this.onCancel();
    }
  }

  /**
   * showing and hiding panel after clicking on down arrow icon
   */
  showDropdown() {
    this.showDropDownElement = !this.showDropDownElement;
    if (this.selectedData.selected.length > 0) {
      this.showSelectedPillsPanel();
    } else {
      this.showCheckboxPanel = false;
      this.showPillsPanel = false;
    }
  }

  /**
   *
   * @param event capturing the keyboard event
   * if enter key has been pressed then:
   * a loader would be shown
   * search string would be emitted
   * checkbox panel will be visible and pills panel will be hidden
   *
   * if all the characters from the textbox have been removed then:
   * pills panel would be shown
   * checkbox panel would be hidden
   */
  performSearch(event) {
    if (event.keyCode === 13) {
      const searchString = event.target.value;
      this.seachStr.emit(searchString);
      this.showSelectedCheckboxPanel();
    }

    if (event.target.value === "") {
      this.showSelectedPillsPanel();
    }
  }

  /**
   *
   * @param item item that has been checked/unchecked
   * @returns if the checkbox has been selected or deselected
   * this method is checking whether the row has been selected,
   * if selected it has been added to an array
   * with every selection the selected item count has been emitted.
   */
  isRowChecked(item) {
    let isSelected: boolean = false;
    const filteredSelectedData = this.selectedData.selected.filter(
      (el) => el.label === item.label && el.value === item.value
    );
    if (filteredSelectedData.length !== 0) {
      isSelected = true;
    }
    return isSelected;
  }

  /**
   *
   * @param item item that has been checked
   * to toggle the selected data's 'checked' value
   */
  toggleAndEmit(item): void {
    item.checked = !item.checked;
    this.selectedData.toggle(item);
    this.setPlaceholder();
  }

  /**
   * on click of 'Clear All' button:
   * hide the pills and checkbox panel and unchecking all the selected values
   * clearing the selected data and emitting the event
   */
  onCancel() {
    this.hidePanels();
    this.checkboxList.forEach(
      (option, index) => (this.checkboxList[index].checked = false)
    );
    this.showDropDownElement = false;
    this.selectedData.clear();
    this.cancelClicked.emit();
    this.setPlaceholder();
  }

  /**
   * on click of 'Apply' button:
   * hide the panel and emit the selected values
   */
  onApply() {
    this.onApplyClicked.emit(this.selectedData.selected);
    this.showDropDownElement = false;
  }

  /**
   * closes the panel
   */
  close() {
    this.showDropDownElement = false;
  }

  /**
   * show pills panel
   */
  showSelectedPillsPanel() {
    this.showPillsPanel = true;
    this.showCheckboxPanel = false;
  }

  /**
   * show checkbox panel
   */
  showSelectedCheckboxPanel() {
    this.showCheckboxPanel = true;
    this.showPillsPanel = false;
  }

  /**
   * hide panels
   */
  hidePanels() {
    this.showCheckboxPanel = false;
    this.showPillsPanel = false;
  }

  /**
   * This function is called to set the
   * placeholder value for multi select drop down
   */
  setPlaceholder() {
    let checkedList = this.selectedData.selected.filter((option, index) => {
      return this.selectedData.selected[index].checked === true;
    }).length;
    if (checkedList === 0) {
      this.placeholderTextOuter = "Select Restaurant(s)";
    } else {
      this.placeholderTextOuter = "Selected";
    }
  }
}
