import {
  Component,
  OnInit,
  Input,
  Output,
  EventEmitter,
  SimpleChanges,
  OnChanges,
} from "@angular/core";
import { v4 as uuid } from "uuid";
import { SelectItem } from "primeng/api";
import { FormGroup, FormControl } from "@angular/forms";
import { MenuItem } from "../popover-menu-button/popover-menu-button.component";

export interface CategorySelectItem extends SelectItem {
  searchText: string;
}

@Component({
  selector: "bre-category-dropdown",
  templateUrl: "./category-dropdown.component.html",
  styleUrls: ["./category-dropdown.component.scss"],
})
export class CategoryDropdownComponent implements OnInit, OnChanges {
  @Input() options: CategorySelectItem[];
  @Input() label: string;
  @Input() parentForm: FormGroup;
  @Input() controlName: string;
  @Input() disabled: boolean;
  @Input() error: boolean;
  @Input() required: boolean;
  @Input() placeholder = "Select";
  @Input() errorMessage: string;
  @Input() hasPopover: boolean;
  @Input() popoverMenuItems: MenuItem[] = [];
  @Input() shouldReset: boolean;
  @Input() parentValue: string;
  @Input() initialValue: any = "";
  @Input() colorChangingLabel: boolean;
  @Input() filteredOptions: CategorySelectItem[] = [];
  @Input() forceSelection = true;
  // For Pill
  @Input() selectedType: string;
  @Input() unselectedType: string;
  @Input() isSelected: boolean;
  @Input() isSelectable: boolean;
  @Input() statusPillHidden: boolean;
  @Input() statusPillText = "";

  @Input() value: any;
  @Output() selectDropdown = new EventEmitter<string>();
  @Output() resetForm = new EventEmitter<boolean>();
  @Output() parentFormChanged = new EventEmitter<any>();
  @Output() searchQuery = new EventEmitter<any>();

  selectedItem: CategorySelectItem;
  labelUUID: string;
  componentUUID: string;
  selectionLabel: string;

  constructor() {
    this.filteredOptions = this.options;
  }

  ngOnInit(): void {
    if (this.label) {
      this.labelUUID = uuid();
      const parsedLabel = this.label.replace(/\s+/g, "_").toLowerCase();
      this.componentUUID = `${parsedLabel}_${uuid()}`;
    } else if (this.isSelectable && this.statusPillText) {
      const parsedStatusPillText = this.statusPillText
        .replace(/\s+/g, "_")
        .toLowerCase();
      this.componentUUID = `${parsedStatusPillText}_${uuid()}`;
    }

    if (!this.initialValue) {
      this.initialValue = "";
    }
    // the PrimeNG autocomplete component is technically being declared as a template-driven form control
    // this + the onSelect method will set the ngModel to reflect what the value of the reactive form is
    if (this.parentForm) {
      this.selectedItem = this.options?.find(
        (x) => x.value === this.parentForm.value[this.controlName]
      );
      this.value = this.parentForm.value[this.controlName];
    } else {
      this.selectedItem = this.options?.find((x) => x.value === this.value);
      this.parentForm = new FormGroup({
        value: new FormControl({ value: this.value, disabled: this.disabled }),
      });
      this.controlName = "value";
    }
    this.subscribeToFormChanges();
  }

  ngOnChanges(changes: SimpleChanges): void {
    this.subscribeToFormChanges();
    if (this.shouldReset) {
      setTimeout(() => {
        this.selectedItem = {
          ...this.options?.find((x) => x.value === this.value),
        };
        this.resetForm.emit(false);
      });
    } else {
      if (changes.value) {
        if (changes.value.previousValue !== changes.value.currentValue) {
          setTimeout(() => {
            this.checkIfPristineAndEmit(
              this.initialValue,
              changes.value.currentValue,
              this.checkCurrentIndex(),
              changes.label
            );
          });
          this.selectedItem = {
            ...this.options?.find((x) => x.value === this.value),
          };
        }
      }
    }
    if (!!this.selectedItem && this.selectedItem.value !== this.value) {
      if (this.value !== "") {
        this.selectedItem = {
          ...this.options?.find((x) => x.value === this.value),
        };
      }
    }
  }
  subscribeToFormChanges() {
    this.parentForm?.valueChanges.subscribe((changes) => {
      if (
        changes[this.controlName] !== undefined &&
        changes[this.controlName] !== this.value
      ) {
        this.value = changes[this.controlName];
        if (this.forceSelection) {
          this.selectedItem = {
            ...this.options?.find((x) => x.value === this.value),
          };
        }
        this.checkIfPristineAndEmit(
          this.initialValue,
          this.value,
          this.checkCurrentIndex(),
          this.label
        );
      }
    });
  }
  onComplete(event): void {
    this.filteredOptions = this.options.filter((x) =>
      x.searchText.toLowerCase().includes(event.query.toLowerCase())
    );
    this.searchQuery.emit({
      query: event.query.toLowerCase(),
      filteredOptions: this.filteredOptions,
    });
  }

  checkCurrentIndex() {
    let currentIndex = null;
    if (this.parentForm.parent && this.parentForm.parent.value) {
      if (
        this.parentForm.parent.value.spokenNameElementSequenceNumber !== null
      ) {
        currentIndex =
          this.parentForm.parent.value.spokenNameElementSequenceNumber - 1;
      } else if (this.parentForm.parent.value.index !== null) {
        currentIndex = this.parentForm.parent.value.index;
      }
    }
    return currentIndex;
  }

  onSelect(event): void {
    // since selectedItem is not directly tied to the value in the reactive form, set it
    this.checkIfPristineAndEmit(
      this.initialValue,
      event.value,
      this.checkCurrentIndex(),
      event.label
    );
  }

  private checkIfPristineAndEmit(initialValue, value, currentIndex, label) {
    this.markFieldAsPristineIfReset(value);
    const isPristine = initialValue === value;
    //if there's an observer for the parentFormChanged Output, emit and use that to update the value
    if (this.parentFormChanged.observers.length > 0) {
      this.parentFormChanged.emit({
        isPristine,
        controlName: this.controlName,
        value,
        currentIndex,
        label,
      });
      this.value = value;
    } else {
      //otherwise, set the value directly in the parent form's control
      this.parentForm.controls[this.controlName].setValue(value);
      this.value = value;
    }
  }

  markFieldAsPristineIfReset(value) {
    if (value === this.initialValue) {
      this.parentForm.get(this.controlName).markAsPristine();
    } else {
      this.parentForm.get(this.controlName).markAsDirty();
    }
  }

  onBlur(event): void {
    // in case of partial deletion of input and user clicks out, reset the selected item to the previous value
    if (this.forceSelection) {
      this.selectedItem = {
        ...this.options?.find((x) => x.value === this.value),
      };
    }
  }

  handleKeyUp($event) {
    if (!this.forceSelection && $event.target.value) {
      this.checkIfPristineAndEmit(
        this.initialValue,
        $event.target.value,
        this.checkCurrentIndex(),
        $event.label
      );
    }
    if (!$event.target.value) {
      this.checkIfPristineAndEmit(
        this.initialValue,
        "",
        this.checkCurrentIndex(),
        ""
      );
    }
  }

  handleClick() {
    this.selectDropdown?.emit(this.statusPillText);
  }

  labelConversion(data): string {
    if (data.label) {
      return data.searchText;
    }
  }
}
