import { async, ComponentFixture, TestBed } from '@angular/core/testing';
import { FormsModule, ReactiveFormsModule, FormGroup, FormControl } from '@angular/forms';
import { DropdownModule } from 'primeng/dropdown';
import { CategoryDropdownComponent } from './category-dropdown.component';
import { InputErrorMessageComponent } from '../input-error-message/input-error-message.component';
import { PopoverMenuButtonComponent } from '../popover-menu-button/popover-menu-button.component';
import { IconButtonComponent } from '../icon-button/icon-button.component';
import { MatMenuModule } from '@angular/material/menu';
import { ControlLabelComponent } from '../control-label/control-label.component';
import { TooltipDirective } from '../tooltip/tooltip.directive';
import { ErrorIconComponent } from '../error-icon/error-icon.component';
import { StatusPillComponent } from '../status-pill/status-pill.component';
import { AutoCompleteModule } from 'primeng/autocomplete';
import { EventEmitter, SimpleChange } from '@angular/core';

describe('CategoryDropdown', () => {
  let component: CategoryDropdownComponent;
  let fixture: ComponentFixture<CategoryDropdownComponent>;

  beforeEach(async(() => {
    TestBed.configureTestingModule({
      imports: [
        ReactiveFormsModule,
        DropdownModule,
        AutoCompleteModule,
        FormsModule,
        MatMenuModule
      ],
      declarations: [
        CategoryDropdownComponent,
        InputErrorMessageComponent,
        PopoverMenuButtonComponent,
        IconButtonComponent,
        ControlLabelComponent,
        TooltipDirective,
        ErrorIconComponent,
        StatusPillComponent
      ]
    })
      .compileComponents();
  }));

  beforeEach(() => {
    fixture = TestBed.createComponent(CategoryDropdownComponent);
    component = fixture.componentInstance;
    fixture.detectChanges();
  });

  it('should create', () => {
    expect(component).toBeTruthy();
  });

  it('should display a label if one has been declared', () => {
    component.label = "Test";
    component.ngOnInit();
    fixture.detectChanges();
    expect(component.componentUUID).toContain('test');
  });

  it('should display a status pill if configured to do so', () => {
    component.isSelectable = true;
    component.statusPillHidden = false;
    component.statusPillText = "Test";
    component.ngOnInit();
    fixture.detectChanges();

    const pill = fixture.nativeElement.querySelector('bre-status-pill');

    expect(pill).toBeDefined();
    expect(component.componentUUID).toContain('test');
  });

  it('should update the selected item when the input is changed to an option whose value matches', () => {
    const item1 = {label: 'test item 1', value: 1, searchText: '1'};
    const item2 = {label: 'test item 2', value: 2, searchText: '2'};
    component.initialValue = 1;
    component.options = [item1, item2];
    component.selectedItem = item1;
    component.value = 2;
    component.ngOnChanges({value: new SimpleChange(1, 2, true)});
    
    expect(component.selectedItem).toEqual(item2);
  });

  it('should emit a search event on complete', () => {
    component.options = [];
    const spy = spyOn(component.searchQuery, 'emit');
    const event = {query: 'test'};
    component.onComplete(event);
    expect(spy).toHaveBeenCalledWith({query: event.query, filteredOptions: []});
  });

  it('should mark the form field as pristine if reset to its initial value and dirty if set to another value', () => {
    component.initialValue = 1;
    component.controlName = 'testControl';
    component.parentForm = new FormGroup({ 'testControl': new FormControl([''])});
    const pristineSpy = spyOn(component.parentForm.get(component.controlName), 'markAsPristine');
    component.onSelect({ value: 1 });

    expect(pristineSpy).toHaveBeenCalled();

    const dirtySpy = spyOn(component.parentForm.get(component.controlName), 'markAsDirty');
    component.onSelect({ value: 2 });

    expect(dirtySpy).toHaveBeenCalled();
  });

  it('should force an item selection on blur if configured to do so by resetting to the previous value', () => {
    const item1 = {label: 'test item 1', value: 1, searchText: '1'};
    const item2 = {label: 'test item 2', value: 2, searchText: '2'};
    component.forceSelection = true;
    component.value = 1;
    component.options = [item1, item2];
    component.selectedItem = null;

    component.onBlur({});

    expect(component.selectedItem).toEqual(item1);
  });

  it('should convert the label displayed to the search text for the autocomplete input', () => {
    const example = {label: 'wrong', searchText: 'right'};

    expect(component.labelConversion(example)).toBe(example.searchText);
  });

  it('should emit a select event on click of the status pill if configured to do so', () => {
    component.selectDropdown = new EventEmitter<any>();
    component.statusPillText = "Test";

    const spy = spyOn(component.selectDropdown, 'emit');
    component.handleClick();
    
    expect(spy).toHaveBeenCalledWith(component.statusPillText);
  });
});
