import {
  Component,
  Input,
  ViewEncapsulation,
  OnInit,
  OnChanges,
  ViewChild,
  Output,
  EventEmitter
} from "@angular/core";
import { v4 as uuid } from "uuid";
import { FormGroup, FormControl } from "@angular/forms";
import { Calendar } from "primeng/calendar";

@Component({
  selector: "bre-calendar",
  templateUrl: "./calendar.component.html",
  styleUrls: ["../input/input.component.scss", "./calendar.component.scss"],
  encapsulation: ViewEncapsulation.None
})
export class CalendarComponent implements OnChanges, OnInit {
  @Input() label: string;
  @Input() placeholder: string = "Select Date Range";
  @Input() dateFormat: string = "mm/dd/yy";
  @Input() selectionMode: string = "single";
  @Input() selectionModeType: string = "";
  @Input() labelType: string = "";
  @Input() parentForm: FormGroup;
  @Input() controlName: string;
  @Input() disabled: boolean;
  @Input() parentValue: string;
  @Input() minDate: Date = undefined;
  @Input() maxDate: Date = undefined;
  @Input() view: string = "date";
  @Input() applyButtonText: string = "Apply";
  @Input() clearButtonText: string = "Clear";
  @Input() disableClearButton: boolean = false;
  @Input() timeOnly: boolean = false;
  @Input() dataType: string = "date";
  @Input() filterCalendar: boolean = false;

  // Status Pill
  @Input() isSelectable: boolean = false;
  @Input() isSelected: boolean;
  @Input() selectedType: string;
  @Input() unselectedType: string;
  @Input() statusPillText: string;
  @Output() calendarSelected = new EventEmitter<any>();
  @ViewChild("calendar") calendar: Calendar;

  private _value: any;
  @Input() get value() {
    return this._value;
  }
  set value(val: any) {
    if (this._value !== val) {
      this._value = val;
      this.valueChange.emit(val);
      this.updateClasses();
    }
  }
  @Output() valueChange = new EventEmitter<any>();

  labelUUID: string;
  componentUUID: string;

  ngOnChanges(changes): void {
    if (changes.value && changes.value.firstChange) {
      this.initForm();
    } else if (changes.value && !changes.value.currentValue) {
      this.parentForm.reset();
      this.updateClasses();
    } else if (
      this.selectionMode !== "range" &&
      changes.value &&
      new Date(changes.value.currentValue).getTime() !==
        new Date(changes.value.previousValue).getTime()
    ) {
      this.parentForm.setValue({ value: new Date(changes.value.currentValue) });
    } else if (changes?.disabled) {
      if (changes?.disabled.currentValue) {
        this.parentForm.controls["value"].disable();
      } else {
        this.parentForm.controls["value"].enable();
      }
    }
  }

  ngOnInit(): void {
    if (this.label) {
      this.labelUUID = uuid();
      const parsedLabel = this.label.replace(/\s+/g, "_").toLowerCase();
      this.componentUUID = `${parsedLabel}_${uuid()}`;
    }
    if (!this.parentForm) {
      this.initForm();
    }
  }

  clearDate() {
    this.value = null;
    this.parentForm.reset();
  }

  hideOverlay() {
    this.calendar.hideOverlay();
  }

  updateValue() {
    this.value = this.parentForm.value.value;
    if (this.isSelectable) {
      this.calendarSelected.emit();
    }
  }

  updateClasses() {
    if (this.calendar?.el) {
      //clean up classes
      this.calendar.el.nativeElement
        .querySelectorAll(".ui-datepicker-group .range-end")
        .forEach((x) =>
          x.classList.remove(
            "range-end",
            "range-left",
            "range-right",
            "boundary-right",
            "boundary-left"
          )
        );

      //execute as macro task because dom needs to update before correct selectors are available
      setTimeout(() => {
        if (this.value) {
          let value =
            this.selectionMode === "single" ? [this.value] : this.value;

          const startMonth = value[0]?.toLocaleString("default", {
            month: "long"
          });
          const startYear = value[0]?.getFullYear()?.toString();
          const endMonth = value[1]?.toLocaleString("default", {
            month: "long"
          });
          const endYear = value[1]?.getFullYear()?.toString();

          const [leftPanelEl, rightPanelEl] =
            this.calendar.el.nativeElement.querySelectorAll(
              ".ui-datepicker-group"
            );

          if (leftPanelEl) {
            this.setPanelClasses(
              leftPanelEl,
              startMonth,
              startYear,
              endMonth,
              endYear
            );
          }
          if (rightPanelEl) {
            this.setPanelClasses(
              rightPanelEl,
              startMonth,
              startYear,
              endMonth,
              endYear
            );
          }
        }
      }, 0);
    }
  }

  private initForm(): void {
    this.labelUUID = uuid();
    if (this.parentForm) {
      this.value = this.parentForm.value[this.controlName];
    } else {
      this.parentForm = new FormGroup({
        value: new FormControl({ value: this.value, disabled: this.disabled })
      });
      this.controlName = "value";
    }
  }

  private setPanelClasses(panelEl, startMonth, startYear, endMonth, endYear) {
    const panelCurrentDays = panelEl.querySelectorAll(
      ".ui-datepicker-current-day:not(.ui-datepicker-other-month)"
    );
    if (this.panelContainsMonthYear(panelEl, startMonth, startYear)) {
      this.setRangeLeftClasses(panelCurrentDays);
    }
    if (this.panelContainsMonthYear(panelEl, endMonth, endYear)) {
      this.setRangeRightClasses(panelCurrentDays);
    }
    this.setBoundaryLeftClasses(panelCurrentDays);
    this.setBoundaryRightClasses(panelCurrentDays);
  }

  private panelContainsMonthYear(panelEl, month, year) {
    if (
      panelEl.querySelector(".ui-datepicker-month") &&
      panelEl.querySelector(".ui-datepicker-year")
    ) {
      return (
        panelEl.querySelector(".ui-datepicker-month").innerText === month &&
        panelEl.querySelector(".ui-datepicker-year").innerText === year
      );
    } else {
      return false;
    }
  }

  private setRangeLeftClasses(panelCurrentDays) {
    panelCurrentDays[0].classList.add("range-end");
    if (this.value[1]) {
      panelCurrentDays[0].classList.add("range-left");
    }
  }

  private setRangeRightClasses(panelCurrentDays) {
    panelCurrentDays[panelCurrentDays.length - 1].classList.add(
      "range-end",
      "range-right"
    );
  }

  private setBoundaryLeftClasses(panelCurrentDays) {
    if (
      panelCurrentDays.length > 0 &&
      !panelCurrentDays[0].classList.contains("range-end")
    ) {
      panelCurrentDays[0].classList.add("boundary-left");
    }
  }

  private setBoundaryRightClasses(panelCurrentDays) {
    if (
      panelCurrentDays.length > 0 &&
      !panelCurrentDays[panelCurrentDays.length - 1].classList.contains(
        "range-end"
      )
    ) {
      panelCurrentDays[panelCurrentDays.length - 1].classList.add(
        "boundary-right"
      );
    }
  }
}
