resource "aws_security_group" "lambda_sg" {
  name_prefix = "${var.tags["Name"]}-LAMBDA-SG"
  vpc_id      = var.vpc_id
  description = "Security group used for the Lambda (Managed by Terraform)"

  depends_on = [module.lambda.role_name]

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }

  tags = var.tags

  timeouts {
    delete = "40m"
  }
}

resource "aws_sqs_queue" "image_cache_queue" {
  name = var.image_cache_queue
  fifo_queue                  = true
  delay_seconds               = 90
  max_message_size            = 262144
  message_retention_seconds   = 86400
  receive_wait_time_seconds   = 10
  visibility_timeout_seconds  = 300
  content_based_deduplication = true
  tags = var.tags
}

locals {
  environment_variables = merge(
    var.tags,
    var.environment_variables,
    tomap({ region = var.region })
  )
}

module "lambda" {
  source           = "git::https://github.com/bre-org/tfm-lambda-terraform-module.git"
  function_name    = "${var.tags["Name"]}-LAMBDA"
  filename         = "../function.zip"
  source_code_hash = "${filebase64sha256("../function.zip")}"
  handler          = "main.lambda_handler"
  runtime          = "python3.7"
  tags             = var.tags
  timeout          = 60
  lambda_layer_arn = var.lambda_layer_arn
  environment      = "${local.environment_variables}"
  vpc_config = {
    security_group_ids = ["${aws_security_group.lambda_sg.id}"]
    subnet_ids         = "${data.aws_subnets.private.ids}"
  }
}

# Allowing API Gateway to Access Lambda
#resource "aws_lambda_permission" "api_gateway_lambda_access" {
#  statement_id  = "AllowAPIGatewayInvoke"
#  action        = "lambda:InvokeFunction"
#  function_name = module.lambda.function_name
#  principal     = "apigateway.amazonaws.com"
#  # The "/*/*" portion grants access from any method on any resource within the API Gateway REST API.
#  source_arn = "${aws_api_gateway_rest_api.api_gateway_lambda.execution_arn}/*/*"
#}

resource "aws_iam_role_policy_attachment" "AWSLambdaVPCAccessExecutionRole" {
  role       = module.lambda.role_name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

resource "aws_iam_role_policy_attachment" "AWSLambdaBasicExecutionRole" {
  role       = module.lambda.role_name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
}

resource "aws_iam_role_policy" "secretsmanager" {
  name = "secretsmanager"
  role = module.lambda.role_name

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "SecretsManager",
            "Effect": "Allow",
            "Action": [
                "secretsmanager:GetRandomPassword",
                "secretsmanager:GetResourcePolicy",
                "secretsmanager:GetSecretValue",
                "secretsmanager:DescribeSecret",
                "secretsmanager:ListSecretVersionIds",
                "lambda:ListFunctions"
            ],
            "Resource": "*"
        }
    ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "AmazonAPIGatewayInvokeFullAccess" {
  role       = module.lambda.role_name
  policy_arn = "arn:aws:iam::aws:policy/AmazonAPIGatewayInvokeFullAccess"
}

resource "aws_iam_role_policy_attachment" "sqs_policy_arn" {
  role       = module.lambda.role_name
  policy_arn = var.sqs_policy_arn
}

resource "aws_lambda_event_source_mapping" "event_source_mapping" {
  event_source_arn = aws_sqs_queue.image_cache_queue.arn
  enabled          = true
  function_name    = module.lambda.arn
  batch_size       = 1
}
