from typing import Any
import git
import os
from logging2 import Logger, LogLevel, StdOutHandler, StdErrHandler
import json
import yaml
import boto3
from aws_requests_auth.boto_utils import BotoAWSRequestsAuth
import requests

stdout = StdOutHandler(level=LogLevel.info)
stderr = StdErrHandler(level=LogLevel.error)
logger = Logger("imageCaching", handlers=[stdout, stderr])

def create_response_body(status_code: int, message: str) -> dict:
    if status_code == 200:
        status = "Success"
        error = None
        data = {
                "message": message
            }
    else:
        status = "Failure"
        data = None
        error = {
                "errorMessage": message
            }
    
    return  {
        "status": status,
        "error": error,
        "data": data
        }

# To checkout GitOps repo
def git_checkout():

    # Get Environment Variables
    gitops_repo = os.environ.get("gitopsRepo", "brep-imagecache-baseline")
    git_url = os.environ.get(
        "gitUrl", "github.com/bre-org/" + gitops_repo + ".git"
    )
    git_url_protocol = os.environ.get("gitUrlProtocol", "https")
    git_branch = os.environ.get("gitBranch", "main")

    # Secret Manager key Name
    sm_name = os.environ.get("smName", "github/bre-cloud-git-reconcile")
    region_name = os.environ.get("region", "us-east-1")

    # Create a Secrets Manager client
    session = boto3.session.Session()
    client = session.client(service_name="secretsmanager", region_name=region_name)
    get_secret_value_response = client.get_secret_value(SecretId=sm_name)
    secret = get_secret_value_response["SecretString"]
    jsecret = json.loads(secret)
    gitusername = jsecret["username"]
    gitpassword = jsecret["password"]

    checkout_directory = "/tmp/" + gitops_repo

    if os.path.exists(checkout_directory):
        logger.info(
            '"{}" Exist, Pulling the latest changes from remote'.format(
                checkout_directory
            )
        )
        repo = git.Repo(checkout_directory)
        pull_remote_changes = repo.remotes.origin.pull()
        logger.info('Pulled changes to local: "{}"'.format(pull_remote_changes))
    else:
        logger.info('Checking out git repository "{}"'.format(git_url))
        git.Repo.clone_from(
            git_url_protocol
            + "://"
            + gitusername
            + ":"
            + gitpassword
            + "@"
            + git_url,
            checkout_directory,
            branch=git_branch
        )

    return checkout_directory


# Update app information in repo brep-imagecache-baseline
def create_bundle_tag(git_path, release_name, applications, commit_msg, bundle_id):
    logger.info("In createReleaseBundle")
    image_cache_yaml = os.path.join(
        git_path, "kube-fledged/kubefledged-imagecache.yaml"
    )

    app_list = []

    # Format apps into app_path:version
    for application in applications:
        application_url = application["applicationUrl"]
        app_list.append(application_url)

    # Replace existing image-version entries with new appList in kubefledged-imagecache.yaml
    with open(image_cache_yaml, "r+") as file:
        content = yaml.full_load(file)
        content["spec"]["cacheSpec"][0]["images"] = app_list
        file.seek(0)
        yaml.dump(content, file)
        file.truncate()
    git_push(git_path, release_name, commit_msg)
    return git_create_tag(git_path, release_name, bundle_id)


# Create tag with release bundle name in brep-imagecache-baseline repo
def git_create_tag(git_path, release_name, bundle_id):
    ####### Check if tag exist and force create - Will also need to update the update release bundle flow ########
    repo = set_git_credentails(git_path)
    logger.info('Creating tag "{}"'.format(release_name))
    tag = repo.create_tag(
        release_name, message='Tag for Release Bundle "{0}"'.format(release_name)
    )
    repo.remotes.origin.push(tag)
    response = updated_deployment_details(repo, release_name, bundle_id)
    return response


def updated_deployment_details(repo, tag, bundle_id):
    env = os.environ.get("env")
    aws_region = os.environ.get("region", "us-east-1")

    asset_url = f"asset.api.{env}.bre.mcd.com"
    aws_auth = BotoAWSRequestsAuth(
        aws_host= asset_url,
        aws_region=aws_region,
        aws_service="execute-api"
    )
    tags = repo.tags
    
    for repo_tag in tags:
        if repo_tag.name == tag:
            tagref = repo_tag
    commit_id = tagref.commit
    request_body = {
        "releaseId": bundle_id,
        "commitId": str(commit_id),
        "gitTag": tag,
        "status": "CREATED"
    }
    response = requests.patch(url = f"https://{asset_url}/bundledRelease/deployment/details", data=json.dumps(request_body), auth=aws_auth)
    response = (json.loads(response.text))
    if response["data"]["isSuccess"]:
        return "Updated deployment details successfully"
    else:
        return "There is an issue to update deployment details"
    

# Delete tag with release bundle name from brep-imagecache-baseline repo
def delete_release_bundle(git_path, release_name):    
    logger.info("In deleteReleaseBundle")
    repo = set_git_credentails(git_path)
    logger.info('Deleting tag "{}"'.format(release_name))
    repo.delete_tag(release_name)
    origin = repo.remote(name="origin")
    origin.push(refspec=(":%s" % release_name))
    

# Push changes to GitOps repository
def git_push(git_path, release_name, commit_msg):
    # To set git config
    repo = set_git_credentails(git_path)

    if repo.is_dirty(untracked_files=True):
        logger.info("Untracked files are detected")
        repo.git.add("--all")
        repo.index.commit(
            "ImageCachingLambda: "
            + commit_msg
            + " release "
            + release_name
            + " on GitOps Repository"
        )
        origin = repo.remote(name="origin")
        origin.push()
        logger.info("Successfully commited to Git repository")
    else:
        logger.info("Nothing to commit, working tree clean")

def set_git_credentails(git_path):
    # Set Git User credentails
    repo = git.Repo(git_path)
    with repo.config_writer() as git_config:
        git_config.set_value("user", "email", "BREP-GHOpsSvcUser@us.mcd.com")
        git_config.set_value("user", "name", "BREP-GHOpsSvcUser")    
    return repo

def lambda_handler(event, context):
    # Read Event values
    print(event)
    try:
        data = event["Records"][0]["body"]
        # logger.info(type(data))
        if type(data) is str:
            data = json.loads(data)
        # logger.info('Event body "{}"'.format(data))
        action = data["httpMethod"]
        release_name = data["bundleName"]
        bundle_id = data["bundleId"]

        if action.upper() != "DELETE":
            applications = data["applications"]
        # logger.info(release_name)
        # Checkout Git Repository
        git_path = git_checkout()
        # logger.info('gitPath path: "{}"'.format(git_path))

        if release_name != "v1.1.0":
            if action.upper() == "POST":
                commit_msg = "Creating"
                message = create_bundle_tag(git_path, release_name, applications, commit_msg, bundle_id)
            elif action.upper() == "PUT":
                commit_msg = "Updating"
                delete_release_bundle(git_path, release_name)
                message = create_bundle_tag(git_path, release_name, applications, commit_msg, bundle_id)
            elif action.upper() == "DELETE":
                delete_release_bundle(git_path, release_name)
                message = "Tag deleted successfully"
            else:
                message = "Not a valid httpMethod"
        else:
            message = "Tag v1.1.0 creates kube-fledged CRDs and cannot be updated/deleted"  
    except Exception as e:
        status_code = 500
        message = str(e)
        return {
                 "statusCode": status_code,
                 "body": json.dumps(create_response_body(status_code, message))
            }

    return {
        "statusCode": 200,
        "body": json.dumps(create_response_body(200, message))
    }
