resource "aws_security_group" "lambda_sg" {
  name_prefix = "${var.tags["Name"]}-LAMBDA-SG"
  vpc_id      = var.vpc_id
  description = "Security group used for the Lambda (Managed by Terraform)"

  depends_on = [module.lambda.role_name]

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }

  tags = var.tags

  timeouts {
    delete = "40m"
  }
}

locals {
  environment_variables = merge(
  var.tags,
  var.environment_variables
  )
}

module "lambda" {
  source           = "git::https://github.com/bre-org/tfm-lambda-terraform-module.git"
  function_name    = "${var.tags["Name"]}-LAMBDA"
  filename         = "../function.zip"
  source_code_hash = "${filebase64sha256("../function.zip")}"
  handler          = "deploymentGenerator.lambda_handler"
  runtime          = "python3.10"
  tags             = var.tags
  timeout          = 60
  lambda_layer_arn = var.lambda_layer_arn
  environment      = "${local.environment_variables}"
  vpc_config = {
    security_group_ids = ["${aws_security_group.lambda_sg.id}"]
    subnet_ids         = "${data.aws_subnets.private.ids}"
  }
}

# This is needed, otherwise our lambda won't get triggered by SNS notification
resource "aws_lambda_permission" "with_sns" {
  statement_id  = "AllowExecutionFromSNS"
  action        = "lambda:InvokeFunction"
  function_name = module.lambda.arn
  principal     = "sns.amazonaws.com"
  source_arn    = module.sns.arn
}

resource "aws_lambda_permission" "with_sns_2" {
  statement_id  = "AllowExecutionFromSNS2"
  action        = "lambda:InvokeFunction"
  function_name = module.lambda.arn
  principal     = "sns.amazonaws.com"
  source_arn    = module.sns2.arn
}

resource "aws_iam_role_policy_attachment" "AWSLambdaVPCAccessExecutionRole" {
  role       = module.lambda.role_name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

resource "aws_iam_role_policy_attachment" "AWSLambdaBasicExecutionRole" {
  role       = module.lambda.role_name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
}

resource "aws_iam_role_policy_attachment" "AmazonSQSFullAccess" {
  role       = module.lambda.role_name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSQSFullAccess"
}

resource "aws_iam_role_policy" "secretsmanager" {
  name = "secretsmanager"
  role = module.lambda.role_name

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "SecretsManager",
            "Effect": "Allow",
            "Action": [
                "secretsmanager:GetRandomPassword",
                "secretsmanager:GetResourcePolicy",
                "secretsmanager:GetSecretValue",
                "secretsmanager:DescribeSecret",
                "secretsmanager:ListSecretVersionIds",
                "lambda:ListFunctions"
            ],
            "Resource": "*"
        }
    ]
}
EOF
}

resource "aws_iam_role_policy" "S3AuditBucketAccess" {
  count  = var.create_auditing_bucket == "true" ? 1 : 0
  name = "S3BucketAccess"
  role = module.lambda.role_name
  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "S3AuditBucketAccess",
            "Effect": "Allow",
            "Action": [
                "s3:PutObject",
                "s3:ListBucket",
		"s3:GetObject"
            ],
            "Resource": [
              "${aws_s3_bucket.bre_deploymentgenerator_audit_bucket[count.index].arn}",
              "${aws_s3_bucket.bre_deploymentgenerator_audit_bucket[count.index].arn}/*"
            ]
        }
    ]
}
EOF
}

resource "aws_iam_role_policy" "S3GetAllAppsBucketAccess" {
  count  = 1
  name = "S3AllAppsBucketAccess"
  role = module.lambda.role_name
  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "S3AuditBucketAccess",
            "Effect": "Allow",
            "Action": [
                "s3:PutObject",
                "s3:GetObject"
            ],
            "Resource": [
              "${aws_s3_bucket.bre_allaotappsdeployment[count.index].arn}",
              "${aws_s3_bucket.bre_allaotappsdeployment[count.index].arn}/*"
            ]
        }
    ]
}

EOF
}
resource "aws_iam_role_policy" "SQSKMSAccess" {
  name = "SQSKMSAccess"
  role = module.lambda.role_name
  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
{ 
  
  "Sid": "AllowKMSFullAccess",
  "Effect": "Allow",
  "Action": [
    "kms:List*",
    "kms:Get*",
    "kms:GenerateDataKey*",
    "kms:Encrypt",
    "kms:Decrypt",
    "kms:ReEncrypt*",
    "kms:DescribeKey",
    "s3:*"
	
  ],
  "Resource": "*"
}
    ]
}
EOF
}
