import os
from unittest.mock import patch
import json
from moto import mock_secretsmanager, mock_sqs, mock_s3
import boto3
import pytest
from . import deploymentGenerator
import yaml

event = {
    "Records": [
        {
            "EventSource": "aws:sns",
            "EventSubscriptionArn": "arn:aws:sns:us-east-1:283388140277:US-EAST-DEV-BRE-ARCH-BRE-DEPLOYMENTGENERATOR-MASTER-SNS:3a59bc0f-a05f-472b-90c6-b2960fd1c7d8",
            "EventVersion": "1.0",
            "Sns": {
                "Message": "Application was added to the asset service "
                "using the following Jenkins build "
                "https://jenkins.bre.mcd.com/job/Admin/job/Restaurant%20Onboarding%20Pipeline/1885/",
                "MessageAttributes": {
                    "buildNumber": {"Type": "String", "Value": "88"},
                    "componentName": {"Type": "String", "Value": "bref-main"},
                    "imageURL": {
                        "Type": "String",
                        "Value": "artifactory.bre.mcd.com/docker/bref-main",
                    },
                    "podName": {"Type": "String", "Value": "bref-main"},
                    "replaceVersion": {"Type": "String", "Value": "88"},
                    "restaurantId": {"Type": "String", "Value": "26"},
                    "restaurantName": {"Type": "String", "Value": "tooling-dev"},
                },
                "MessageId": "80cb9fda-7288-5fc5-b2e4-c7f97a5a2f51",
                "Signature": "RgOJSBdh/SWXTwKnNk/JO4KqXGh+fyFVMJpMDSUMP/3s2n1O3xSGHa8KrKY5ObiZBwQfVU4TM+JBQsBOuH3NBQJdA7IXg+6pEMOmpsVx6thZkBQZ9lZfzbOfkgh5NvcWc/h4li1MjfLsTpq7iPsuKAtBDIn3zq4gyknvY39IARmfRBvjp188WPI3VqA0dPmPWJjqgRsFVbbG+p3cyevSadn+3ysX8qQDCOmVpxdCk45otunsy6XPH+n02RNCjLQZg7vFEHxCz820ExKcVif+hIj1Hm7aIq74D8SYGniIUWBrB2GB8XNKA7KhEImiVYD1s5us9UB4UfLmq5m8k5DN3A==",
                "SignatureVersion": "1",
                "SigningCertUrl": "https://sns.us-east-1.amazonaws.com/SimpleNotificationService-a86cb10b4e1f29c941702d737128f7b6.pem",
                "Subject": "bref-main: Successfully added to asset " "service",
                "Timestamp": "2020-11-09T09:06:26.883Z",
                "TopicArn": "arn:aws:sns:us-east-1:283388140277:US-EAST-DEV-BRE-ARCH-BRE-DEPLOYMENTGENERATOR-MASTER-SNS",
                "Type": "Notification",
                "UnsubscribeUrl": "https://sns.us-east-1.amazonaws.com/?Action=Unsubscribe&SubscriptionArn=arn:aws:sns:us-east-1:283388140277:US-EAST-DEV-BRE-ARCH-BRE-DEPLOYMENTGENERATOR-MASTER-SNS:3a59bc0f-a05f-472b-90c6-b2960fd1c7d8",
            },
        }
    ]
}

docker_message_attributes = {
    k: v.get("Value")
    for k, v in event["Records"][0]["Sns"]["MessageAttributes"].items()
}

project_name, app_name = docker_message_attributes["podName"].lower().split("-")[:2]
# /tmp/bre-lambda-yml-templates/bref/main
git_template_path = os.path.join(
    "/tmp/bre-lambda-yml-templates", project_name, app_name
)
restaurant_id = "14"


@pytest.fixture(scope="module")
def create_yml():
    os.makedirs(
        os.path.join("/tmp/bre-lambda-yml-templates", project_name), exist_ok=True
    )
    with open("template.yml", "r") as f:
        contents = f.read()
    yml_path = git_template_path + ".yml"
    if not os.path.exists(yml_path):
        with open(yml_path, "w") as f:
            f.write(contents)


@pytest.fixture(scope="module")
def aws_credentials():
    """Mock AWS credentials"""
    os.environ["AWS_SECRET_ACCESS_KEY"] = "mock"
    os.environ["AWS_ACCESS_KEY_ID"] = "mock"
    os.environ["AWS_SECURITY_TOKEN"] = "mock"
    os.environ["AWS_SESSION_TOKEN"] = "mock"


@pytest.yield_fixture(scope="module")
def secret_manager_client(aws_credentials):
    """Create a mock secrets client"""
    with mock_secretsmanager():
        session = boto3.session.Session()
        secret_client = session.client(
            service_name="secretsmanager", region_name="us-east-1"
        )
        yield secret_client


@pytest.yield_fixture(scope="module")
def sqs_client(aws_credentials):
    with mock_sqs():
        queue_client = boto3.client("sqs")
        yield queue_client


@pytest.yield_fixture(scope="module")
def s3_client(aws_credentials):
    with mock_s3():
        client = boto3.client("s3", region_name="us-east-1")
        yield client


@patch("git.Repo")
@patch("git.Git")
def test_gitCheckout(mock_git, mock_repo, secret_manager_client):
    # mock git cloning
    p = mock_git.return_value = False
    type(mock_repo.clone_from.return_value).bare = p

    # create dummy secrets
    secret_manager_client.put_secret_value(
        SecretId="github/bre-cloud-git-reconcile",
        SecretString=json.dumps({"username": "BRE_LAMBDA", "password": "1234"}),
    )
    git_path = deploymentGenerator.gitCheckout(
        docker_message_attributes, project_name, app_name
    )
    assert git_template_path == git_path


def create_mock_k8s_response(restaurant_name):
    content = (
        open("template.yml", "r+")
        .read()
        .format(
            PodName=docker_message_attributes["podName"],
            NewVersion=docker_message_attributes["buildNumber"],
            ReplaceVersion=docker_message_attributes["replaceVersion"],
            ImageURL=docker_message_attributes["imageURL"],
            ComponentName=docker_message_attributes["componentName"],
            PortName=docker_message_attributes["podName"][:15],
            AwsAccount="524430043955",
            RestaurantName=restaurant_name,
            Environment="prod",
        )
    )

    multi_doc = list(yaml.safe_load_all(content))
    mock_metadata = json.dumps(
        {
            "metadata": {
                "id": "88",
                "restaurantIds": "14",
                "pod": "bref-main",
                "namespace": project_name,
                "deploymentGroupId": "0",
            }
        },
        sort_keys=False,
        indent=None,
    )
    k8s_mock = {"metadata": mock_metadata}
    for doc in multi_doc:
        kind = doc["kind"].lower()
        k8s_mock[kind] = json.dumps(doc)
    return k8s_mock


def test_k8sGenerator(requests_mock, create_yml):
    # TAKES VALUES FROM ATTRIBUTES
    # SENDS REQUEST TO URL/restaurantid
    # check if yml file exists
    # does random things on the yml file
    # creates a k8s object
    # validates k8s and returns it

    mock_response = {
        "data": {
            "id": "14",
            "type": "restaurants",
            "attributes": {
                "created": "2020-02-07T14:18:34.000+0000",
                "name": "TestRestaurant",
                "description": "Test description",
                "updated": "2020-10-20T13:32:22.000+0000",
            },
            "relationships": {
                "market": {
                    "links": {
                        "self": "https://asset.ui-api.prod.bre.mcd.com/restaurant_assets/restaurants/14/relationships/market",
                        "related": "https://asset.ui-api.prod.bre.mcd.com/restaurant_assets/restaurants/14/market",
                    }
                },
                "components": {
                    "links": {
                        "self": "https://asset.ui-api.prod.bre.mcd.com/restaurant_assets/restaurants/14/relationships/components",
                        "related": "https://asset.ui-api.prod.bre.mcd.com/restaurant_assets/restaurants/14/components",
                    }
                },
                "ownerOperator": {
                    "links": {
                        "self": "https://asset.ui-api.prod.bre.mcd.com/restaurant_assets/restaurants/14/relationships/ownerOperator",
                        "related": "https://asset.ui-api.prod.bre.mcd.com/restaurant_assets/restaurants/14/ownerOperator",
                    }
                },
            },
            "links": {
                "self": "https://asset.ui-api.prod.bre.mcd.com/restaurant_assets/restaurants/14"
            },
        }
    }

    restaurant_id = "14"
    requests_mock.get(
        deploymentGenerator.assetsRestaurantUrl + f"/{restaurant_id}",
        json=mock_response,
    )

    k8s_mock = create_mock_k8s_response("TestRestaurant".lower())
    k8s = deploymentGenerator.k8sGenerator(
        docker_message_attributes,
        project_name,
        app_name,
        git_template_path,
        restaurant_id,
    )
    metadata = json.loads(k8s["metadata"])
    del metadata["metadata"]["date"]
    k8s["metadata"] = json.dumps(
        metadata,
        sort_keys=False,
        indent=None,
    )
    assert k8s == k8s_mock


def test_pushDeploymentMessage(sqs_client, s3_client):
    # SENDS MESSAGEE TO SQS
    # PUTS OBJECT IN S3
    # TODO: ADD THIS

    # create mock queues
    sqs_client.create_queue(
        QueueName=deploymentGenerator.queuePrefix
        + restaurant_id
        + deploymentGenerator.queueSuffix
    )
    # create mock bucket
    s3_client.create_bucket(Bucket=deploymentGenerator.auditS3Bucket)

    # ADD PARAMS
    k8s_mock = create_mock_k8s_response("TestRestaurant".lower())
    return_value = deploymentGenerator.pushDeploymentMessage(
        app_name,
        restaurant_id,
        k8s_mock,
        docker_message_attributes["buildNumber"],
    )
    assert return_value is None


def test_findRestaurantIdsFirmware(requests_mock):
    mock_response = {"data": [{"id": "14"}]}
    requests_mock.get(
        deploymentGenerator.assetsRestaurantUrl,
        json=mock_response,
    )
    restaurant_ids = deploymentGenerator.findRestaurantIdsFirmware(
        docker_message_attributes["replaceVersion"],
        docker_message_attributes["componentName"],
    )

    mock_restaurant_ids = ["14"]
    assert mock_restaurant_ids == restaurant_ids


def test_createMessageBodyFirmware(requests_mock):
    firmwareStringValue = json.dumps(
        {
            "deviceName": docker_message_attributes["componentName"],
            "deviceId": ["20"],
            "replaceVersion": docker_message_attributes["replaceVersion"],
            "deployVersion": docker_message_attributes["buildNumber"],
            "restaurantId": "14",
            "imageURL": docker_message_attributes["imageURL"],
        },
        sort_keys=False,
        indent=None,
    )
    metadataStringValue = json.dumps(
        {
            "metadata": {
                "pod": "iot-device-configurator",
                "namespace": "iot",
                "deploymentGroupId": "0",
            }
        },
        sort_keys=False,
        indent=None,
    )
    mock_message_body = {
        "firmware": {"StringValue": firmwareStringValue, "DataType": "String"},
        "metadata": {"StringValue": metadataStringValue, "DataType": "String"},
    }
    mock_response = {"data": [{"id": "20"}]}
    requests_mock.get(deploymentGenerator.assetsComponentUrl, json=mock_response)
    message_body = deploymentGenerator.createMessageBodyFirmware(
        docker_message_attributes["replaceVersion"],
        docker_message_attributes["componentName"],
        "14",
        docker_message_attributes["imageURL"],
        docker_message_attributes["buildNumber"],
        "0",
    )
    assert mock_message_body == message_body


def test_pushDeploymentMessageFirmware(sqs_client, s3_client, requests_mock):
    # mock request
    mock_response = {"data": [{"id": "20"}]}
    requests_mock.get(deploymentGenerator.assetsComponentUrl, json=mock_response)
    # create mock queues
    sqs_client.create_queue(
        QueueName=deploymentGenerator.queuePrefix
        + restaurant_id
        + deploymentGenerator.queueSuffix
    )
    # create mock bucket
    s3_client.create_bucket(Bucket=deploymentGenerator.auditS3Bucket)

    return_value = deploymentGenerator.pushDeploymentMessageFirmware(
        [restaurant_id],
        docker_message_attributes["replaceVersion"],
        docker_message_attributes["componentName"],
        docker_message_attributes["imageURL"],
        docker_message_attributes["buildNumber"],
        "0",
    )
    assert return_value is None


def test_validateInputFirmware():
    result = deploymentGenerator.validateInputFirmware(docker_message_attributes)
    assert result is True