import json
import requests
import boto3
import yaml
import os
import shutil
import git
import sys
import pprint
import kubernetes_validate
from logging2 import Logger, LogLevel, StdOutHandler, StdErrHandler
from datetime import datetime
import base64

stdout = StdOutHandler(level=LogLevel.error)
stderr = StdErrHandler(level=LogLevel.error)
logger = Logger("deploymentGenerator", handlers=[stdout, stderr])

assetsUrl = os.environ.get(
    "assetsUrl", "https://asset-service.bre.mcd.com/restaurant_assets"
)
# Bucket with App Info
awsS3Bucket = os.environ.get(
    "awsS3Bucket", "us-east-prod-bre-all-aot-apps-json"
)
# File With App Info
imageDataFile = os.environ.get(
    "imageDataFile", "allAppsInfo.json"
)

assetsRestaurantUrl = assetsUrl + "/restaurants"
assetsComponentUrl = assetsUrl + "/components"
componentTypeGreengrassDevice = "GREENGRASS_DEVICE"

# Setup boto3 resources
sqs = boto3.resource("sqs")
s3 = boto3.resource("s3")

auditS3Bucket = os.environ.get(
    "auditBucketName", "us-east-prod-bre-deploymentgenerator-auditing"
)

currentDate = datetime.now().strftime("%Y-%m-%d-%H:%M:%S")

# Declare an empty queue
queueName = None
queue = None
queuePrefix = os.environ.get("queuePrefix", "US-EAST-PROD-BRE-ARCH-BRE-")
queueSuffix = os.environ.get("queueSuffix", "-SQS-DEPLOYMENT.fifo")


def lambda_handler(event, context):
    # pprint.pprint(event)

    snsProps = event["Records"][0]["Sns"]
    messageAttributes = snsProps["MessageAttributes"]

    messageAttributes = {k: v["Value"] for k, v in messageAttributes.items()}
    logger.debug("MessageAttributes: {}".format(json.dumps(messageAttributes)))

    if "deploymentType" not in messageAttributes:
        deploymentType = "default"
    elif (
        "deploymentType" in messageAttributes
        and messageAttributes["deploymentType"] == "docker"
    ):
        deploymentType = "docker"
    elif (
        "deploymentType" in messageAttributes
        and messageAttributes["deploymentType"] == "firmware"
    ):
        deploymentType = "firmware"
    else:
        deploymentType = "unknown"
    if deploymentType == "default" or deploymentType == "docker":
        logger.debug("DeploymentType is {}".format(deploymentType))
        templateName = messageAttributes["podName"].lower()
        # Split template name and get first value
        projectName = templateName.split("-")[0]
        # Split template name and get second value
        appName = templateName.split("-", 1)[1]

        restaurantIds = []
        # restaurantNames = []
        try:
            if not messageAttributes["restaurantId"]:
                # Check for empty string. This logic can be removed once Tech Labs has shifted to using the UI or the SNS Lambda
                raise KeyError
            elif eval(messageAttributes["restaurantId"]):
                # Check if list is not empty
                if deploymentType == "docker":
                    restaurantIds = json.loads(
                        messageAttributes["restaurantId"])
                elif deploymentType == "default":
                    restaurantIds = messageAttributes["restaurantId"].split(
                        ",")
                print(restaurantIds)
            else:
                raise KeyError
        except KeyError:
            assetsUrlParams = {
                "filter[components.name]": templateName,
                "filter[components.reportedVersion]": messageAttributes[
                    "replaceVersion"
                ],
                "fields": "id,name",
            }

            assetsUrlResponse = requests.get(
                assetsRestaurantUrl, params=assetsUrlParams
            )

            assetsUrlResponseJson = json.loads(assetsUrlResponse.text)
            restaurantIds = [element["id"]
                             for element in assetsUrlResponseJson["data"]]

            logger.info('Restaurant IDs: "{}"'.format(restaurantIds))
            logger.debug("Assets Service Url: {}".format(
                assetsUrlResponse.url))
            logger.debug(
                "Response from assetsRestaurantUrl: {}".format(
                    assetsUrlResponse.text)
            )
            logger.debug("MessageAttributes: {}".format(
                json.dumps(messageAttributes)))

            if not restaurantIds:
                logger.error(
                    "Assets service did not return a list of restaurants for {} for version {}! Asset Service Url: {}".format(
                        templateName,
                        messageAttributes["replaceVersion"],
                        assetsUrlResponse.url,
                    )
                )
                sys.exit(1)

        buildNumber = messageAttributes["buildNumber"]
        logger.info("BuildNumber: {}".format(buildNumber))

        gitTemplatePath = gitCheckout(messageAttributes, projectName, appName)

        # Loop through all restaurantIds retrieved from asset service, generate deployments and then push to SQS & S3
        for restaurantId in restaurantIds:
            generateDeployment = k8sGenerator(
                messageAttributes, projectName, appName, gitTemplatePath, restaurantId
            )
            pushDeploymentMessage(
                appName, restaurantId, generateDeployment, buildNumber
            )

    elif deploymentType == "firmware":
        logger.debug("DeploymentType is {}".format(deploymentType))
        if validateInputFirmware(messageAttributes) == False:
            sys.exit(1)
        else:
            logger.info("Message Attributes: {}".format(messageAttributes))
            if "componentName" in messageAttributes:
                componentName = messageAttributes["componentName"]
            else:
                componentName = ''
            buildNumber = messageAttributes["buildNumber"]
            replaceVersion = messageAttributes["replaceVersion"]
            imageURL = messageAttributes["imageURL"]
            vendor = messageAttributes["vendor"]
            modelNumber = messageAttributes["modelNumber"]

            if "deploymentGroupId" in messageAttributes:
                deploymentGroupId = messageAttributes["deploymentGroupId"]
            else:
                deploymentGroupId = "0"

        restaurantIds = []

        if eval(messageAttributes["restaurantId"]):
            restaurantIds = json.loads(messageAttributes["restaurantId"])
            logger.info("Restaurant IDs: {}".format(restaurantIds))
        else:
            restaurantIds = findRestaurantIdsFirmware(
                replaceVersion, componentName)

        pushDeploymentMessageFirmware(
            vendor,
            modelNumber,
            restaurantIds,
            replaceVersion,
            componentName,
            imageURL,
            buildNumber,
            deploymentGroupId,
        )

    else:
        logger.debug("DeploymentType is {}".format(deploymentType))


def pushDeploymentMessageFirmware(
    vendor,
    modelNumber,
    restaurantIds,
    replaceVersion,
    componentName,
    imageURL,
    buildNumber,
    deploymentGroupId,
):
    for restaurantId in restaurantIds:
        messageBody = createMessageBodyFirmware(
            vendor,
            modelNumber,
            replaceVersion,
            componentName,
            restaurantId,
            imageURL,
            buildNumber,
            deploymentGroupId,
        )
        messageAttributes = {}
        try:
            restaurantIdStr = str(restaurantId)
            queueName = queuePrefix + restaurantIdStr + queueSuffix

            logger.info("Queue name: {}".format(queueName))

            queue = sqs.get_queue_by_name(QueueName=queueName)
            # If queue exists, send message
            _response = queue.send_message(
                MessageGroupId=restaurantIdStr,
                MessageBody=json.dumps(messageBody),
                MessageAttributes=messageAttributes,
            )
            logger.info(
                'Pushed SQS message for restaurant id "{}", firmware names starting with "{}", replace version "{}"'.format(
                    restaurantIdStr, componentName, replaceVersion
                )
            )
        except:
            logger.error(
                'SQS queue with Restaurant ID of "{}" does not exist'.format(
                    restaurantIdStr
                )
            )

        try:
            s3.meta.client.head_bucket(Bucket=auditS3Bucket)
            logger.info(
                'S3 Bucket with name "{}" exists'.format(auditS3Bucket))

            # If bucket exists, send deployment message
            object = s3.Object(
                auditS3Bucket,
                restaurantIdStr + "/" + currentDate + "-deployment-message.json",
            )
            object.put(Body=json.dumps(messageBody, indent=2))

            logger.info(
                'Pushed deployment message to S3 bucket with name of "{}" for restaurant id "{}", firmware names starting with "{}"'.format(
                    auditS3Bucket, restaurantIdStr, componentName
                )
            )
        except:

            logger.error(
                'Bucket with bucket name of "{}" does not exist'.format(
                    auditS3Bucket)
            )


def createMessageBodyFirmware(
    vendor,
    modelNumber,
    replaceVersion,
    componentName,
    restaurantId,
    imageURL,
    buildNumber,
    deploymentGroupId,
):
    # assetsUrlParams = {
    #     "filter[type.name]": componentTypeGreengrassDevice,
    #     "filter[reportedVersion]": replaceVersion,
    #     "filter[name][like]": componentName + "%",
    #     "filter[restaurant.id]": restaurantId,
    #     "fields": "id",
    # }
    assetsUrlParams = {
		"filter[vendor.name]": vendor,
		"filter[reportedVersion]": replaceVersion,
		"filter[modelNumber]": modelNumber,
		"filter[restaurant.id]": restaurantId,
		"fields": "id",
	}

    assetsUrlResponse = requests.get(
        assetsComponentUrl, params=assetsUrlParams)

    assetsUrlResponseJson = json.loads(assetsUrlResponse.text)

    componentIds = [element["id"] for element in assetsUrlResponseJson["data"]]
    logger.info('Component IDs (Firmware): "{}"'.format(componentIds))
    logger.debug(
        "Assets Service Components endpoint Url: {}".format(
            assetsUrlResponse.url)
    )
    logger.debug("Response: {}".format(assetsUrlResponse.text))

    if not componentIds:
        logger.error(
            "Assets service did not return a list of components for Component Type {}, Reported Version {}, Component Name starting with {}, Restaurant ID {}! Asset Service Url: {}".format(
                componentTypeGreengrassDevice,
                replaceVersion,
                componentName,
                restaurantId,
                assetsUrlResponse.url,
            )
        )
        sys.exit(1)

    firmwareStringValue = json.dumps(
        {
            "deviceName": componentName,
            "deviceId": componentIds,
            "replaceVersion": replaceVersion,
            "deployVersion": buildNumber,
            "restaurantId": restaurantId,
            "imageURL": imageURL,
            "deploymentGroupId": deploymentGroupId,
        },
        sort_keys=False,
        indent=None,
    )

    metadataStringValue = json.dumps(
        {
            "metadata": {
                "pod": "iot-thing-configurator",
                "namespace": "iot",
            }
        },
        sort_keys=False,
        indent=None,
    )

    messageBody = {
        "firmware": {"StringValue": firmwareStringValue, "DataType": "String"},
        "metadata": {"StringValue": metadataStringValue, "DataType": "String"},
    }
    return messageBody


def findRestaurantIdsFirmware(replaceVersion, componentName):
    assetsUrlParams = {
        "filter[components.type.name]": componentTypeGreengrassDevice,
        "filter[components.reportedVersion]": replaceVersion,
        "filter[components.name][like]": componentName + "%",
        "fields": "id",
    }

    assetsUrlResponse = requests.get(
        assetsRestaurantUrl, params=assetsUrlParams)

    assetsUrlResponseJson = json.loads(assetsUrlResponse.text)

    restaurantIds = [element["id"]
                     for element in assetsUrlResponseJson["data"]]

    logger.info('Restaurant IDs (Firmware): "{}"'.format(restaurantIds))
    logger.debug("Assets Service Url: {}".format(assetsUrlResponse.url))
    logger.debug("Response from assetsRestaurantUrl: {}".format(
        assetsUrlResponse.text))

    if not restaurantIds:
        logger.error(
            "Assets service did not return a list of restaurants for Component Type {}, Reported Version {}, Component Name starting with {}! Asset Service Url: {}".format(
                componentTypeGreengrassDevice,
                replaceVersion,
                componentName,
                assetsUrlResponse.url,
            )
        )
        sys.exit(1)

    return restaurantIds


def validateInputFirmware(messageAttributes):
    result = True
    # if (
    #     "componentName" not in messageAttributes
    #     or not messageAttributes["componentName"].strip()
    # ):
    #     logger.error(
    #         "componentName is not provided in the input message attributes")
    #     result = False

    if (
        "buildNumber" not in messageAttributes
        or not messageAttributes["buildNumber"].strip()
    ):
        logger.error(
            "buildNumber is not provided in the input message attributes")
        result = False

    if (
        "replaceVersion" not in messageAttributes
        or not messageAttributes["replaceVersion"].strip()
    ):
        logger.error(
            "replaceVersion is not provided in the input message attributes")
        result = False

    if "imageURL" not in messageAttributes or not messageAttributes["imageURL"].strip():
        logger.error(
            "deploymentUrl is not provided in the input message attributes")
        result = False
    return result


def gitCheckout(messageAttributes, projectName, appName):
    # Get Environment Variables
    # Git URL without protocol
    gitUrl = os.environ.get(
        "gitUrl", "github.com/bre-org/bre-lambda-yml-templates.git"
    )
    # Protocol, if not specified use http
    gitUrlProtocol = os.environ.get("gitUrlProtocol", "https")
    # Git branch to use
    gitBranch = os.environ.get("gitBranch", "master")
    if "network" in messageAttributes and (messageAttributes["network"] != 'x.y' or messageAttributes["network"] == "" ):
        gitBranch = os.environ.get("gitBranch", "Addition-of-configuration-map-to-yaml-files")
    # Secret Manager key Name
    smName = os.environ.get("smName", "github/bre-cloud-git-reconcile")
    # AWS Region Name
    regionName = os.environ.get("AWS_REGION", "us-east-1")

    logger.debug('gitUrl: "{}"'.format(gitUrl))
    logger.debug('gitUrlProtocol: "{}"'.format(gitUrlProtocol))
    logger.debug('gitBranch: "{}"'.format(gitBranch))
    logger.debug('smName: "{}"'.format(smName))
    logger.debug('regionName: "{}"'.format(regionName))
    logger.debug('projectName: "{}"'.format(projectName))
    logger.debug('appName: "{}"'.format(appName))
    logger.debug('gitBranch: "{}"'.format(gitBranch))

    # Create a Secrets Manager client
    session = boto3.session.Session()
    client = session.client(
        service_name="secretsmanager", region_name=regionName)
    get_secret_value_response = client.get_secret_value(SecretId=smName)
    secret = get_secret_value_response["SecretString"]
    jsecret = json.loads(secret)
    gitusername = jsecret["username"]
    gitpassword = jsecret["password"]

    checkoutDirectory = "/tmp/bre-lambda-yml-templates"
    templateFileName = projectName + "/" + appName
    gitTemplatePath = checkoutDirectory + "/" + templateFileName

    if os.path.exists(checkoutDirectory):
        logger.info("Removing existing {} git directory!".format(
            checkoutDirectory))
        shutil.rmtree(checkoutDirectory)
        logger.info('Checking out git repository "{}"'.format(gitUrl))
        git.Repo.clone_from(
            gitUrlProtocol + "://" + gitusername + ":" + gitpassword + "@" + gitUrl,
            checkoutDirectory,
            branch=gitBranch,
            depth=1
        )
    else:
        logger.info('Checking out git repository "{}"'.format(gitUrl))
        git.Repo.clone_from(
            gitUrlProtocol + "://" + gitusername + ":" + gitpassword + "@" + gitUrl,
            checkoutDirectory,
            branch=gitBranch,
            depth=1
        )

    return gitTemplatePath
    # print(gitTemplatePath)


def k8sGenerator(
    messageAttributes, projectName, appName, gitTemplatePath, restaurantId
):
    # Check if our SNS message has attribute `newVersion` if not use build number
    # NewVersion is only in `RELEASE` SNS message attributes
    newVersion = (
        messageAttributes["newVersion"]
        if "newVersion" in messageAttributes
        else messageAttributes["buildNumber"]
    )

    podName = messageAttributes["podName"]
    imageURL = messageAttributes["imageURL"]
    componentName = messageAttributes["componentName"]
    replaceVersion = messageAttributes["replaceVersion"]
    # networkID = messageAttributes["network"]
    deploymentPortName = podName[:15]
    fullPathToTemplate = gitTemplatePath + ".yml"
    buildNumber = messageAttributes["buildNumber"]
    
    if "deploymentGroupId" in messageAttributes:
        deploymentGroupId = messageAttributes["deploymentGroupId"]
    else:
        deploymentGroupId = "0"
    
    if "network" in messageAttributes:
        networkID = messageAttributes["network"]
    else:
        networkID = "NA"
    currentDate = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    awsAccount = None
    env = None
    restaurantIdStr = str(restaurantId)
    # To get Lambda Public IP for troubleshooting
    publicIP = requests.get('http://checkip.amazonaws.com').text.rstrip()
    logger.info(f"Public IP: {publicIP}")
    # Retrieve restaurant for the given restaurantId. Name is only used currently in the case of cloud deployments to retrieve the cluster name
    logger.info(
        f"Requesting asset service with restaurantId: {restaurantIdStr}")
    logger.info(
        f"Requesting asset service with URL: {assetsRestaurantUrl}")
    restaurantNameResponse = requests.get(
        assetsRestaurantUrl + "/" + restaurantIdStr)
    logger.info(
        f"Response Is : {restaurantNameResponse}")    
    restaurantNameResponseJson = json.loads(restaurantNameResponse.text)
    restaurantName = restaurantNameResponseJson["data"]["attributes"]["name"].lower(
    )
    # restaurantName = "03381"
    # Read the S3 Bucket
    s3Obj = s3.Object(awsS3Bucket, imageDataFile)
    getDeploymentsUrlResponse = s3Obj.get()['Body'].read()
    # getDeploymentsUrlResponse = open('allAppsInfo.json').read()
    getDeploymentsUrlResponseJson = json.loads(getDeploymentsUrlResponse)
    finalArgs = {}
    # Search Loop through the object and Find the Correct data using ImageName
    for deploymentData in getDeploymentsUrlResponseJson["data"]:

        templateName = messageAttributes["podName"].lower()
        if deploymentData['imageName'] == templateName and 'secret' in deploymentData :
            appSecret = json.loads(json.dumps(deploymentData['secret']))
            # read each secret decode it and replace {StoreID} and encrypt it (base64)
            for key, value in appSecret.items():
                # This is doing all the Magic
                updatedValue = base64.b64encode(base64.b64decode(value.encode('ascii')).decode('ascii').replace("{StoreID}", restaurantIdStr).encode('ascii')).decode('ascii')
                logger.info('Updated Key - value : "{}"-"{}"'.format(key,updatedValue))
                finalArgs[key] = updatedValue
   
    if os.path.exists(fullPathToTemplate):
        # Default to the Prod environment, which is using AwsAccount and Environment at the edge.
        env = "prod"
        awsAccount = "524430043955"

        if restaurantName == "tooling-dev":
            env = "dev"
            awsAccount = "283388140277"
        elif restaurantName == "tooling-int":
            env = "int"
            awsAccount = "593265675765"
        elif restaurantName == "tooling-stg":
            env = "stg"
            awsAccount = "688810906228"
            

        
            # add the secret to the dict and existing values as well
        finalArgs['PodName'] = podName
        finalArgs['NewVersion'] = newVersion
        finalArgs['ReplaceVersion'] = replaceVersion
        finalArgs['ImageURL'] = imageURL
        finalArgs['ComponentName'] = componentName
        finalArgs['PortName'] = deploymentPortName
        finalArgs['AwsAccount'] = awsAccount
        finalArgs['RestaurantName'] = restaurantName
        finalArgs['Environment'] = env
        finalArgs['networkinfo'] = networkID

        logger.info('Updated Final Args : "{}"'.format(json.dumps(finalArgs)))
        gitTemplate = (
            open(fullPathToTemplate, "r+")
            .read()
            .format(
                **finalArgs
            )
        )
        
       # print(gitTemplate)
        
        multiDoc = list(yaml.safe_load_all(gitTemplate))

        metadata = json.dumps(
            {
                "metadata": {
                    "id": buildNumber,
                    "date": currentDate,
                    "restaurantIds": restaurantId,
                    "pod": podName,
                    "namespace": projectName,
                    "deploymentGroupId": deploymentGroupId,
                }
            },
            sort_keys=False,
            indent=None,
        )

        k8s = {"metadata": metadata}

        hasDuplicate = hasMultipleKinds(multiDoc)

        for doc in multiDoc:
            kindName = doc["kind"].lower()

            try:
                if "metadata" in doc:
                    if "name" in doc["metadata"]:
                        objectName = doc["metadata"]["name"]
            except KeyError:
                logger.info(
                    "Does not have metadata name {} {}".format(
                        e.path, e.message)
                )

            if hasDuplicate and len(objectName) > 0:
                docKeyName = f"{kindName}|{objectName}"
            else:
                docKeyName = kindName

            k8s[docKeyName] = json.dumps(doc)
            try:
                kubernetes_validate.validate(doc, "1.16.1", strict=True)
            except kubernetes_validate.SchemaNotFoundError as e:
                # SchemaNotFoundError due to CRD
                # Allowing deployment since YAML syntax is already validated by json.dumps
                logger.error("SchemaNotFoundError {}".format(e.message))
            except kubernetes_validate.ValidationError as e:
                logger.error("{} {}".format(e.path, e.message))
                raise

        logger.info('Generated k8s messages for "{}"'.format(appName))

        return k8s

    else:
        logger.error(
            'Kubernetes messages could not be generated. Template file not found: "{}"! Exiting!'.format(
                fullPathToTemplate
            )
        )
        sys.exit(1)


def pushDeploymentMessage(appName, restaurantId, deploymentMessage, buildNumber):
    messageAttributes = {}
    MessageBody = {}

    for k, v in deploymentMessage.items():
        MessageBody[k] = {"DataType": "String", "StringValue": v}

    try:
        restaurantIdStr = str(restaurantId)
        queueName = queuePrefix + restaurantIdStr + queueSuffix

        logger.info("Queue name: {}".format(queueName))

        queue = sqs.get_queue_by_name(QueueName=queueName)

        # If queue exists, send message
        _response = queue.send_message(
            MessageGroupId=restaurantIdStr,
            MessageBody=json.dumps(MessageBody),
            MessageAttributes=messageAttributes,
        )
        logger.info(
            'Pushed SQS message to restaurant "{}" for application "{}" with buildNumber "{}"'.format(
                restaurantIdStr, appName, buildNumber
            )
        )
    except:

        logger.error(
            'SQS queue with Restaurant ID of "{}" does not exist'.format(
                restaurantIdStr
            )
        )

    try:
        s3.meta.client.head_bucket(Bucket=auditS3Bucket)
        logger.info('S3 Bucket with name "{}" exists'.format(auditS3Bucket))

        # If deployment bucket exists, send deployment message
        object = s3.Object(
            auditS3Bucket,
            restaurantIdStr + "/" + currentDate + "-deployment-message.json",
        )
        object.put(Body=json.dumps(MessageBody, indent=2))

        logger.info(
            'Pushed deployment message to S3 bucket with name of "{}" for restaurant "{}" for application "{}"'.format(
                auditS3Bucket, restaurantIdStr, appName
            )
        )
    except:

        logger.error(
            'Bucket with bucket name of "{}" does not exist'.format(
                auditS3Bucket)
        )


def hasMultipleKinds(items):
    history = []
    for item in items:
        logger.debug(f"checking dups for item: {item['kind'].lower()}")
        kindName = item["kind"].lower()
        if kindName in history:
            return True
        history.append(kindName)

    return False
